/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.InputEvent.ALT_DOWN_MASK;
import static java.awt.event.KeyEvent.*;
import static javax.swing.KeyStroke.getKeyStroke;
import static net.argius.stew.ui.window.Resource.*;

import java.awt.event.*;

import javax.swing.*;
import javax.swing.text.*;
import javax.swing.undo.*;

/**
 * [^CṽeLXgGAB
 */
final class ConsoleTextArea extends JTextArea {

    private final UndoManager undoManager;

    private int homePosition;

    ConsoleTextArea() {
        // [CX^X]
        this.undoManager = ActionUtility.setUndoAction(this);
        ((AbstractDocument)getDocument()).setDocumentFilter(new PrivateDocumentFilter(this));
        // [ANV]
        ActionUtility actionUtility = new ActionUtility(this);
        final int shortcutKey = Resource.getMenuShortcutKeyMask();
        InputMap imap = getInputMap();
        // commit
        final String keyCommit = "commit";
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                int cp = getCaretPosition();
                int ep = getEndPosition();
                if (cp != ep) {
                    setCaretPosition(ep);
                    return;
                }
                fireActionPerformed();
            }

        }, keyCommit);
        imap.put(getKeyStroke(VK_ENTER, 0), keyCommit);
        imap.put(getKeyStroke(VK_M, shortcutKey), keyCommit);
        // copy or break
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                if (getSelectedText() == null) {
                    sendBreak();
                } else {
                    Action copyAction = new DefaultEditorKit.CopyAction();
                    copyAction.actionPerformed(e);
                }
            }

        }, "copyOrBreak", getKeyStroke(VK_C, shortcutKey));
        // break
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                sendBreak();
            }

        }, "break", getKeyStroke(VK_B, ALT_DOWN_MASK));
        // new line
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                insert(EOL, getCaretPosition());
            }

        }, "newLine", getKeyStroke(VK_ENTER, shortcutKey));
        // home
        actionUtility.bindAction(new AbstractAction() {

            public void actionPerformed(ActionEvent e) {
                setCaretPosition(getHomePosition());
            }

        }, "home", getKeyStroke(VK_HOME, 0));
    }

    /**
     * ActionListener̓o^B
     * @param listener ActionListener
     */
    void addActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.add(ActionListener.class, listener);
    }

    /**
     * ActionListener̍폜B
     * @param listener ActionListener
     */
    void removeActionListener(ActionListener listener) {
        assert listener != null;
        listenerList.remove(ActionListener.class, listener);
    }

    /**
     * Action̒ʒmB
     */
    void fireActionPerformed() {
        ActionEvent event = new ActionEvent(this, 0, getEditableText());
        for (ActionListener listener : listenerList.getListeners(ActionListener.class)) {
            listener.actionPerformed(event);
        }
    }

    boolean canUndo() {
        return undoManager.canUndo();
    }

    boolean canRedo() {
        return undoManager.canRedo();
    }

    @Override
    public void append(String string) {
        super.append(string);
        homePosition += string.length();
        setCaretPosition(getEndPosition());
    }

    /**
     * bZ[W̖m͈͂uB
     * m͈͂Ƃ́Avvg烁bZ[W̖܂ł̂ƁB
     * @param string u镶
     */
    void replace(String string) {
        replaceRange(string, homePosition, getEndPosition());
    }

    /**
     * bZ[WNAB
     */
    void clear() {
        homePosition = 0;
        setText(EMPTY_STRING);
    }

    /**
     * ҏW\ȃeLXg̎擾B
     * @return ҏW\ȃeLXg
     */
    String getEditableText() {
        try {
            return getText(homePosition, getEndPosition() - homePosition);
        } catch (BadLocationException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * w肵ʒuҏW\ǂ𒲍B
     * @param position ʒu
     * @return w肵ʒuҏW\Ȃ<code>true</code>AłȂ<code>false</code>
     */
    boolean isEditablePosition(int position) {
        return (position >= homePosition);
    }

    /**
     * ҏWJnʒu̎擾B
     * @return ҏWJnʒu
     */
    int getHomePosition() {
        return homePosition;
    }

    /**
     * ʒu̎擾B
     * @return ʒu
     */
    int getEndPosition() {
        Document document = getDocument();
        Position position = document.getEndPosition();
        return position.getOffset() - 1;
    }

    /**
     * ҏWJnʒu̍ĐݒB 
     */
    void resetHomePosition() {
        undoManager.discardAllEdits();
        homePosition = getEndPosition();
    }

    /**
     * f𑗐MB
     */
    void sendBreak() {
        append(Resource.getString("ConsoleTextArea.breakprompt") + " ");
        resetHomePosition();
        validate();
    }

    @Override
    public void updateUI() {
        if (getCaret() == null) {
            super.updateUI();
        } else {
            final int p = getCaretPosition();
            super.updateUI();
            setCaretPosition(p);
        }
    }

    /**
     * ConsoleTextAreâ߂̃hLgtB^B
     */
    private static final class PrivateDocumentFilter extends DocumentFilter {

        private final ConsoleTextArea consoleTextArea;

        PrivateDocumentFilter(ConsoleTextArea consoleTextArea) {
            this.consoleTextArea = consoleTextArea;
        }

        @Override
        public void insertString(FilterBypass fb, int offset, String string, AttributeSet attr) throws BadLocationException {
            if (consoleTextArea.isEditablePosition(offset)) {
                super.insertString(fb, offset, string, attr);
            }
        }

        @Override
        public void remove(FilterBypass fb, int offset, int length) throws BadLocationException {
            if (consoleTextArea.isEditablePosition(offset)) {
                super.remove(fb, offset, length);
            }
        }

        @Override
        public void replace(FilterBypass fb, int offset, int length, String text, AttributeSet attrs) throws BadLocationException {
            if (consoleTextArea.isEditablePosition(offset)) {
                super.replace(fb, offset, length, text, attrs);
            }
        }

    }

}
