/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.command;

import java.io.*;
import java.sql.*;

import net.argius.logging.*;
import net.argius.stew.*;

/**
 * DownloadR}hB
 * w肵f[^t@CɕۑB
 */
public final class Download extends Command {

    private static final Logger log = LoggerFactory.getLogger(Download.class);

    @Override
    public void execute(Connection conn, Parameter parameter) throws CommandException {
        if (!parameter.has(2)) {
            throw new UsageException(getUsage());
        }
        final String root = parameter.at(1);
        final String sql = parameter.after(2);
        if (log.isDebugEnabled()) {
            log.debug("root: " + root);
            log.debug("SQL: " + sql);
        }
        try {
            Statement stmt = prepareStatement(conn, parameter.asString());
            try {
                ResultSet rs = executeQuery(stmt, sql);
                try {
                    download(rs, root);
                } finally {
                    rs.close();
                }
            } finally {
                stmt.close();
            }
        } catch (IOException ex) {
            throw new CommandException(ex);
        } catch (SQLException ex) {
            throw new CommandException(ex);
        }
    }

    private void download(ResultSet rs, String root) throws IOException, SQLException {
        ResultSetMetaData meta = rs.getMetaData();
        final int columnCount = meta.getColumnCount();
        assert columnCount >= 1;
        final int columnType = meta.getColumnType(1);
        final boolean isBinary;
        switch (columnType) {
            case Types.BLOB:
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY:
                // XXX ͖̔
                isBinary = true;
                break;
            default:
                isBinary = false;
        }
        byte[] bytes = new byte[8192];
        int count = 0;
        while (rs.next()) {
            ++count;
            StringBuilder buffer = new StringBuilder();
            for (int i = 2; i <= columnCount; i++) {
                buffer.append(rs.getString(i));
            }
            InputStream is = (isBinary) ? rs.getBinaryStream(1) : rs.getAsciiStream(1);
            try {
                final File path = resolvePath(root);
                final File file = (columnCount == 1) ? path : new File(path, buffer.toString());
                if (file.exists()) {
                    throw new IOException(getMessage("e.file-already-exists",
                                                     file.getAbsolutePath()));
                }
                final File dir = file.getParentFile();
                if (!dir.isDirectory()) {
                    if (log.isDebugEnabled()) {
                        log.debug(String.format("mkdir [%s]", dir.getAbsolutePath()));
                    }
                    if (dir.mkdirs()) {
                        outputMessage("i.did-mkdir", dir);
                    } else {
                        throw new IOException(getMessage("e.failed-mkdir-filedir", file));
                    }
                }
                long transfered = 0L;
                OutputStream os = new FileOutputStream(file);
                try {
                    while (true) {
                        int read = is.read(bytes);
                        if (read <= 0) {
                            break;
                        }
                        os.write(bytes, 0, read);
                        transfered += read;
                    }
                } finally {
                    os.close();
                }
                outputMessage("i.downloaded", transfered / 1024f, file);
            } finally {
                is.close();
            }
        }
        outputMessage("i.selected", count);
    }

}
