/*
 *  Copyright 2010 argius
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */
package net.argius.stew.ui.window;

import static java.awt.event.KeyEvent.*;
import static javax.swing.JOptionPane.*;
import static javax.swing.KeyStroke.getKeyStroke;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import net.argius.stew.ui.window.TextSearch.*;

/**
 * eLXgplB
 */
final class TextSearchPanel extends JPanel {

    private final List<TextSearch> targets;
    private final JTextField text;
    private final JCheckBox useRegexCheck;
    private final JCheckBox ignoreCaseCheck;

    private TextSearch currentTarget;
    private boolean searchBackward;

    TextSearchPanel(JFrame frame) {
        // [CX^X]
        this.targets = new ArrayList<TextSearch>();
        this.text = new JTextField(24);
        this.useRegexCheck = new JCheckBox(getMessage("useregex"));
        this.ignoreCaseCheck = new JCheckBox(getMessage("ignorecase"));
        setLayout(new FlowLayout());
        setVisible(false);
        final JButton close = new JButton(Resource.getImageIcon("close.png"));
        final JTextField text = this.text;
        final JButton forward = new JButton(getMessage("forward"));
        final JButton backward = new JButton(getMessage("backward"));
        final JCheckBox useRegexCheck = this.useRegexCheck;
        final JCheckBox ignoreCaseCheck = this.ignoreCaseCheck;
        // [CAEg]
        // FIXME EBhEƉẼR|[lg؂Ă܂
        close.setToolTipText(getMessage("close"));
        close.setMargin(new Insets(0, 0, 0, 0));
        forward.setMargin(new Insets(0, 7, 0, 7));
        backward.setMargin(new Insets(0, 7, 0, 7));
        add(close);
        add(new JLabel("     "));
        add(new JLabel(getMessage("label")));
        add(text);
        add(forward);
        add(backward);
        add(useRegexCheck);
        add(ignoreCaseCheck);
        // [tH[JXݒ]
        final FocusTraversalPolicy parentPolicy = frame.getFocusTraversalPolicy();
        frame.setFocusTraversalPolicy(new LayoutFocusTraversalPolicy() {

            @Override
            public Component getComponentAfter(Container focusCycleRoot, Component component) {
                if (component == ignoreCaseCheck) {
                    return close;
                }
                return parentPolicy.getComponentAfter(focusCycleRoot, component);
            }

            @Override
            public Component getComponentBefore(Container focusCycleRoot, Component component) {
                if (component == close) {
                    return ignoreCaseCheck;
                }
                return parentPolicy.getComponentBefore(focusCycleRoot, component);
            }

        });
        // [Cxgݒ]
        final Action closeAction = new AbstractAction("close") {

            public void actionPerformed(ActionEvent e) {
                setVisible(false);
            }

        };
        close.addActionListener(closeAction);
        for (JComponent c : new JComponent[]{close, text, forward, backward, useRegexCheck,
                                             ignoreCaseCheck}) {
            ActionUtility.getInstance(c).bindAction(closeAction, getKeyStroke(VK_ESCAPE, 0));
        }
        ActionUtility.getInstance(text).bindAction(new AbstractAction("search") {

            public void actionPerformed(ActionEvent e) {
                startSearch();
            }

        }, getKeyStroke(VK_ENTER, 0));
        forward.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                setSearchBackward(false);
                startSearch();
            }

        });
        backward.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                setSearchBackward(true);
                startSearch();
            }

        });
        addComponentListener(new ComponentAdapter() {

            @Override
            public void componentResized(ComponentEvent e) {
                validate();
            }

        });
    }

    @Override
    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
            text.requestFocus();
        } else {
            for (TextSearch target : targets) {
                target.reset();
            }
            if (currentTarget instanceof Component) {
                Component c = (Component)currentTarget;
                c.requestFocus();
            }
        }
    }

    /**
     * Ώۂ̒ǉB
     * @param target ^[Qbg
     */
    void addTarget(TextSearch target) {
        targets.add(target);
    }

    /**
     * Ώۂ̍폜B
     * @param target ^[Qbg
     */
    void removeTarget(TextSearch target) {
        targets.remove(target);
    }

    /**
     * ݂̌Ώۂݒ肷B
     * @param currentTarget ݂̌Ώ
     */
    void setCurrentTarget(TextSearch currentTarget) {
        this.currentTarget = currentTarget;
    }

    /**
     * tɌ邩ǂݒ肷B
     * @param searchBackward tɌ邩ǂ
     */
    void setSearchBackward(boolean searchBackward) {
        this.searchBackward = searchBackward;
    }

    /**
     * JnB
     */
    void startSearch() {
        final String s = text.getText();
        if (s == null || s.length() == 0 || currentTarget == null) {
            return;
        }
        final boolean useRegularExpression = useRegexCheck.isSelected();
        final boolean ignoreCase = ignoreCaseCheck.isSelected();
        Matcher matcher = new Matcher(s, useRegularExpression, ignoreCase);
        matcher.setBackward(searchBackward);
        matcher.setContinuously(true);
        for (TextSearch target : targets) {
            target.reset();
        }
        final boolean found = currentTarget.search(matcher);
        if (!found) {
            Component parent;
            if (currentTarget instanceof Component) {
                parent = ((Component)currentTarget).getParent();
            } else {
                parent = getParent();
            }
            showMessageDialog(parent, getMessage("message.notfound", s), null, WARNING_MESSAGE);
        }
    }

    private String getMessage(String key, Object... args) {
        return Resource.getString("TextSearchPanel." + key, args);
    }

}
