#version 330 core

layout(location = 0) in vec3 vertexPos;

// note that the default interpolation qualifier is smooth,
// where the choices are a) flat, b) noperspective, c) smooth
// EG:  smooth out vec3 worldNormal

out vec3 position;
out vec3 worldNormal;
out vec2 waveDirection;

uniform mat4 MVP;

/////////////////////////////// new 21dec14 begin
out float depth;
out float wlev;

uniform float hmax;
//uniform float r1;
uniform float r2;
/////////////////////////////// new 21dec14 end

uniform float waterlevel;
uniform float mytime;

uniform vec3 wPos;
uniform vec3 wRad;

const float pi = 3.14159;
const float twopi = 2.0*pi;


// recall r2 = 7 = radius where sand meets ocean
// recall r1 = 5 = radius @ max height for atoll
// circular waves begin to diminish here:
const float channelRadius = 7.0; //6.0;

// ...and disappear by here:
const float lagoonRadius = 3.5;


const int numWaves=4;


const float amplitude[4] = float[4]( 0.01, 0.02, 0.014, 0.018 );

const float wavelength[4]= float[4]( 4,1,2,3 );

const float speed[4]     = float[4]( 0.25, 0.3, 0.4, 0.5 );
const float dir[4] = float[4]( -0.8*pi, -0.7*pi, -0.6*pi, -0.5*pi );

const float diry[4] = float[4]( sin(dir[0]), sin(dir[1]), sin(dir[2]), sin(dir[3]) );
const float dirx[4] = float[4]( cos(dir[0]), cos(dir[1]), cos(dir[2]), cos(dir[3]) );


vec2 waveDir(int i) {
	return vec2( dirx[i], diry[i] );
}


// this version has radially-decreasing amplitudes
// for a distance effect...modified 31dec14
// ...looks Ok but would like smaller wavelengths @ greater distance
// but that complicates the derivatives.

float wave(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+1;
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( waveDir(i), vec2(x, y));
    return amplitude[i]/r * sin(theta*frequency+mytime*phase);

	 // W = A*f/r
	 // so quotient rule says
	 // W' = A*f'/r - A*f*r'/r^2
	 // where dr/dx = x/sqrt(x*x+y*y) = x/rr ~= x/r
	 // and similarly dr/dy = y/rr ~= y/r
	 // thus ( usr approximation r when rr is not bounded away from zero )
	 // dW/dx ~= A/r * df/dx - A/r/r * f * x/r
	 // dW/dy ~= A/r * df/dy - A/r/r * f * y/r
}

float waveHeight(float x, float y) {
    float height = 0.0;
    for (int i = 0; i < numWaves; ++i)
        height += wave(i, x, y);
    return height;
}

float dWavedx(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+1;
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( vec2( dirx[i], diry[i]), vec2(x, y));
    float A = amplitude[i];

    // use x/r to approximate the singularity:  x/sqrt(x*x+y*y)
	 return
	 	A/r * dirx[i]*frequency * cos(theta*frequency+mytime*phase) -
		A/r/r * x/rr            * sin(theta*frequency+mytime*phase);
}

float dWavedy(int i, float x, float y) {
	float rr=sqrt(x*x+y*y);
	float r=rr+1;
    float frequency = 2*pi/wavelength[i];
    float phase = speed[i] * frequency;
    float theta = dot( vec2( dirx[i], diry[i]), vec2(x, y));
    float A = amplitude[i];

	 // use y/r to approximate the singularity:  y/sqrt(x*x+y*y)
    return  
	 	A/r * diry[i]*frequency * cos(theta*frequency+mytime*phase) -
		A/r/r * y/rr            * sin(theta*frequency+mytime*phase);
}



//////////////////////// begin circular wave insert //////////////////////

const float cspeed=0.3;
const float cAmp=0.03;

float f(float r) {
	float a = -pi * (mytime*cspeed + r);
	return 0.5 - 0.5*sin(a) - 0.5*sin(2.0*a)/2 - 0.5*sin(3.0*a)/3.0 - 0.5*sin(4.0*a)/4.0;
}

float F(float x, float z) {
	float r = sqrt(x*x+z*z);

	float diminish = (r-lagoonRadius)/(channelRadius-lagoonRadius);
	if( diminish>1 ) diminish = 1;

	return cAmp*f(r)*diminish;
}

float dFdx(float x, float z) {
	float r=sqrt(x*x+z*z);

	float diminish = (r-lagoonRadius)/(channelRadius-lagoonRadius);
	if( diminish>1 ) diminish = 1;

	float a = -pi * (mytime*cspeed + r);
	float drdx = x/r;
	float dFdf = cAmp;
	float dfda = -0.5*cos(a)-0.5*cos(2.0*a)-0.5*cos(3.0*a)-0.5*cos(4.0*a);
	float dadr = -pi;
	return diminish*dFdf*dfda*dadr*drdx;
}

float dFdz(float x, float z) {
	float r=sqrt(x*x+z*z);

	float diminish = (r-lagoonRadius)/(channelRadius-lagoonRadius);
	if( diminish>1 ) diminish = 1;

	float a = -pi * (mytime*cspeed + r);
	float drdz = z/r;
	float dFdf = cAmp;
	float dfda = -0.5*cos(a)-0.5*cos(2.0*a)-0.5*cos(3.0*a)-0.5*cos(4.0*a);
	float dadr = -pi;
	return diminish*dFdf*dfda*dadr*drdz;
}

///////////////////////// end circular wave insert /////////////////////////







vec3 waveNormal(float x, float z) {
    float dx = 0.0;
    float dz = 0.0;

    for (int i = 0; i < numWaves; ++i) {
        dx += dWavedx(i, x, z);
        dz += dWavedy(i, x, z);
    }

	 dx += dFdx(x,z); // addenda for circular waves
	 dz += dFdz(x,z);

    vec3 n = vec3(-dx, 1.0, -dz); //my Y plays roll of Conrod's Z
    return normalize(n);
}



// WARNING:  this must match function in gameutils.adb
// waterlevel is a uniform (currently -0.5)
// hmax = 0.25 = maxHt [above MSL] of island sand @ center
// r3 =20
// r2 = 7 = radius where sand meets ocean (sandht=waterlevel)
// r1 = 5 = radius @ max height for atoll
float sandht( float x, float z ) {
	float ht;
	float r = sqrt( x*x + z*z );
	float a = -hmax/r2/r2;

	ht = a*r*r + hmax+waterlevel;

	return ht;

} // sandht(10)~ -0.7, sandht(20)~ -2.25




//////////////// begin inner insert //////////////////////////////
const float iamplitude[4] = float[4]( 0.0005, 0.0004, 0.0003, 0.0002 );
//const float iwavelength[4]= float[4]( 0.16, 0.28, 0.32, 0.20 );
const float iwavelength[4]= float[4]( 0.32, 0.56, 0.64, 0.40 );
//const float ispeed[4]     = float[4]( 0.07, 0.03, 0.05, 0.1 );
const float ispeed[4]     = float[4]( 0.14, 0.08, 0.10, 0.16 );
//const float idir[4] = float[4]( 2*pi/3, 5*pi/6, 3*pi/10, 3*pi/2 );
const float idir[4] = float[4]( -pi/3, -pi/6, 13*pi/10, pi/2 );
const float idiry[4] = float[4]( sin(idir[0]), sin(idir[1]), sin(idir[2]), sin(idir[3]) );
const float idirx[4] = float[4]( cos(idir[0]), cos(idir[1]), cos(idir[2]), cos(idir[3]) );


float iwave(int i, float x, float y) {
    float frequency = 2*pi/iwavelength[i];
    float phase = ispeed[i] * frequency;
    float theta = dot( vec2( idirx[i], idiry[i] ), vec2(x, y));
    return iamplitude[i] * sin(theta*frequency+mytime*phase);
}

float iwaveHeight(float x, float y) {
    float height = 0.0;
    for (int i = 0; i < numWaves; ++i)
        height += iwave(i, x, y);
    return height;
}

float idWavedx(int i, float x, float y) {
    float frequency = 2*pi/iwavelength[i];
    float phase = ispeed[i] * frequency;
    float theta = dot( vec2( idirx[i], idiry[i]), vec2(x, y));
    float A = iamplitude[i] * idirx[i] * frequency;
    return A * cos(theta * frequency + mytime * phase);
}

float idWavedy(int i, float x, float y) {
    float frequency = 2*pi/iwavelength[i];
    float phase = ispeed[i] * frequency;
    float theta = dot( vec2( idirx[i], idiry[i]), vec2(x, y));
    float A = iamplitude[i] * idiry[i] * frequency;
    return A * cos(theta * frequency + mytime * phase);
}

vec3 iwaveNormal(float x, float y) {
    float dx = 0.0;
    float dy = 0.0;
    for (int i = 0; i < numWaves; ++i) {
        dx += idWavedx(i, x, y);
        dy += idWavedy(i, x, y);
    }
    //vec3 n = vec3(-dx, -dy, 1.0); --Conrod:  Z is up
    vec3 n = vec3(-dx, 1.0, -dy); //my Y plays roll of Conrod's Z
    return normalize(n);
}



//////////////// end inner insert //////////////////////////////







void main(){

	vec4 pos = vec4(vertexPos,1);
	float norm=sqrt(pos.x*pos.x+pos.z*pos.z);

if( norm < lagoonRadius ) // r1==5
{

  	pos.y = waterlevel + iwaveHeight(pos.x, pos.z);
	depth = pos.y - sandht(pos.x,pos.z);
  	worldNormal = iwaveNormal(pos.x, pos.z);
   position = pos.xyz/pos.w;
   gl_Position = MVP * pos;

}

else
{

	//float sumwt=0.0;
	float sumwt=cAmp; // 21jul15 addendum
	for(int i=0; i<numWaves; ++i) sumwt += amplitude[i];

	waveDirection=vec2(0,0);
	for(int i=0; i<numWaves; ++i) waveDirection += waveDir(i)*amplitude[i]/sumwt;

	waveDirection += cAmp/sumwt * (-pos.x/norm,-pos.z/norm); // 21jul15 addendum

	normalize(waveDirection);

  	pos.y = waterlevel + waveHeight(pos.x, pos.z);
	pos.y += F(pos.x, pos.z);
  	worldNormal = waveNormal(pos.x, pos.z);

   position = pos.xyz/pos.w;
	depth = (pos.y - sandht(pos.x,pos.z));
	wlev = waterlevel;
   gl_Position = MVP * pos;

} // end outer if

}

// this version calculates water depth and sends to frag-shader
// so it can adjust opacity according to depth,
// and so it can determine the surf line for foam.

//--
//-- Copyright (C) 2016  <fastrgv@gmail.com>
//--
//-- This program is free software: you can redistribute it and/or modify
//-- it under the terms of the GNU General Public License as published by
//-- the Free Software Foundation, either version 3 of the License, or
//-- (at your option) any later version.
//--
//-- This program is distributed in the hope that it will be useful,
//-- but WITHOUT ANY WARRANTY; without even the implied warranty of
//-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//-- GNU General Public License for more details.
//--
//-- You may read the full text of the GNU General Public License
//-- at <http://www.gnu.org/licenses/>.
//--

