/*
 * Decompiled with CFR 0.152.
 */
package org.exist.dom.persistent;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.EXistException;
import org.exist.backup.RawDataBackup;
import org.exist.dom.QName;
import org.exist.dom.persistent.QNamePool;
import org.exist.storage.BrokerPool;
import org.exist.storage.BrokerPoolService;
import org.exist.storage.BrokerPoolServiceException;
import org.exist.storage.DBBroker;
import org.exist.storage.io.VariableByteInput;
import org.exist.storage.io.VariableByteInputStream;
import org.exist.storage.io.VariableByteOutputStream;
import org.exist.util.Configuration;
import org.exist.util.FileUtils;
import org.exist.util.hashtable.Object2IntHashMap;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

public class SymbolTable
implements BrokerPoolService,
Closeable {
    private static final Logger LOG = LogManager.getLogger(SymbolTable.class);
    private static final String FILE_NAME = "symbols.dbx";
    public static final short FILE_FORMAT_VERSION_ID = 8;
    public static final short LEGACY_FILE_FORMAT_VERSION_ID = 7;
    public static final int LENGTH_LOCAL_NAME = 2;
    public static final int LENGTH_NS_URI = 2;
    public static final char ATTR_NAME_PREFIX = '@';
    protected final SymbolCollection localNameSymbols = new LocalNameSymbolCollection(SymbolType.NAME, 200);
    protected final SymbolCollection namespaceSymbols = new SymbolCollection(SymbolType.NAMESPACE, 200);
    protected final SymbolCollection mimeTypeSymbols = new SymbolCollection(SymbolType.MIMETYPE, 32);
    private final QNamePool namePool = new QNamePool();
    private boolean changed = false;
    private Path file;
    private final VariableByteOutputStream outBuffer = new VariableByteOutputStream(512);
    private OutputStream os = null;

    @Override
    public void configure(Configuration configuration) {
        Path dataDir = (Path)configuration.getProperty("db-connection.data-dir");
        this.file = dataDir.resolve(SymbolTable.getFileName());
    }

    @Override
    public void prepare(BrokerPool pool) throws BrokerPoolServiceException {
        try {
            if (!Files.isReadable(this.file)) {
                this.saveSymbols();
            } else {
                this.loadSymbols();
            }
        }
        catch (EXistException e) {
            throw new BrokerPoolServiceException(e);
        }
    }

    @Override
    public void stop(DBBroker systemBroker) throws BrokerPoolServiceException {
        try {
            this.close();
        }
        catch (IOException e) {
            throw new BrokerPoolServiceException(e);
        }
    }

    public static final String getFileName() {
        return FILE_NAME;
    }

    public synchronized QName getQName(short type, String namespaceURI, String localName, String prefix) {
        byte itype = type == 2 ? (byte)1 : 0;
        QName qn = this.namePool.get(itype, namespaceURI, localName, prefix);
        if (qn == null) {
            qn = this.namePool.add(itype, namespaceURI, localName, prefix);
        }
        return qn;
    }

    public synchronized short getSymbol(Element element) {
        return (short)this.localNameSymbols.getId(element.getLocalName());
    }

    public synchronized short getSymbol(Attr attr) {
        String key = '@' + attr.getLocalName();
        return (short)this.localNameSymbols.getId(key);
    }

    public synchronized short getSymbol(String name) {
        if (name.length() == 0) {
            throw new IllegalArgumentException("name is empty");
        }
        return (short)this.localNameSymbols.getId(name);
    }

    public synchronized short getNSSymbol(String ns) {
        if (ns == null || ns.length() == 0) {
            return 0;
        }
        return (short)this.namespaceSymbols.getId(ns);
    }

    public synchronized int getMimeTypeId(String mimeType) {
        return this.mimeTypeSymbols.getId(mimeType);
    }

    public synchronized boolean hasChanged() {
        return this.changed;
    }

    public synchronized String getName(short id) {
        return this.localNameSymbols.getSymbol(id);
    }

    public synchronized String getMimeType(int id) {
        return this.mimeTypeSymbols.getSymbol(id);
    }

    public synchronized String getNamespace(short id) {
        return this.namespaceSymbols.getSymbol(id);
    }

    private synchronized void writeAll(VariableByteOutputStream os) throws IOException {
        os.writeFixedInt(8);
        this.localNameSymbols.write(os);
        this.namespaceSymbols.write(os);
        this.mimeTypeSymbols.write(os);
        this.changed = false;
    }

    protected final void read(VariableByteInput is) throws IOException {
        this.localNameSymbols.clear();
        this.namespaceSymbols.clear();
        this.mimeTypeSymbols.clear();
        while (is.available() > 0) {
            this.readEntry(is);
        }
    }

    private void readEntry(VariableByteInput is) throws IOException {
        byte type = is.readByte();
        int id = is.readInt();
        String key = is.readUTF();
        switch (SymbolType.valueOf(type)) {
            case NAME: {
                this.localNameSymbols.add(id, key);
                break;
            }
            case NAMESPACE: {
                this.namespaceSymbols.add(id, key);
                break;
            }
            case MIMETYPE: {
                this.mimeTypeSymbols.add(id, key);
            }
        }
    }

    protected final void readLegacy(VariableByteInput istream) throws IOException {
        short id;
        String key;
        int i;
        istream.readShort();
        istream.readShort();
        int count = istream.readInt();
        for (i = 0; i < count; ++i) {
            key = istream.readUTF();
            id = istream.readShort();
            this.localNameSymbols.add(id, key);
        }
        count = istream.readInt();
        for (i = 0; i < count; ++i) {
            key = istream.readUTF();
            id = istream.readShort();
            this.namespaceSymbols.add(id, key);
        }
        count = istream.readInt();
        for (i = 0; i < count; ++i) {
            istream.readUTF();
            istream.readShort();
        }
        count = istream.readInt();
        for (int i2 = 0; i2 < count; ++i2) {
            key = istream.readUTF();
            int mimeId = istream.readInt();
            this.mimeTypeSymbols.add(mimeId, key);
        }
        this.changed = false;
    }

    public final Path getFile() {
        return this.file;
    }

    private void saveSymbols() throws EXistException {
        try (VariableByteOutputStream os = new VariableByteOutputStream(256);
             OutputStream fos = Files.newOutputStream(this.getFile(), new OpenOption[0]);){
            this.writeAll(os);
            fos.write(os.toByteArray());
        }
        catch (FileNotFoundException e) {
            throw new EXistException("File not found: " + this.getFile().toAbsolutePath().toString(), e);
        }
        catch (IOException e) {
            throw new EXistException("IO error occurred while creating " + this.getFile().toAbsolutePath().toString(), e);
        }
    }

    private synchronized void loadSymbols() throws EXistException {
        try (InputStream fis = Files.newInputStream(this.getFile(), new OpenOption[0]);){
            VariableByteInputStream is = new VariableByteInputStream(fis);
            int magic = is.readFixedInt();
            if (magic == 7) {
                LOG.info("Converting legacy symbols.dbx to new format...");
                this.readLegacy(is);
                this.saveSymbols();
            } else {
                if (magic != 8) {
                    throw new EXistException("Symbol table was created by an olderor newer version of eXist (file id: " + magic + "). To avoid damage, the database will stop.");
                }
                this.read(is);
            }
        }
        catch (FileNotFoundException e) {
            throw new EXistException("Could not read " + this.getFile().toAbsolutePath().toString(), e);
        }
        catch (IOException e) {
            throw new EXistException("IO error occurred while reading " + this.getFile().toAbsolutePath().toString() + ": " + e.getMessage(), e);
        }
    }

    public void backupSymbolsTo(OutputStream os) throws IOException {
        Files.copy(this.getFile(), os);
    }

    public void backupToArchive(RawDataBackup backup) throws IOException {
        try {
            OutputStream os = backup.newEntry(FileUtils.fileName(this.getFile()));
            this.backupSymbolsTo(os);
        }
        finally {
            backup.closeEntry();
        }
    }

    public void flush() throws EXistException {
    }

    private OutputStream getOutputStream() throws IOException {
        if (this.os == null) {
            this.os = Files.newOutputStream(this.getFile(), StandardOpenOption.APPEND);
        }
        return this.os;
    }

    @Override
    public void close() throws IOException {
        if (this.os != null) {
            this.os.close();
        }
    }

    private class LocalNameSymbolCollection
    extends SymbolCollection {
        public LocalNameSymbolCollection(SymbolType symbolType, int initialSize) {
            super(symbolType, initialSize);
        }

        @Override
        protected void addSymbolById(int id, String name) {
            if (name.charAt(0) == '@') {
                super.addSymbolById(id, name.substring(1));
            } else {
                super.addSymbolById(id, name);
            }
        }
    }

    protected class SymbolCollection {
        private final SymbolType symbolType;
        private final Object2IntHashMap<String> symbolsByName;
        private String[] symbolsById;
        protected short offset = 0;

        public SymbolCollection(SymbolType symbolType, int initialSize) {
            this.symbolType = symbolType;
            this.symbolsByName = new Object2IntHashMap(initialSize);
            this.symbolsById = new String[initialSize];
        }

        private SymbolType getSymbolType() {
            return this.symbolType;
        }

        private int add(int id, String name) {
            this.symbolsById = this.ensureCapacity(this.symbolsById, id);
            this.addSymbolById(id, name);
            this.addSymbolByName(name, id);
            if (id > this.offset) {
                this.offset = (short)id;
            }
            return id;
        }

        protected void addSymbolById(int id, String name) {
            this.symbolsById[id] = name;
        }

        protected void addSymbolByName(String name, int id) {
            this.symbolsByName.put(name, id);
        }

        protected String[] ensureCapacity(String[] array, int max) {
            if (array.length <= max) {
                String[] newArray = new String[max * 3 / 2];
                System.arraycopy(array, 0, newArray, 0, array.length);
                return newArray;
            }
            return array;
        }

        private void clear() {
            this.offset = 0;
        }

        public synchronized String getSymbol(int id) {
            if (id <= 0 || id > this.offset) {
                return "";
            }
            return this.symbolsById[id];
        }

        public synchronized int getId(String name) {
            int id = this.symbolsByName.get(name);
            if (id != -1) {
                return id;
            }
            if (this.offset == Short.MAX_VALUE) {
                return -1;
            }
            this.offset = (short)(this.offset + 1);
            id = this.add(this.offset, name);
            this.write(id, name);
            SymbolTable.this.changed = true;
            return id;
        }

        protected final void write(VariableByteOutputStream os) throws IOException {
            for (String symbol : this.symbolsByName) {
                int id = this.symbolsByName.get(symbol);
                if (id < 0) {
                    LOG.error("Symbol Table: symbolTypeId=" + (Object)((Object)this.getSymbolType()) + ", symbol='" + symbol + "', id=" + id);
                }
                this.writeEntry(id, symbol, os);
            }
        }

        private void write(int id, String key) {
            SymbolTable.this.outBuffer.clear();
            try {
                this.writeEntry(id, key, SymbolTable.this.outBuffer);
                SymbolTable.this.getOutputStream().write(SymbolTable.this.outBuffer.toByteArray());
                SymbolTable.this.getOutputStream().flush();
            }
            catch (FileNotFoundException e) {
                LOG.error("Symbol table: file not found!", (Throwable)e);
            }
            catch (IOException e) {
                LOG.error("Symbol table: caught exception while writing!", (Throwable)e);
            }
        }

        private void writeEntry(int id, String key, VariableByteOutputStream os) throws IOException {
            os.writeByte(this.getSymbolType().getTypeId());
            os.writeInt(id);
            os.writeUTF(key);
        }
    }

    public static enum SymbolType {
        NAME(0),
        NAMESPACE(1),
        MIMETYPE(2);

        private final byte typeId;

        private SymbolType(byte typeId) {
            this.typeId = typeId;
        }

        public final byte getTypeId() {
            return this.typeId;
        }

        public static SymbolType valueOf(byte typeId) {
            for (SymbolType symbolType : SymbolType.values()) {
                if (symbolType.getTypeId() != typeId) continue;
                return symbolType;
            }
            throw new IllegalArgumentException("No such enumerated value for typeId:" + typeId);
        }
    }
}

