/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.btree;

import java.io.IOException;
import java.io.PrintStream;
import java.io.StringWriter;
import java.io.Writer;
import java.nio.file.Path;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.storage.BrokerPool;
import org.exist.storage.BufferStats;
import org.exist.storage.DefaultCacheManager;
import org.exist.storage.btree.BTAbstractLoggable;
import org.exist.storage.btree.BTreeCallback;
import org.exist.storage.btree.BTreeException;
import org.exist.storage.btree.CreateBTNodeLoggable;
import org.exist.storage.btree.DBException;
import org.exist.storage.btree.IndexQuery;
import org.exist.storage.btree.InsertValueLoggable;
import org.exist.storage.btree.Paged;
import org.exist.storage.btree.RemoveValueLoggable;
import org.exist.storage.btree.SetPageLinkLoggable;
import org.exist.storage.btree.SetParentLoggable;
import org.exist.storage.btree.TreeMetrics;
import org.exist.storage.btree.UpdatePageLoggable;
import org.exist.storage.btree.UpdateValueLoggable;
import org.exist.storage.btree.Value;
import org.exist.storage.cache.BTreeCache;
import org.exist.storage.cache.BTreeCacheable;
import org.exist.storage.cache.Cache;
import org.exist.storage.journal.JournalException;
import org.exist.storage.journal.JournalManager;
import org.exist.storage.journal.LogEntryTypes;
import org.exist.storage.journal.LogException;
import org.exist.storage.journal.Loggable;
import org.exist.storage.lock.Lock;
import org.exist.storage.txn.Txn;
import org.exist.util.ByteConversion;
import org.exist.util.FileUtils;
import org.exist.util.Lockable;
import org.exist.xquery.TerminatedException;

public class BTree
extends Paged
implements Lockable {
    protected static final Logger LOGSTATS = LogManager.getLogger((String)"org.exist.statistics");
    public static final long KEY_NOT_FOUND = -1L;
    protected static final byte LEAF = 1;
    protected static final byte BRANCH = 2;
    protected static final int MIN_SPACE_PER_KEY = 32;
    public static final byte LOG_INSERT_VALUE = 32;
    public static final byte LOG_CREATE_BNODE = 33;
    public static final byte LOG_UPDATE_PAGE = 34;
    public static final byte LOG_SET_PARENT = 35;
    public static final byte LOG_UPDATE_VALUE = 36;
    public static final byte LOG_REMOVE_VALUE = 37;
    public static final byte LOG_SET_LINK = 38;
    private final BrokerPool pool;
    protected final DefaultCacheManager cacheManager;
    protected Cache<BTreeNode> cache;
    private final BTreeFileHeader fileHeader;
    protected final Optional<JournalManager> logManager;
    protected final byte fileId;
    private double splitFactor = -1.0;

    protected BTree(BrokerPool pool, byte fileId, boolean recoveryEnabled, DefaultCacheManager cacheManager) throws DBException {
        super(pool);
        this.pool = pool;
        this.cacheManager = cacheManager;
        this.fileId = fileId;
        this.fileHeader = (BTreeFileHeader)this.getFileHeader();
        this.fileHeader.setPageCount(0L);
        this.fileHeader.setTotalCount(0L);
        this.logManager = recoveryEnabled && pool.isRecoveryEnabled() ? pool.getJournalManager() : Optional.empty();
    }

    protected boolean isRecoveryEnabled() {
        return this.logManager.isPresent() && this.pool.isRecoveryEnabled();
    }

    public BTree(BrokerPool pool, byte fileId, boolean recoveryEnabled, DefaultCacheManager cacheManager, Path file) throws DBException {
        this(pool, fileId, recoveryEnabled, cacheManager);
        this.setFile(file);
    }

    @Override
    public short getFileVersion() {
        return -1;
    }

    public boolean create(short fixedKeyLen) throws DBException {
        if (super.create()) {
            this.initCache();
            try {
                this.createRootNode(null);
            }
            catch (IOException e) {
                LOG.warn("Can not create database file " + this.getFile().toAbsolutePath().toString(), (Throwable)e);
                return false;
            }
            this.fileHeader.setFixedKeyLen(fixedKeyLen);
            try {
                this.fileHeader.write();
            }
            catch (IOException e) {
                throw new DBException("Error while writing file header: " + e.getMessage());
            }
        }
        return true;
    }

    @Override
    public boolean open(short expectedVersion) throws DBException {
        if (super.open(expectedVersion)) {
            this.initCache();
            return true;
        }
        return false;
    }

    @Override
    public void closeAndRemove() {
        super.closeAndRemove();
        this.cacheManager.deregisterCache(this.cache);
    }

    @Override
    public Lock getLock() {
        return null;
    }

    protected void initCache() {
        this.cache = new BTreeCache<BTreeNode>(FileUtils.fileName(this.getFile()), this.cacheManager.getDefaultInitialSize(), 1.5, 0.0, "BTREE");
        this.cacheManager.registerCache(this.cache);
    }

    protected void setSplitFactor(double factor) {
        if (factor > 1.0) {
            throw new IllegalArgumentException("splitFactor should be <= 1 > 0");
        }
        this.splitFactor = factor;
    }

    public long addValue(Value value, long pointer) throws IOException, BTreeException {
        return this.addValue(null, value, pointer);
    }

    public long addValue(Txn transaction, Value value, long pointer) throws IOException, BTreeException {
        return this.getRootNode().addValue(transaction, value, pointer);
    }

    public long removeValue(Value value) throws IOException, BTreeException {
        return this.removeValue(null, value);
    }

    public long removeValue(Txn transaction, Value value) throws IOException, BTreeException {
        return this.getRootNode().removeValue(transaction, value);
    }

    public void remove(IndexQuery query, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
        this.remove(null, query, callback);
    }

    public void remove(Txn transaction, IndexQuery query, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
        if (query != null && query.getOperator() == 7) {
            Value val1 = query.getValue(0);
            byte[] data1 = val1.getData();
            byte[] data2 = new byte[data1.length];
            System.arraycopy(data1, 0, data2, 0, data1.length);
            int n = data2.length - 1;
            data2[n] = (byte)(data2[n] + 1);
            query = new IndexQuery(query.getOperator(), val1, new Value(data2));
        }
        this.getRootNode().remove(transaction, query, callback);
    }

    protected void removeSequential(Txn transaction, BTreeNode page, IndexQuery query, BTreeCallback callback) throws TerminatedException {
        long next = page.pageHeader.getNextPage();
        while (next != -1L) {
            BTreeNode nextPage = this.getBTreeNode(next);
            for (int i = 0; i < nextPage.nKeys; ++i) {
                boolean test = query.testValue(nextPage.keys[i]);
                if (query.getOperator() != -1 && !test) {
                    return;
                }
                if (!test) continue;
                if (transaction != null && this.isRecoveryEnabled() && nextPage.pageHeader.getStatus() == 1) {
                    RemoveValueLoggable log = new RemoveValueLoggable(transaction, this.fileId, nextPage.page.getPageNum(), i, nextPage.keys[i], nextPage.ptrs[i]);
                    this.writeToLog(log, nextPage);
                }
                if (callback != null) {
                    callback.indexInfo(nextPage.keys[i], nextPage.ptrs[i]);
                }
                nextPage.removeKey(i);
                nextPage.removePointer(i);
                nextPage.recalculateDataLen();
                --i;
            }
            next = nextPage.pageHeader.getNextPage();
        }
    }

    public long findValue(Value value) throws IOException, BTreeException {
        return this.getRootNode().findValue(value);
    }

    public void query(IndexQuery query, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
        if (query != null && query.getOperator() == 7) {
            Value val1 = query.getValue(0);
            byte[] data1 = val1.getData();
            byte[] data2 = new byte[data1.length];
            System.arraycopy(data1, 0, data2, 0, data1.length);
            int n = data2.length - 1;
            data2[n] = (byte)(data2[n] + 1);
            query = new IndexQuery(query.getOperator(), val1, new Value(data2));
        }
        this.getRootNode().query(query, callback);
    }

    public void query(IndexQuery query, Value prefix, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
        this.getRootNode().query(query, prefix, callback);
    }

    protected void scanSequential(BTreeNode page, IndexQuery query, Value keyPrefix, BTreeCallback callback) throws TerminatedException {
        while (page != null) {
            for (int i = 0; i < page.nKeys; ++i) {
                if (keyPrefix != null && page.keys[i].comparePrefix(keyPrefix) > 0) {
                    return;
                }
                boolean test = query.testValue(page.keys[i]);
                if (query.getOperator() != -1 && !test) {
                    return;
                }
                if (!test) continue;
                callback.indexInfo(page.keys[i], page.ptrs[i]);
            }
            long next = page.pageHeader.getNextPage();
            if (next != -1L) {
                page = this.getBTreeNode(next);
                continue;
            }
            page = null;
        }
    }

    private BTreeNode createBTreeNode(Txn transaction, byte status, BTreeNode parent, boolean reuseDeleted) {
        try {
            Paged.Page page = this.getFreePage(reuseDeleted);
            BTreeNode node = new BTreeNode(page, true);
            if (transaction != null && this.isRecoveryEnabled() && status == 1) {
                CreateBTNodeLoggable loggable = new CreateBTNodeLoggable(transaction, this.fileId, status, page.getPageNum(), parent != null ? parent.page.getPageNum() : -1L);
                this.writeToLog(loggable, node);
            }
            node.pageHeader.setStatus(status);
            node.setPointers(new long[0]);
            node.setParent(parent);
            node.write();
            return node;
        }
        catch (IOException e) {
            LOG.error("Failed to create a BTree node", (Throwable)e);
            return null;
        }
    }

    private BTreeNode getBTreeNode(long pageNum) {
        try {
            BTreeNode node = this.cache.get(pageNum);
            if (node == null) {
                Paged.Page page = this.getPage(pageNum);
                node = new BTreeNode(page, false);
                node.read();
            }
            int increment = node.pageHeader.getStatus() == 2 ? 2 : 1;
            this.cache.add(node, increment);
            return node;
        }
        catch (IOException e) {
            LOG.error("Failed to get BTree node on page " + pageNum, (Throwable)e);
            return null;
        }
    }

    protected void setRootNode(BTreeNode rootNode) throws IOException {
        this.fileHeader.setRootPage(rootNode.page.getPageNum());
        this.fileHeader.write();
        this.cache.add(rootNode, 2);
    }

    protected long createRootNode(Txn transaction) throws IOException {
        BTreeNode root = this.createBTreeNode(transaction, (byte)1, null, true);
        this.setRootNode(root);
        return root.page.getPageNum();
    }

    protected BTreeNode getRootNode() {
        try {
            BTreeNode node = this.cache.get(this.fileHeader.getRootPage());
            if (node == null) {
                Paged.Page page = this.getPage(this.fileHeader.getRootPage());
                node = new BTreeNode(page, false);
                node.read();
            }
            this.cache.add(node, 2);
            return node;
        }
        catch (IOException e) {
            LOG.warn("Failed to get root btree node", (Throwable)e);
            return null;
        }
    }

    public void dump(Writer writer) throws IOException, BTreeException {
        BTreeNode root = this.getRootNode();
        LOG.debug("ROOT = " + root.page.getPageNum());
        root.dump(writer);
    }

    public TreeMetrics treeStatistics() throws IOException {
        TreeMetrics metrics = new TreeMetrics(FileUtils.fileName(this.getFile()));
        BTreeNode root = this.getRootNode();
        root.treeStatistics(metrics);
        return metrics;
    }

    @Override
    public boolean flush() throws DBException {
        boolean flushed = this.cache.flush();
        return flushed |= super.flush();
    }

    @Override
    public void close() throws DBException {
        if (!this.isReadOnly()) {
            this.flush();
        }
        super.close();
    }

    protected void dumpValue(Writer writer, Value value, int status) throws IOException {
        byte[] data = value.getData();
        writer.write(91);
        writer.write(Paged.hexDump(data));
        writer.write(93);
    }

    public void rawScan(IndexQuery query, BTreeCallback callback) throws IOException, TerminatedException {
        long pages = this.getFileHeader().getTotalCount();
        int i = 1;
        while ((long)i < pages) {
            Paged.Page page = this.getPage(i);
            page.read();
            if (page.getPageHeader().getStatus() == 1) {
                BTreeNode node = new BTreeNode(page, false);
                node.read();
                node.scanRaw(query, callback);
            }
            ++i;
        }
    }

    private TreeInfo scanTree(boolean removeBranches) throws IOException, TerminatedException, DBException {
        Paged.Page page;
        HashSet<Long> pagePointers = new HashSet<Long>();
        HashSet<Long> nextPages = new HashSet<Long>();
        ArrayList<Long> branchPages = new ArrayList<Long>();
        int pageCount = 0;
        long pages = this.getFileHeader().getTotalCount();
        for (long i = 0L; i < pages; ++i) {
            BTreeNode node = this.cache.get(i);
            if (node != null) {
                page = node.page;
            } else {
                page = this.getPage(i);
                page.read();
            }
            if (page.getPageHeader().getStatus() == 1) {
                ++pageCount;
                if (node == null) {
                    node = new BTreeNode(page, false);
                    node.read();
                }
                this.cache.add(node);
                pagePointers.add(node.page.getPageNum());
                if (node.pageHeader.getNextPage() == -1L) continue;
                nextPages.add(node.pageHeader.getNextPage());
                continue;
            }
            if (page.getPageHeader().getStatus() != 2) continue;
            branchPages.add(page.getPageNum());
        }
        pagePointers.removeAll(nextPages);
        if (pagePointers.size() > 1) {
            LOG.error("Found multiple start pages: [" + pagePointers.stream().map(l -> Long.toString(l)).collect(Collectors.joining(", ")) + "]");
            throw new DBException("More than one start page found for btree: " + FileUtils.fileName(this.getFile()));
        }
        if (removeBranches) {
            Iterator iterator = branchPages.iterator();
            while (iterator.hasNext()) {
                long p = (Long)iterator.next();
                page = this.getPage(p);
                page.read();
                BTreeNode node = new BTreeNode(page, false);
                node.read();
                this.cache.remove(node);
                this.unlinkPages(page);
                page.getPageHeader().setDirty(true);
            }
        }
        return new TreeInfo((Long)pagePointers.iterator().next(), pageCount);
    }

    public void scanSequential(PrintStream out, long pageNum, BTreeCallback callback) throws IOException, TerminatedException {
        while (pageNum != -1L) {
            out.print(pageNum + " ");
            BTreeNode node = this.getBTreeNode(pageNum);
            node.scanRaw(null, callback);
            pageNum = node.pageHeader.getNextPage();
        }
        out.println();
    }

    public void scanSequential(PrintStream out) throws TerminatedException, IOException, DBException {
        TreeInfo info = this.scanTree(false);
        out.println("Sequential scan...");
        this.scanSequential(out, info.firstPage, (value, pointer) -> true);
    }

    public void rebuild() throws TerminatedException, IOException, DBException {
        TreeInfo info = this.scanTree(true);
        if (info.leafPages == 1) {
            BTreeNode root = this.getBTreeNode(info.firstPage);
            this.setRootNode(root);
            this.cache.add(root);
        } else {
            BTreeNode root = this.createBTreeNode(null, (byte)2, null, false);
            this.setRootNode(root);
            BTreeNode node = this.getBTreeNode(info.firstPage);
            root.insertPointer(info.firstPage, 0);
            this.cache.add(root);
            node.setParent(root);
            node.saved = false;
            this.cache.add(node);
            long rightPageNum = node.pageHeader.getNextPage();
            while (rightPageNum != -1L) {
                node = this.getBTreeNode(rightPageNum);
                rightPageNum = node.pageHeader.getNextPage();
                if (node.nKeys < 1) continue;
                Value key = node.keys[0];
                BTreeNode parent = this.findParent(key);
                if (parent == null) {
                    throw new IOException("Parent is null for page " + node.page.getPageNum());
                }
                if (parent.pageHeader.getStatus() != 2) {
                    throw new IOException("Not a branch page: " + parent.page.getPageNum());
                }
                parent.promoteValue(null, key, node);
            }
        }
    }

    private BTreeNode findParent(Value key) throws IOException {
        BTreeNode node;
        BTreeNode last = node = this.getRootNode();
        while (node.pageHeader.getStatus() != 1) {
            last = node;
            try {
                int idx = node.searchKey(key);
                idx = idx < 0 ? -(idx + 1) : idx + 1;
                node = node.getChildNode(idx);
            }
            catch (Exception e) {
                e.printStackTrace();
                throw new IOException("Error while scanning page " + node.page.getPageNum());
            }
        }
        return last;
    }

    private void writeToLog(Loggable loggable, BTreeNode node) {
        if (this.logManager.isPresent()) {
            try {
                this.logManager.get().journal(loggable);
                node.page.getPageHeader().setLsn(loggable.getLsn());
            }
            catch (JournalException e) {
                LOG.warn(e.getMessage(), (Throwable)e);
            }
        }
    }

    protected boolean requiresRedo(Loggable loggable, Paged.Page page) {
        return loggable.getLsn() > page.getPageHeader().getLsn();
    }

    protected void redoCreateBTNode(CreateBTNodeLoggable loggable) throws LogException {
        BTreeNode node = this.cache.get(loggable.pageNum);
        if (node == null) {
            try {
                Paged.Page page = this.getPage(loggable.pageNum);
                page.read();
                if (!(page.getPageHeader().getStatus() != 2 && page.getPageHeader().getStatus() != 1 || page.getPageHeader().getLsn() == -1L || this.requiresRedo(loggable, page))) {
                    node = new BTreeNode(page, false);
                    node.read();
                    return;
                }
                node = new BTreeNode(page, true);
                node.pageHeader.setStatus(loggable.status);
                node.setPointers(new long[0]);
                node.write();
                node.pageHeader.setLsn(loggable.getLsn());
                node.pageHeader.parentPage = loggable.parentNum;
                int increment = node.pageHeader.getStatus() == 2 ? 2 : 1;
                this.cache.add(node, increment);
            }
            catch (IOException e) {
                throw new LogException(e.getMessage(), e);
            }
        }
    }

    protected void redoInsertValue(InsertValueLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (this.requiresRedo(loggable, node.page)) {
            node.insertKey(loggable.key, loggable.idx);
            node.insertPointer(loggable.pointer, loggable.pointerIdx);
            node.adjustDataLen(loggable.idx);
            node.pageHeader.setLsn(loggable.getLsn());
        }
    }

    protected void undoInsertValue(InsertValueLoggable loggable) throws LogException {
        try {
            this.removeValue(null, loggable.key);
        }
        catch (IOException | BTreeException e) {
            LOG.error("Failed to undo: " + loggable.dump(), (Throwable)e);
        }
    }

    protected void redoUpdateValue(UpdateValueLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (node.page.getPageHeader().getLsn() != -1L && this.requiresRedo(loggable, node.page)) {
            if (loggable.idx > node.ptrs.length) {
                LOG.warn(node.page.getPageInfo() + "; loggable.idx = " + loggable.idx + "; node.ptrs.length = " + node.ptrs.length);
                StringWriter writer = new StringWriter();
                try {
                    this.dump(writer);
                }
                catch (Exception e) {
                    LOG.warn((Object)e);
                    e.printStackTrace();
                }
                LOG.warn(writer.toString());
                throw new LogException("Critical error during recovery");
            }
            ((BTreeNode)node).ptrs[loggable.idx] = loggable.pointer;
            node.pageHeader.setLsn(loggable.getLsn());
            node.saved = false;
        }
    }

    protected void undoUpdateValue(UpdateValueLoggable loggable) throws LogException {
        try {
            this.addValue(null, loggable.key, loggable.oldPointer);
        }
        catch (IOException | BTreeException e) {
            LOG.error("Failed to undo: " + loggable.dump(), (Throwable)e);
        }
    }

    protected void redoRemoveValue(RemoveValueLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (node.page.getPageHeader().getLsn() != -1L && this.requiresRedo(loggable, node.page)) {
            node.removeKey(loggable.idx);
            node.removePointer(loggable.idx);
            node.recalculateDataLen();
            node.pageHeader.setLsn(loggable.getLsn());
        }
    }

    protected void undoRemoveValue(RemoveValueLoggable loggable) throws LogException {
        try {
            this.addValue(null, loggable.oldValue, loggable.oldPointer);
        }
        catch (IOException | BTreeException e) {
            LOG.error("Failed to undo: " + loggable.dump(), (Throwable)e);
        }
    }

    protected void redoUpdatePage(UpdatePageLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (this.requiresRedo(loggable, node.page)) {
            node.prefix = loggable.prefix;
            BTreeNode.access$502(node, loggable.values);
            node.nKeys = loggable.values.length;
            node.pageHeader.setValueCount((short)node.nKeys);
            node.setPointers(loggable.pointers);
            node.recalculateDataLen();
            node.pageHeader.setLsn(loggable.getLsn());
        }
    }

    protected void redoSetParent(SetParentLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (this.requiresRedo(loggable, node.page)) {
            node.pageHeader.parentPage = loggable.parentNum;
            node.pageHeader.setLsn(loggable.getLsn());
            node.saved = false;
        }
    }

    protected void redoSetPageLink(SetPageLinkLoggable loggable) throws LogException {
        BTreeNode node = this.getBTreeNode(loggable.pageNum);
        if (this.requiresRedo(loggable, node.page)) {
            node.pageHeader.setNextPage(loggable.nextPage);
            node.pageHeader.setLsn(loggable.getLsn());
            node.saved = false;
        }
    }

    @Override
    public Paged.FileHeader createFileHeader(int pageSize) {
        return new BTreeFileHeader(pageSize);
    }

    @Override
    public Paged.PageHeader createPageHeader() {
        return new BTreePageHeader();
    }

    public BufferStats getIndexBufferStats() {
        return new BufferStats(this.cache.getBuffers(), this.cache.getUsedBuffers(), this.cache.getHits(), this.cache.getFails());
    }

    public void printStatistics() {
        NumberFormat nf = NumberFormat.getPercentInstance();
        StringBuilder buf = new StringBuilder();
        buf.append(FileUtils.fileName(this.getFile())).append(" INDEX ");
        buf.append("Buffers occupation : ");
        if (this.cache.getBuffers() == 0 && this.cache.getUsedBuffers() == 0) {
            buf.append("N/A");
        } else {
            buf.append(nf.format((float)this.cache.getUsedBuffers() / (float)this.cache.getBuffers()));
        }
        buf.append(" (").append(this.cache.getUsedBuffers()).append(" out of ").append(this.cache.getBuffers()).append(")");
        buf.append(" Cache efficiency : ");
        if (this.cache.getHits() == 0 && this.cache.getFails() == 0) {
            buf.append("N/A");
        } else {
            buf.append(nf.format((float)this.cache.getHits() / (float)(this.cache.getFails() + this.cache.getHits())));
        }
        LOGSTATS.info(buf.toString());
    }

    static {
        LogEntryTypes.addEntryType((byte)32, InsertValueLoggable::new);
        LogEntryTypes.addEntryType((byte)36, UpdateValueLoggable::new);
        LogEntryTypes.addEntryType((byte)37, RemoveValueLoggable::new);
        LogEntryTypes.addEntryType((byte)33, CreateBTNodeLoggable::new);
        LogEntryTypes.addEntryType((byte)34, UpdatePageLoggable::new);
        LogEntryTypes.addEntryType((byte)35, SetParentLoggable::new);
        LogEntryTypes.addEntryType((byte)38, SetPageLinkLoggable::new);
    }

    protected static class BTreePageHeader
    extends Paged.PageHeader {
        private short valueCount = 0;
        private long parentPage = -1L;

        public BTreePageHeader() {
        }

        public BTreePageHeader(byte[] data, int offset) throws IOException {
            super(data, offset);
        }

        @Override
        public int read(byte[] data, int offset) throws IOException {
            offset = super.read(data, offset);
            this.parentPage = ByteConversion.byteToLong(data, offset);
            this.valueCount = ByteConversion.byteToShort(data, offset += 8);
            return offset + 2;
        }

        @Override
        public int write(byte[] data, int offset) throws IOException {
            offset = super.write(data, offset);
            ByteConversion.longToByte(this.parentPage, data, offset);
            ByteConversion.shortToByte(this.valueCount, data, offset += 8);
            return offset + 2;
        }

        public final void setValueCount(short valueCount) {
            this.valueCount = valueCount;
            this.setDirty(true);
        }

        public final short getValueCount() {
            return this.valueCount;
        }

        public final short getPointerCount() {
            if (this.getStatus() == 2) {
                return (short)(this.valueCount + 1);
            }
            return this.valueCount;
        }
    }

    protected class BTreeFileHeader
    extends Paged.FileHeader {
        private long rootPage;
        private short fixedLen;

        public BTreeFileHeader(long pageCount, int pageSize) {
            super(BTree.this, pageCount, pageSize);
            this.rootPage = 0L;
            this.fixedLen = (short)-1;
        }

        public BTreeFileHeader(int pageSize) {
            super(BTree.this, 1024L, pageSize);
            this.rootPage = 0L;
            this.fixedLen = (short)-1;
        }

        @Override
        public int read(byte[] buf) throws IOException {
            int offset = super.read(buf);
            this.rootPage = ByteConversion.byteToLong(buf, offset);
            this.fixedLen = ByteConversion.byteToShort(buf, offset += 8);
            return offset += 2;
        }

        @Override
        public int write(byte[] buf) throws IOException {
            int offset = super.write(buf);
            ByteConversion.longToByte(this.rootPage, buf, offset);
            ByteConversion.shortToByte(this.fixedLen, buf, offset += 8);
            return offset += 2;
        }

        public final void setRootPage(long rootPage) {
            this.rootPage = rootPage;
            this.setDirty(true);
        }

        public final long getRootPage() {
            return this.rootPage;
        }

        public short getFixedKeyLen() {
            return this.fixedLen;
        }

        public void setFixedKeyLen(short keyLen) {
            this.fixedLen = keyLen;
        }

        @Override
        public int getMaxKeySize() {
            return this.getWorkSize() / 2 - 32;
        }
    }

    protected final class BTreeNode
    implements BTreeCacheable {
        private static final int DEFAULT_INITIAL_ENTRIES = 32;
        private final Paged.Page page;
        private final BTreePageHeader pageHeader;
        private Value[] keys;
        private Value prefix = Value.EMPTY_VALUE;
        private int nKeys = 0;
        private long[] ptrs;
        private int nPtrs = 0;
        private int refCount = 0;
        private int timestamp = 0;
        private boolean saved = true;
        private int currentDataLen = -1;
        private boolean allowUnload = true;

        public BTreeNode(Paged.Page page, boolean newPage) {
            this.page = page;
            this.pageHeader = (BTreePageHeader)page.getPageHeader();
            if (newPage) {
                this.keys = new Value[32];
                this.ptrs = new long[33];
                this.pageHeader.setValueCount((short)0);
                this.saved = false;
            }
        }

        public void setParent(BTreeNode parent) {
            if (parent != null) {
                this.pageHeader.parentPage = parent.page.getPageNum();
            } else {
                this.pageHeader.parentPage = -1L;
            }
            this.saved = false;
        }

        public BTreeNode getParent() {
            if (this.pageHeader.parentPage != -1L) {
                return BTree.this.getBTreeNode(this.pageHeader.parentPage);
            }
            return null;
        }

        @Override
        public int getReferenceCount() {
            return this.refCount;
        }

        @Override
        public int incReferenceCount() {
            if (this.refCount < 10000) {
                ++this.refCount;
            }
            return this.refCount;
        }

        @Override
        public void setReferenceCount(int count) {
            this.refCount = count;
        }

        @Override
        public void setTimestamp(int timestamp) {
            this.timestamp = timestamp;
        }

        @Override
        public boolean allowUnload() {
            return this.allowUnload;
        }

        @Override
        public int getTimestamp() {
            return this.timestamp;
        }

        @Override
        public boolean isInnerPage() {
            return this.pageHeader.getStatus() == 2;
        }

        @Override
        public boolean sync(boolean syncJournal) {
            if (this.isDirty()) {
                try {
                    this.write();
                    if (BTree.this.isRecoveryEnabled() && syncJournal) {
                        BTree.this.logManager.get().flush(true, false);
                    }
                    return true;
                }
                catch (IOException e) {
                    Paged.LOG.error("IO error while writing page: " + this.page.getPageNum(), (Throwable)e);
                }
            }
            return false;
        }

        @Override
        public long getKey() {
            return this.page.getPageNum();
        }

        @Override
        public int decReferenceCount() {
            return this.refCount > 0 ? (this.refCount = this.refCount - 1) : 0;
        }

        @Override
        public boolean isDirty() {
            return !this.saved;
        }

        private void setValues(Value[] vals) {
            this.keys = vals;
            this.nKeys = vals.length;
            this.pageHeader.setValueCount((short)this.nKeys);
            this.saved = false;
        }

        private void setPointers(long[] pointers) {
            this.ptrs = pointers;
            this.nPtrs = pointers.length;
            this.saved = false;
        }

        private int getDataLen() {
            return this.currentDataLen < 0 ? this.recalculateDataLen() : this.currentDataLen;
        }

        private int recalculateDataLen() {
            int n = this.currentDataLen = this.ptrs == null ? 0 : this.nPtrs * 8;
            if (BTree.this.fileHeader.getFixedKeyLen() < 0) {
                this.currentDataLen += 2 * this.nKeys;
            }
            if (this.pageHeader.getStatus() == 2) {
                this.currentDataLen += this.prefix.getLength() + 2;
            }
            if (this.pageHeader.getStatus() == 1) {
                this.currentDataLen += this.nKeys - 1;
            }
            for (int i = 0; i < this.nKeys; ++i) {
                if (this.pageHeader.getStatus() == 1 && i > 0) {
                    int prefix = this.keys[i].commonPrefix(this.keys[i - 1]);
                    if (prefix < 0 || prefix > 127) {
                        prefix = 0;
                    }
                    this.currentDataLen += this.keys[i].getLength() - prefix;
                    continue;
                }
                this.currentDataLen += this.keys[i].getLength();
            }
            return this.currentDataLen;
        }

        private void adjustDataLen(int idx) {
            if (this.currentDataLen < 0) {
                this.recalculateDataLen();
                return;
            }
            if (this.pageHeader.getStatus() == 1 && idx > 0) {
                int prefix;
                if (idx + 1 < this.nKeys) {
                    prefix = this.calculatePrefixLen(idx + 1, idx - 1);
                    this.currentDataLen -= this.keys[idx + 1].getLength() - prefix;
                    prefix = this.calculatePrefixLen(idx + 1, idx);
                    this.currentDataLen += this.keys[idx + 1].getLength() - prefix;
                }
                prefix = this.calculatePrefixLen(idx, idx - 1);
                this.currentDataLen += this.keys[idx].getLength() - prefix;
                ++this.currentDataLen;
            } else {
                this.currentDataLen += this.keys[idx].getLength();
                if (this.pageHeader.getStatus() == 1) {
                    ++this.currentDataLen;
                }
            }
            this.currentDataLen += 8;
            if (BTree.this.fileHeader.getFixedKeyLen() < 0) {
                this.currentDataLen += 2;
            }
        }

        private int calculatePrefixLen(int idx0, int idx1) {
            int prefix = this.keys[idx0].commonPrefix(this.keys[idx1]);
            if (prefix < 0 || prefix > 127) {
                prefix = 0;
            }
            return prefix;
        }

        private int getPivot(int preferred) {
            if (this.nKeys == 2) {
                return 1;
            }
            int totalLen = this.getKeyDataLen();
            int currentLen = 0;
            int pivot = this.nKeys - 1;
            for (int i = 0; i < this.nKeys - 1; ++i) {
                if (this.pageHeader.getStatus() == 1 && i > 0) {
                    int prefix = this.keys[i].commonPrefix(this.keys[i - 1]);
                    if (prefix < 0 || prefix > 127) {
                        prefix = 0;
                    }
                    currentLen += this.keys[i].getLength() - prefix;
                } else {
                    currentLen += this.keys[i].getLength();
                }
                if (currentLen <= totalLen / 2 && i + 1 != preferred) continue;
                pivot = currentLen > BTree.this.fileHeader.getWorkSize() ? i : i + 1;
                break;
            }
            return pivot;
        }

        private int getKeyDataLen() {
            int totalLen = 0;
            for (int i = 0; i < this.nKeys; ++i) {
                if (this.pageHeader.getStatus() == 1 && i > 0) {
                    int prefix = this.keys[i].commonPrefix(this.keys[i - 1]);
                    if (prefix < 0 || prefix > 127) {
                        prefix = 0;
                    }
                    totalLen += this.keys[i].getLength() - prefix;
                    continue;
                }
                totalLen += this.keys[i].getLength();
            }
            return totalLen;
        }

        private boolean mustSplit() {
            if (this.pageHeader.getValueCount() != this.nKeys) {
                throw new RuntimeException("Wrong value count");
            }
            return this.getDataLen() > BTree.this.fileHeader.getWorkSize();
        }

        private void read() throws IOException {
            int i;
            short keyLen;
            byte[] data = this.page.read();
            short valSize = keyLen = BTree.this.fileHeader.getFixedKeyLen();
            int p = 0;
            if (this.pageHeader.getStatus() == 2) {
                short prefixSize = ByteConversion.byteToShort(data, p);
                p += 2;
                if (prefixSize == 0) {
                    this.prefix = Value.EMPTY_VALUE;
                } else {
                    this.prefix = new Value(data, p, prefixSize);
                    p += prefixSize;
                }
            }
            this.nKeys = this.pageHeader.getValueCount();
            this.keys = new Value[this.nKeys * 3 / 2 + 1];
            for (i = 0; i < this.nKeys; ++i) {
                if (keyLen < 0) {
                    valSize = ByteConversion.byteToShort(data, p);
                    p += 2;
                }
                if (this.pageHeader.getStatus() == 1 && i > 0) {
                    int prefixLen = data[p++] & 0xFF;
                    try {
                        byte[] t = new byte[valSize];
                        if (prefixLen > 0) {
                            System.arraycopy(this.keys[i - 1].data(), this.keys[i - 1].start(), t, 0, prefixLen);
                        }
                        System.arraycopy(data, p, t, prefixLen, valSize - prefixLen);
                        p += valSize - prefixLen;
                        this.keys[i] = new Value(t);
                        continue;
                    }
                    catch (Exception e) {
                        e.printStackTrace();
                        Paged.LOG.error("prefixLen = " + prefixLen + "; i = " + i + "; nKeys = " + this.nKeys);
                        throw new IOException(e.getMessage());
                    }
                }
                this.keys[i] = new Value(data, p, valSize);
                p += valSize;
            }
            this.nPtrs = this.pageHeader.getPointerCount();
            this.ptrs = new long[this.nPtrs * 3 / 2 + 1];
            for (i = 0; i < this.nPtrs; ++i) {
                this.ptrs[i] = ByteConversion.byteToLong(data, p);
                p += 8;
            }
        }

        private void write() throws IOException {
            int i;
            if (this.nKeys != this.pageHeader.getValueCount()) {
                throw new RuntimeException("nkeys: " + this.nKeys + " valueCount: " + this.pageHeader.getValueCount());
            }
            byte[] temp = new byte[BTree.this.fileHeader.getWorkSize()];
            int p = 0;
            if (this.pageHeader.getStatus() == 2) {
                ByteConversion.shortToByte((short)this.prefix.getLength(), temp, p);
                p += 2;
                if (this.prefix.getLength() > 0) {
                    System.arraycopy(this.prefix.data(), this.prefix.start(), temp, p, this.prefix.getLength());
                    p += this.prefix.getLength();
                }
            }
            short keyLen = BTree.this.fileHeader.getFixedKeyLen();
            for (i = 0; i < this.nKeys; ++i) {
                if (keyLen < 0) {
                    ByteConversion.shortToByte((short)this.keys[i].getLength(), temp, p);
                    p += 2;
                }
                if (this.pageHeader.getStatus() == 1 && i > 0) {
                    int prefixLen = this.keys[i].commonPrefix(this.keys[i - 1]);
                    if (prefixLen < 0 || prefixLen > 127) {
                        prefixLen = 0;
                    }
                    temp[p++] = (byte)prefixLen;
                    System.arraycopy(this.keys[i].data(), this.keys[i].start() + prefixLen, temp, p, this.keys[i].getLength() - prefixLen);
                    p += this.keys[i].getLength() - prefixLen;
                    continue;
                }
                byte[] data = this.keys[i].getData();
                if (p + data.length > temp.length) {
                    throw new IOException("calculated: " + this.getDataLen() + "; required: " + (p + data.length));
                }
                System.arraycopy(data, 0, temp, p, data.length);
                p += data.length;
            }
            for (i = 0; i < this.nPtrs; ++i) {
                ByteConversion.longToByte(this.ptrs[i], temp, p);
                p += 8;
            }
            BTree.this.writeValue(this.page, new Value(temp));
            this.saved = true;
        }

        private BTreeNode getChildNode(int idx) throws IOException {
            if (this.pageHeader.getStatus() == 2 && idx >= 0 && idx < this.nPtrs) {
                return BTree.this.getBTreeNode(this.ptrs[idx]);
            }
            return null;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long removeValue(Txn transaction, Value key) throws IOException, BTreeException {
            int idx = this.searchKey(key);
            switch (this.pageHeader.getStatus()) {
                case 2: {
                    idx = idx < 0 ? -(idx + 1) : idx + 1;
                    return this.getChildNode(idx).removeValue(transaction, key);
                }
                case 1: {
                    if (idx < 0) {
                        return -1L;
                    }
                    try {
                        this.allowUnload = false;
                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), idx, this.keys[idx], this.ptrs[idx]);
                            BTree.this.writeToLog(log, this);
                        }
                        long oldPtr = this.ptrs[idx];
                        this.removeKey(idx);
                        this.removePointer(idx);
                        this.recalculateDataLen();
                        long l = oldPtr;
                        return l;
                    }
                    finally {
                        this.allowUnload = true;
                    }
                }
            }
            throw new BTreeException("Invalid Page Type In removeValue");
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long addValue(Txn transaction, Value value, long pointer) throws IOException, BTreeException {
            if (value == null) {
                return -1L;
            }
            int idx = this.searchKey(value);
            switch (this.pageHeader.getStatus()) {
                case 2: {
                    idx = idx < 0 ? -(idx + 1) : idx + 1;
                    return this.getChildNode(idx).addValue(transaction, value, pointer);
                }
                case 1: {
                    try {
                        this.allowUnload = false;
                        if (idx >= 0) {
                            long oldPtr = this.ptrs[idx];
                            if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                UpdateValueLoggable loggable = new UpdateValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), idx, value, pointer, oldPtr);
                                BTree.this.writeToLog(loggable, this);
                            }
                            this.ptrs[idx] = pointer;
                            this.saved = false;
                            long l = oldPtr;
                            return l;
                        }
                        idx = -(idx + 1);
                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                            InsertValueLoggable loggable = new InsertValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), idx, value, idx, pointer);
                            BTree.this.writeToLog(loggable, this);
                        }
                        this.insertKey(value, idx);
                        this.insertPointer(pointer, idx);
                        this.adjustDataLen(idx);
                        if (this.mustSplit()) {
                            if (BTree.this.splitFactor > 0.0 && (double)idx > (double)this.nKeys * BTree.this.splitFactor && value.getLength() < BTree.this.fileHeader.getWorkSize() / 4) {
                                this.split(transaction, idx == 0 ? 1 : idx);
                            } else {
                                this.split(transaction);
                            }
                        }
                    }
                    finally {
                        this.allowUnload = true;
                    }
                    return -1L;
                }
            }
            throw new BTreeException("Invalid Page Type In addValue: " + this.pageHeader.getStatus() + "; " + this.page.getPageInfo());
        }

        private void promoteValue(Txn transaction, Value value, BTreeNode rightNode) throws IOException, BTreeException {
            boolean split;
            int idx = this.searchKey(value);
            idx = idx < 0 ? -(idx + 1) : idx + 1;
            this.insertKey(value, idx);
            this.insertPointer(rightNode.page.getPageNum(), idx + 1);
            rightNode.setParent(this);
            rightNode.saved = false;
            BTree.this.cache.add(rightNode);
            this.saved = false;
            BTree.this.cache.add(this);
            boolean bl = split = this.recalculateDataLen() > BTree.this.fileHeader.getWorkSize();
            if (split) {
                this.split(transaction);
            }
        }

        private void split(Txn transaction) throws IOException, BTreeException {
            this.split(transaction, -1);
        }

        private void split(Txn transaction, int pivot) throws IOException, BTreeException {
            Value separator;
            long[] rightPtrs;
            Value[] rightVals;
            long[] leftPtrs;
            Value[] leftVals;
            short vc = this.pageHeader.getValueCount();
            pivot = this.getPivot(pivot);
            switch (this.pageHeader.getStatus()) {
                case 2: {
                    leftVals = new Value[pivot];
                    leftPtrs = new long[leftVals.length + 1];
                    rightVals = new Value[vc - (pivot + 1)];
                    rightPtrs = new long[rightVals.length + 1];
                    System.arraycopy(this.keys, 0, leftVals, 0, leftVals.length);
                    System.arraycopy(this.ptrs, 0, leftPtrs, 0, leftPtrs.length);
                    System.arraycopy(this.keys, leftVals.length + 1, rightVals, 0, rightVals.length);
                    System.arraycopy(this.ptrs, leftPtrs.length, rightPtrs, 0, rightPtrs.length);
                    separator = this.keys[leftVals.length];
                    if (this.prefix == null || this.prefix.getLength() <= 0) break;
                    byte[] t = new byte[this.prefix.getLength() + separator.getLength()];
                    System.arraycopy(this.prefix.data(), this.prefix.start(), t, 0, this.prefix.getLength());
                    System.arraycopy(separator.data(), separator.start(), t, this.prefix.getLength(), separator.getLength());
                    separator = new Value(t);
                    break;
                }
                case 1: {
                    leftVals = new Value[pivot];
                    leftPtrs = new long[leftVals.length];
                    rightVals = new Value[vc - pivot];
                    rightPtrs = new long[rightVals.length];
                    System.arraycopy(this.keys, 0, leftVals, 0, leftVals.length);
                    System.arraycopy(this.ptrs, 0, leftPtrs, 0, leftPtrs.length);
                    System.arraycopy(this.keys, leftVals.length, rightVals, 0, rightVals.length);
                    System.arraycopy(this.ptrs, leftPtrs.length, rightPtrs, 0, rightPtrs.length);
                    separator = this.keys[leftVals.length];
                    break;
                }
                default: {
                    throw new BTreeException("Invalid Page Type In split");
                }
            }
            if (transaction != null && BTree.this.isRecoveryEnabled() && this.pageHeader.getStatus() == 1) {
                UpdatePageLoggable log = new UpdatePageLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), this.prefix, leftVals, leftVals.length, leftPtrs, leftPtrs.length);
                BTree.this.writeToLog(log, this);
            }
            this.setValues(leftVals);
            this.setPointers(leftPtrs);
            this.recalculateDataLen();
            BTreeNode parent = this.getParent();
            if (parent == null) {
                BTAbstractLoggable log;
                parent = BTree.this.createBTreeNode(transaction, (byte)2, null, false);
                if (transaction != null && BTree.this.isRecoveryEnabled() && this.pageHeader.getStatus() == 1) {
                    SetParentLoggable log2 = new SetParentLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), parent.page.getPageNum());
                    BTree.this.writeToLog(log2, this);
                }
                this.setParent(parent);
                BTreeNode rNode = BTree.this.createBTreeNode(transaction, this.pageHeader.getStatus(), parent, false);
                rNode.setValues(rightVals);
                rNode.setPointers(rightPtrs);
                rNode.setAsParent();
                if (this.pageHeader.getStatus() == 2) {
                    rNode.prefix = this.prefix;
                    rNode.growPrefix();
                } else {
                    if (transaction != null && BTree.this.isRecoveryEnabled()) {
                        log = new SetPageLinkLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), rNode.page.getPageNum());
                        BTree.this.writeToLog(log, this);
                    }
                    this.pageHeader.setNextPage(rNode.page.getPageNum());
                }
                if (transaction != null && BTree.this.isRecoveryEnabled() && this.pageHeader.getStatus() == 1) {
                    log = new UpdatePageLoggable(transaction, BTree.this.fileId, rNode.page.getPageNum(), rNode.prefix, rNode.keys, rNode.nKeys, rightPtrs, rightPtrs.length);
                    BTree.this.writeToLog(log, rNode);
                }
                rNode.recalculateDataLen();
                parent.prefix = separator;
                parent.setValues(new Value[]{Value.EMPTY_VALUE});
                parent.setPointers(new long[]{this.page.getPageNum(), rNode.page.getPageNum()});
                parent.recalculateDataLen();
                BTree.this.cache.add(parent);
                BTree.this.setRootNode(parent);
                if (rNode.mustSplit()) {
                    Paged.LOG.debug(FileUtils.fileName(BTree.this.getFile()) + " right node requires second split: " + rNode.getDataLen());
                    rNode.split(transaction);
                }
                BTree.this.cache.add(rNode);
            } else {
                BTAbstractLoggable log;
                BTreeNode rNode = BTree.this.createBTreeNode(transaction, this.pageHeader.getStatus(), parent, false);
                rNode.setValues(rightVals);
                rNode.setPointers(rightPtrs);
                rNode.setAsParent();
                if (this.pageHeader.getStatus() == 2) {
                    rNode.prefix = this.prefix;
                    rNode.growPrefix();
                } else {
                    if (transaction != null && BTree.this.isRecoveryEnabled()) {
                        log = new SetPageLinkLoggable(transaction, BTree.this.fileId, rNode.page.getPageNum(), this.pageHeader.getNextPage());
                        BTree.this.writeToLog(log, this);
                        log = new SetPageLinkLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), rNode.page.getPageNum());
                        BTree.this.writeToLog(log, this);
                    }
                    rNode.pageHeader.setNextPage(this.pageHeader.getNextPage());
                    this.pageHeader.setNextPage(rNode.page.getPageNum());
                }
                if (transaction != null && BTree.this.isRecoveryEnabled() && this.pageHeader.getStatus() == 1) {
                    log = new UpdatePageLoggable(transaction, BTree.this.fileId, rNode.page.getPageNum(), rNode.prefix, rNode.keys, rNode.nKeys, rightPtrs, rightPtrs.length);
                    BTree.this.writeToLog(log, rNode);
                }
                rNode.recalculateDataLen();
                if (rNode.mustSplit()) {
                    Paged.LOG.debug(FileUtils.fileName(BTree.this.getFile()) + " right node requires second split: " + rNode.getDataLen());
                    rNode.split(transaction);
                }
                BTree.this.cache.add(rNode);
                parent.promoteValue(transaction, separator, rNode);
            }
            BTree.this.cache.add(this);
            if (this.mustSplit()) {
                Paged.LOG.debug(FileUtils.fileName(BTree.this.getFile()) + "left node requires second split: " + this.getDataLen());
                this.split(transaction);
            }
        }

        private void setAsParent() {
            if (this.pageHeader.getStatus() == 2) {
                for (int i = 0; i < this.nPtrs; ++i) {
                    BTreeNode node = BTree.this.getBTreeNode(this.ptrs[i]);
                    node.setParent(this);
                    BTree.this.cache.add(node);
                }
            }
        }

        private long findValue(Value value) throws IOException, BTreeException {
            int idx = this.searchKey(value);
            switch (this.pageHeader.getStatus()) {
                case 2: {
                    idx = idx < 0 ? -(idx + 1) : idx + 1;
                    BTreeNode child = this.getChildNode(idx);
                    if (child == null) {
                        throw new BTreeException("Unexpected " + idx + ", " + this.page.getPageNum() + ": value '" + value.toString() + "' doesn't exist");
                    }
                    return child.findValue(value);
                }
                case 1: {
                    if (idx < 0) {
                        return -1L;
                    }
                    return this.ptrs[idx];
                }
            }
            throw new BTreeException("Invalid Page Type In findValue");
        }

        public String toString() {
            StringWriter writer = new StringWriter();
            try {
                this.dump(writer);
            }
            catch (Exception e) {
                Paged.LOG.error((Object)e);
            }
            return writer.toString();
        }

        private void treeStatistics(TreeMetrics metrics) throws IOException {
            metrics.addPage(this.pageHeader.getStatus());
            if (this.pageHeader.getStatus() == 2) {
                for (int i = 0; i < this.nPtrs; ++i) {
                    BTreeNode child = this.getChildNode(i);
                    child.treeStatistics(metrics);
                }
            }
        }

        private void dump(Writer writer) throws IOException, BTreeException {
            int i;
            if (this.page.getPageNum() == BTree.this.fileHeader.getRootPage()) {
                writer.write("ROOT: ");
            }
            writer.write(this.page.getPageNum() + ": ");
            writer.write(this.pageHeader.getStatus() == 2 ? "BRANCH: " : "LEAF: ");
            writer.write(this.saved ? "SAVED: " : "DIRTY: ");
            if (this.pageHeader.getStatus() == 2) {
                writer.write("PREFIX: ");
                BTree.this.dumpValue(writer, this.prefix, this.pageHeader.getStatus());
                writer.write(": ");
            }
            writer.write("NEXT: ");
            writer.write(Long.toString(this.pageHeader.getNextPage()));
            writer.write(": ");
            for (i = 0; i < this.nKeys; ++i) {
                if (i > 0) {
                    writer.write(32);
                }
                BTree.this.dumpValue(writer, this.keys[i], this.pageHeader.getStatus());
            }
            writer.write(10);
            if (this.pageHeader.getStatus() == 2) {
                writer.write("-----------------------------------------------------------------------------------------\n");
                writer.write(this.page.getPageNum() + " POINTERS: ");
                for (i = 0; i < this.nPtrs; ++i) {
                    writer.write(this.ptrs[i] + " ");
                }
                writer.write(10);
            }
            writer.write("-----------------------------------------------------------------------------------------\n");
            if (this.pageHeader.getStatus() == 2) {
                for (i = 0; i < this.nPtrs; ++i) {
                    BTreeNode child = this.getChildNode(i);
                    child.dump(writer);
                }
            }
        }

        private void query(IndexQuery query, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
            block53: {
                block54: {
                    if (query == null || query.getOperator() == 0 || query.getOperator() == 8) break block54;
                    Value[] qvals = query.getValues();
                    int leftIdx = this.searchKey(qvals[0]);
                    int rightIdx = qvals.length > 1 ? this.searchKey(qvals[qvals.length - 1]) : leftIdx;
                    boolean pos = query.getOperator() >= 0;
                    switch (this.pageHeader.getStatus()) {
                        case 2: {
                            leftIdx = leftIdx < 0 ? -(leftIdx + 1) : leftIdx + 1;
                            rightIdx = rightIdx < 0 ? -(rightIdx + 1) : rightIdx + 1;
                            block4 : switch (query.getOperator()) {
                                case -6: 
                                case -5: 
                                case -4: 
                                case 4: 
                                case 5: 
                                case 6: 
                                case 7: 
                                case 10: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((i >= leftIdx && i <= rightIdx) != pos) continue;
                                        this.getChildNode(i).query(query, callback);
                                        if (query.getOperator() != 7) {
                                            if (query.getOperator() == 10) break block4;
                                            continue;
                                        }
                                        break block53;
                                    }
                                    break block53;
                                }
                                case -1: {
                                    this.getChildNode(0).query(query, callback);
                                    break;
                                }
                                case 1: {
                                    this.getChildNode(leftIdx).query(query, callback);
                                    break;
                                }
                                case 3: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i > leftIdx) && (pos || i < leftIdx)) continue;
                                        this.getChildNode(i).query(query, callback);
                                    }
                                    break block53;
                                }
                                case -3: 
                                case 2: {
                                    this.getChildNode(leftIdx).query(query, callback);
                                    break;
                                }
                                case -2: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i < leftIdx) && (pos || i > leftIdx)) continue;
                                        this.getChildNode(i).query(query, callback);
                                    }
                                    break block53;
                                }
                                default: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        this.getChildNode(i).query(query, callback);
                                    }
                                    break block53;
                                }
                            }
                            break block53;
                        }
                        case 1: {
                            block12 : switch (query.getOperator()) {
                                case 1: {
                                    if (leftIdx >= 0) {
                                        callback.indexInfo(this.keys[leftIdx], this.ptrs[leftIdx]);
                                        break;
                                    }
                                    break block53;
                                }
                                case -1: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (i == leftIdx) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    this.scanNextPage(query, null, callback);
                                    break;
                                }
                                case -5: 
                                case -4: 
                                case 4: 
                                case 5: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i < leftIdx || i > rightIdx) && (pos || i > leftIdx && i < rightIdx) || !query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block53;
                                }
                                case 7: 
                                case 10: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = leftIdx; i < rightIdx && i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    if (rightIdx >= this.nPtrs) {
                                        this.scanNextPage(query, null, callback);
                                        break;
                                    }
                                    break block53;
                                }
                                case -6: 
                                case 6: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (pos && (i < leftIdx || i > rightIdx) || !query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block53;
                                }
                                case 3: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i > leftIdx) && (pos || i < leftIdx) || !query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block53;
                                }
                                case -3: 
                                case 2: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = leftIdx; i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    this.scanNextPage(query, null, callback);
                                    break;
                                }
                                case -2: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i < leftIdx) && (pos || i > leftIdx)) continue;
                                        if (query.testValue(this.keys[i])) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                            continue;
                                        }
                                        if (query.getOperator() == 7) break block12;
                                    }
                                    break block53;
                                }
                                default: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block53;
                                }
                            }
                            break block53;
                        }
                        default: {
                            throw new BTreeException("Invalid Page Type In query");
                        }
                    }
                }
                switch (this.pageHeader.getStatus()) {
                    case 2: {
                        for (int i = 0; i < this.nPtrs; ++i) {
                            this.getChildNode(i).query(query, callback);
                        }
                        break;
                    }
                    case 1: {
                        for (int i = 0; i < this.nKeys; ++i) {
                            if (query != null && query.getOperator() == 8 && !query.testValue(this.keys[i])) continue;
                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                        }
                        break;
                    }
                    default: {
                        throw new BTreeException("Invalid Page Type In query");
                    }
                }
            }
        }

        private void query(IndexQuery query, Value keyPrefix, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
            block36: {
                block35: {
                    if (query == null || query.getOperator() == 0 || query.getOperator() == 8) break block35;
                    Value[] qvals = query.getValues();
                    int leftIdx = this.searchKey(qvals[0]);
                    int pfxIdx = this.searchKey(keyPrefix);
                    switch (this.pageHeader.getStatus()) {
                        case 2: {
                            leftIdx = leftIdx < 0 ? -(leftIdx + 1) : leftIdx + 1;
                            pfxIdx = pfxIdx < 0 ? -(pfxIdx + 1) : pfxIdx + 1;
                            switch (query.getOperator()) {
                                case 1: {
                                    this.getChildNode(leftIdx).query(query, keyPrefix, callback);
                                    break;
                                }
                                case -1: {
                                    this.getChildNode(pfxIdx).query(query, keyPrefix, callback);
                                    break;
                                }
                                case 3: {
                                    for (int i = pfxIdx; i <= leftIdx && i < this.nPtrs; ++i) {
                                        this.getChildNode(i).query(query, keyPrefix, callback);
                                    }
                                    break block36;
                                }
                                case -2: {
                                    for (int i = pfxIdx; i <= leftIdx && i < this.nPtrs; ++i) {
                                        this.getChildNode(i).query(query, keyPrefix, callback);
                                    }
                                    break block36;
                                }
                                case -3: 
                                case 2: {
                                    this.getChildNode(leftIdx).query(query, keyPrefix, callback);
                                }
                            }
                            break block36;
                        }
                        case 1: {
                            pfxIdx = pfxIdx < 0 ? -(pfxIdx + 1) : pfxIdx + 1;
                            switch (query.getOperator()) {
                                case 1: {
                                    if (leftIdx >= 0) {
                                        callback.indexInfo(this.keys[leftIdx], this.ptrs[leftIdx]);
                                        break;
                                    }
                                    break block36;
                                }
                                case -1: {
                                    for (int i = pfxIdx; i < this.nPtrs && this.keys[i].comparePrefix(keyPrefix) <= 0; ++i) {
                                        if (i == leftIdx) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    this.scanNextPage(query, keyPrefix, callback);
                                    break;
                                }
                                case 3: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = pfxIdx; i < leftIdx; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block36;
                                }
                                case -2: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = pfxIdx; i <= leftIdx && i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    break block36;
                                }
                                case -3: 
                                case 2: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = leftIdx; i < this.nPtrs; ++i) {
                                        if (this.keys[i].comparePrefix(keyPrefix) > 0) {
                                            return;
                                        }
                                        if (!query.testValue(this.keys[i])) continue;
                                        callback.indexInfo(this.keys[i], this.ptrs[i]);
                                    }
                                    this.scanNextPage(query, keyPrefix, callback);
                                }
                            }
                            break block36;
                        }
                        default: {
                            throw new BTreeException("Invalid Page Type In query");
                        }
                    }
                }
                switch (this.pageHeader.getStatus()) {
                    case 2: {
                        for (int i = 0; i < this.nPtrs; ++i) {
                            this.getChildNode(i).query(query, callback);
                        }
                        break;
                    }
                    case 1: {
                        for (int i = 0; i < this.nKeys; ++i) {
                            if (query.getOperator() == 8 && !query.testValue(this.keys[i])) continue;
                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                        }
                        break;
                    }
                    default: {
                        throw new BTreeException("Invalid Page Type In query");
                    }
                }
            }
        }

        protected void scanRaw(IndexQuery query, BTreeCallback callback) throws TerminatedException {
            for (int i = 0; i < this.nKeys; ++i) {
                if (query != null && !query.testValue(this.keys[i])) continue;
                callback.indexInfo(this.keys[i], this.ptrs[i]);
            }
        }

        protected void scanNextPage(IndexQuery query, Value keyPrefix, BTreeCallback callback) throws TerminatedException {
            long next = this.pageHeader.getNextPage();
            if (next != -1L) {
                BTreeNode nextPage = BTree.this.getBTreeNode(next);
                BTree.this.scanSequential(nextPage, query, keyPrefix, callback);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private void remove(Txn transaction, IndexQuery query, BTreeCallback callback) throws IOException, BTreeException, TerminatedException {
            if (query != null && query.getOperator() != 0 && query.getOperator() != 8) {
                Value[] qvals = query.getValues();
                int leftIdx = this.searchKey(qvals[0]);
                int rightIdx = qvals.length > 1 ? this.searchKey(qvals[qvals.length - 1]) : leftIdx;
                boolean pos = query.getOperator() >= 0;
                switch (this.pageHeader.getStatus()) {
                    case 2: {
                        leftIdx = leftIdx < 0 ? -(leftIdx + 1) : leftIdx + 1;
                        rightIdx = rightIdx < 0 ? -(rightIdx + 1) : rightIdx + 1;
                        switch (query.getOperator()) {
                            case -6: 
                            case -5: 
                            case -4: 
                            case 4: 
                            case 5: 
                            case 6: 
                            case 7: 
                            case 10: {
                                for (int i = 0; i < this.nPtrs; ++i) {
                                    if ((i >= leftIdx && i <= rightIdx) != pos) continue;
                                    this.getChildNode(i).remove(transaction, query, callback);
                                    if (query.getOperator() == 7) return;
                                }
                                return;
                            }
                            case -1: 
                            case 1: {
                                int i;
                                for (i = 0; i < this.nPtrs; ++i) {
                                    if (pos && i != leftIdx) continue;
                                    this.getChildNode(i).remove(transaction, query, callback);
                                }
                            }
                            case -3: 
                            case 3: {
                                int i;
                                for (i = 0; i < this.nPtrs; ++i) {
                                    if ((!pos || i > leftIdx) && (pos || i < leftIdx)) continue;
                                    this.getChildNode(i).remove(transaction, query, callback);
                                }
                                return;
                            }
                            case -2: 
                            case 2: {
                                for (int i = 0; i < this.nPtrs; ++i) {
                                    if ((!pos || i < leftIdx) && (pos || i > leftIdx)) continue;
                                    this.getChildNode(i).remove(transaction, query, callback);
                                }
                                return;
                            }
                            default: {
                                for (int i = 0; i < this.nPtrs; ++i) {
                                    this.getChildNode(i).remove(transaction, query, callback);
                                }
                                return;
                            }
                        }
                    }
                    case 1: {
                        try {
                            this.allowUnload = false;
                            switch (query.getOperator()) {
                                case 1: {
                                    if (leftIdx < 0) return;
                                    if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                        RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), leftIdx, this.keys[leftIdx], this.ptrs[leftIdx]);
                                        BTree.this.writeToLog(log, this);
                                    }
                                    if (callback != null) {
                                        callback.indexInfo(this.keys[leftIdx], this.ptrs[leftIdx]);
                                    }
                                    this.removeKey(leftIdx);
                                    this.removePointer(leftIdx);
                                    this.recalculateDataLen();
                                    return;
                                }
                                case -1: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (i == leftIdx) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                    }
                                    return;
                                }
                                case -5: 
                                case -4: 
                                case 4: 
                                case 5: 
                                case 10: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i < leftIdx || i > rightIdx) && (pos || i > leftIdx && i < rightIdx) || !query.testValue(this.keys[i])) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                        --i;
                                    }
                                    return;
                                }
                                case 7: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = leftIdx; i < rightIdx && i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                        --i;
                                    }
                                    if (rightIdx < this.nPtrs) return;
                                    BTree.this.removeSequential(transaction, this, query, callback);
                                    return;
                                }
                                case -6: 
                                case 6: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    if (rightIdx < 0) {
                                        rightIdx = -(rightIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (pos && (i < leftIdx || i > rightIdx) || !query.testValue(this.keys[i])) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                        --i;
                                    }
                                    return;
                                }
                                case -3: 
                                case 3: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i > leftIdx) && (pos || i < leftIdx) || !query.testValue(this.keys[i])) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                        --i;
                                    }
                                    return;
                                }
                                case -2: 
                                case 2: {
                                    if (leftIdx < 0) {
                                        leftIdx = -(leftIdx + 1);
                                    }
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if ((!pos || i < leftIdx) && (pos || i > leftIdx)) continue;
                                        if (query.testValue(this.keys[i])) {
                                            if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                                RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                                BTree.this.writeToLog(log, this);
                                            }
                                            if (callback != null) {
                                                callback.indexInfo(this.keys[i], this.ptrs[i]);
                                            }
                                            this.removeKey(i);
                                            this.removePointer(i);
                                            this.recalculateDataLen();
                                            --i;
                                            continue;
                                        }
                                        if (query.getOperator() != 7) continue;
                                        return;
                                    }
                                    return;
                                }
                                default: {
                                    for (int i = 0; i < this.nPtrs; ++i) {
                                        if (!query.testValue(this.keys[i])) continue;
                                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                                            BTree.this.writeToLog(log, this);
                                        }
                                        if (callback != null) {
                                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                                        }
                                        this.removeKey(i);
                                        this.removePointer(i);
                                        this.recalculateDataLen();
                                        --i;
                                    }
                                    return;
                                }
                            }
                        }
                        finally {
                            this.allowUnload = true;
                        }
                    }
                    default: {
                        throw new BTreeException("Invalid Page Type In query");
                    }
                }
            }
            switch (this.pageHeader.getStatus()) {
                case 2: {
                    for (int i = 0; i < this.nPtrs; ++i) {
                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                            BTree.this.writeToLog(log, this);
                        }
                        if (callback != null) {
                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                        }
                        this.removeKey(i);
                        this.removePointer(i);
                        this.recalculateDataLen();
                        --i;
                    }
                    return;
                }
                case 1: {
                    for (int i = 0; i < this.nKeys; ++i) {
                        if (query.getOperator() == 8 && !query.testValue(this.keys[i])) continue;
                        if (transaction != null && BTree.this.isRecoveryEnabled()) {
                            RemoveValueLoggable log = new RemoveValueLoggable(transaction, BTree.this.fileId, this.page.getPageNum(), i, this.keys[i], this.ptrs[i]);
                            BTree.this.writeToLog(log, this);
                        }
                        if (callback != null) {
                            callback.indexInfo(this.keys[i], this.ptrs[i]);
                        }
                        this.removeKey(i);
                        this.removePointer(i);
                        this.recalculateDataLen();
                        --i;
                    }
                    return;
                }
                default: {
                    throw new BTreeException("Invalid Page Type In query");
                }
            }
        }

        private void growPrefix() {
            if (this.nKeys == 0) {
                return;
            }
            if (this.nKeys == 1) {
                if (this.keys[0].getLength() > 0) {
                    byte[] newPrefix = new byte[this.prefix.getLength() + this.keys[0].getLength()];
                    System.arraycopy(this.prefix.data(), this.prefix.start(), newPrefix, 0, this.prefix.getLength());
                    System.arraycopy(this.keys[0].data(), this.keys[0].start(), newPrefix, this.prefix.getLength(), this.keys[0].getLength());
                    this.prefix = new Value(newPrefix);
                    this.keys[0] = Value.EMPTY_VALUE;
                }
                return;
            }
            int max = Integer.MAX_VALUE;
            Value first = this.keys[0];
            for (int i = 1; i < this.nKeys; ++i) {
                Value value = this.keys[i];
                int idx = Math.abs(value.compareTo(first));
                if (idx >= max) continue;
                max = idx;
            }
            int addChars = max - 1;
            if (addChars > 0) {
                byte[] pdata = new byte[this.prefix.getLength() + addChars];
                System.arraycopy(this.prefix.data(), this.prefix.start(), pdata, 0, this.prefix.getLength());
                System.arraycopy(this.keys[0].data(), this.keys[0].start(), pdata, this.prefix.getLength(), addChars);
                this.prefix = new Value(pdata);
                for (int i = 0; i < this.nKeys; ++i) {
                    Value key = this.keys[i];
                    this.keys[i] = new Value(key.data(), key.start() + addChars, key.getLength() - addChars);
                }
                this.recalculateDataLen();
            }
        }

        private void shrinkPrefix(int newLen) {
            int diff = this.prefix.getLength() - newLen;
            Value[] nv = new Value[this.nKeys];
            for (int i = 0; i < this.nKeys; ++i) {
                Value value = this.keys[i];
                byte[] ndata = new byte[value.getLength() + diff];
                System.arraycopy(this.prefix.data(), this.prefix.start() + newLen, ndata, 0, diff);
                System.arraycopy(value.data(), value.start(), ndata, diff, value.getLength());
                nv[i] = new Value(ndata);
            }
            this.keys = nv;
            this.prefix = new Value(this.prefix.data(), this.prefix.start(), newLen);
        }

        private void insertKey(Value val, int idx) {
            if (this.pageHeader.getStatus() == 2) {
                if (this.nKeys == 0) {
                    this.prefix = val;
                    val = Value.EMPTY_VALUE;
                } else {
                    int pfxLen = val.checkPrefix(this.prefix);
                    if (pfxLen < this.prefix.getLength()) {
                        this.shrinkPrefix(pfxLen);
                    }
                    val = new Value(val.data(), val.start() + pfxLen, val.getLength() - pfxLen);
                }
            }
            this.resizeKeys(this.nKeys + 1);
            System.arraycopy(this.keys, idx, this.keys, idx + 1, this.nKeys - idx);
            this.keys[idx] = val;
            this.pageHeader.setValueCount((short)(++this.nKeys));
            this.saved = false;
        }

        private void removeKey(int idx) {
            try {
                System.arraycopy(this.keys, idx + 1, this.keys, idx, this.nKeys - idx - 1);
            }
            catch (ArrayIndexOutOfBoundsException e) {
                Paged.LOG.error("keys: " + this.nKeys + " idx: " + idx);
            }
            this.pageHeader.setValueCount((short)(--this.nKeys));
            this.saved = false;
        }

        private void insertPointer(long ptr, int idx) {
            this.resizePtrs(this.nPtrs + 1);
            System.arraycopy(this.ptrs, idx, this.ptrs, idx + 1, this.nPtrs - idx);
            this.ptrs[idx] = ptr;
            ++this.nPtrs;
            this.saved = false;
        }

        private void removePointer(int idx) {
            System.arraycopy(this.ptrs, idx + 1, this.ptrs, idx, this.nPtrs - idx - 1);
            --this.nPtrs;
            this.saved = false;
        }

        private int searchKey(Value key) {
            if (this.pageHeader.getStatus() == 2 && this.prefix != null && this.prefix.getLength() > 0) {
                if (key.getLength() < this.prefix.getLength()) {
                    return key.compareTo(this.prefix) <= 0 ? -1 : -(this.nKeys + 1);
                }
                int pfxCmp = key.comparePrefix(this.prefix);
                if (pfxCmp < 0) {
                    return -1;
                }
                if (pfxCmp > 0) {
                    return -(this.nKeys + 1);
                }
                key = new Value(key.data(), key.start() + this.prefix.getLength(), key.getLength() - this.prefix.getLength());
            }
            int low = 0;
            int high = this.nKeys - 1;
            while (low <= high) {
                int mid = low + high >> 1;
                Value midVal = this.keys[mid];
                int cmp = midVal.compareTo(key);
                if (cmp < 0) {
                    low = mid + 1;
                    continue;
                }
                if (cmp > 0) {
                    high = mid - 1;
                    continue;
                }
                return mid;
            }
            return -(low + 1);
        }

        private void resizeKeys(int minCapacity) {
            int oldCapacity = this.keys.length;
            if (minCapacity > oldCapacity) {
                Value[] oldData = this.keys;
                int newCapacity = oldCapacity * 3 / 2 + 1;
                if (newCapacity < minCapacity) {
                    newCapacity = minCapacity;
                }
                this.keys = new Value[newCapacity];
                System.arraycopy(oldData, 0, this.keys, 0, this.nKeys);
            }
        }

        private void resizePtrs(int minCapacity) {
            int oldCapacity = this.ptrs.length;
            if (minCapacity > oldCapacity) {
                long[] oldData = this.ptrs;
                int newCapacity = oldCapacity * 3 / 2 + 1;
                if (newCapacity < minCapacity) {
                    newCapacity = minCapacity;
                }
                this.ptrs = new long[newCapacity];
                System.arraycopy(oldData, 0, this.ptrs, 0, this.nPtrs);
            }
        }

        static /* synthetic */ Value[] access$502(BTreeNode x0, Value[] x1) {
            x0.keys = x1;
            return x1;
        }
    }

    protected static class TreeInfo {
        final long firstPage;
        final int leafPages;

        TreeInfo(long firstPage, int leafs) {
            this.firstPage = firstPage;
            this.leafPages = leafs;
        }
    }
}

