/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.journal;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.text.DateFormat;
import java.util.Optional;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.EXistException;
import org.exist.config.annotation.ConfigurationClass;
import org.exist.config.annotation.ConfigurationFieldAsAttribute;
import org.exist.storage.BrokerPool;
import org.exist.storage.journal.FileSyncThread;
import org.exist.storage.journal.JournalException;
import org.exist.storage.journal.LogException;
import org.exist.storage.journal.Loggable;
import org.exist.storage.journal.Lsn;
import org.exist.storage.lock.FileLock;
import org.exist.storage.txn.Checkpoint;
import org.exist.util.FileUtils;
import org.exist.util.ReadOnlyException;
import org.exist.util.sanity.SanityCheck;

@ConfigurationClass(value="journal")
public class Journal {
    private static final Logger LOG = LogManager.getLogger(Journal.class);
    public static final String RECOVERY_SYNC_ON_COMMIT_ATTRIBUTE = "sync-on-commit";
    public static final String RECOVERY_JOURNAL_DIR_ATTRIBUTE = "journal-dir";
    public static final String RECOVERY_SIZE_LIMIT_ATTRIBUTE = "size";
    public static final String PROPERTY_RECOVERY_SIZE_LIMIT = "db-connection.recovery.size-limit";
    public static final String PROPERTY_RECOVERY_JOURNAL_DIR = "db-connection.recovery.journal-dir";
    public static final String PROPERTY_RECOVERY_SYNC_ON_COMMIT = "db-connection.recovery.sync-on-commit";
    public static final String LOG_FILE_SUFFIX = "log";
    public static final String BAK_FILE_SUFFIX = ".bak";
    public static final String LCK_FILE = "journal.lck";
    public static final int LOG_ENTRY_HEADER_LEN = 11;
    public static final int LOG_ENTRY_BASE_LEN = 13;
    public static final int DEFAULT_MAX_SIZE = 10;
    private static final long MIN_REPLACE = 0x100000L;
    @ConfigurationFieldAsAttribute(value="size")
    private final int journalSizeLimit;
    private FileOutputStream os;
    private FileChannel channel;
    private final FileSyncThread syncThread;
    private final Object latch = new Object();
    @ConfigurationFieldAsAttribute(value="journal-dir")
    private final Path dir;
    private FileLock fileLock;
    private int currentFile = 0;
    private int inFilePos = 0;
    private ByteBuffer currentBuffer;
    private long currentLsn = -1L;
    private long lastLsnWritten = -1L;
    private long lastSyncLsn = -1L;
    private boolean inRecovery = false;
    private final BrokerPool pool;
    @ConfigurationFieldAsAttribute(value="sync-on-commit")
    private static final boolean DEFAULT_SYNC_ON_COMMIT = true;
    private final boolean syncOnCommit;
    private final Path fsJournalDir;

    public Journal(BrokerPool pool, Path directory) throws EXistException {
        Optional<Path> logDir;
        this.pool = pool;
        this.fsJournalDir = directory.resolve("fs.journal");
        this.currentBuffer = ByteBuffer.allocateDirect(0x100000);
        this.syncThread = new FileSyncThread(this.latch);
        this.syncThread.start();
        this.syncOnCommit = pool.getConfiguration().getProperty(PROPERTY_RECOVERY_SYNC_ON_COMMIT, true);
        if (LOG.isDebugEnabled()) {
            LOG.debug("SyncOnCommit = " + this.syncOnCommit);
        }
        if ((logDir = Optional.ofNullable((Path)pool.getConfiguration().getProperty(PROPERTY_RECOVERY_JOURNAL_DIR))).isPresent()) {
            Path f = logDir.get();
            if (!f.isAbsolute()) {
                f = pool.getConfiguration().getExistHome().map(h -> Optional.of(h.resolve((Path)logDir.get()))).orElse(pool.getConfiguration().getConfigFilePath().map(p -> p.getParent().resolve((Path)logDir.get()))).orElse(f);
            }
            if (!Files.exists(f, new LinkOption[0])) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Output directory for journal files does not exist. Creating " + f.toAbsolutePath().toString());
                }
                try {
                    Files.createDirectories(f, new FileAttribute[0]);
                }
                catch (IOException | SecurityException e) {
                    throw new EXistException("Failed to create output directory: " + f.toAbsolutePath().toString());
                }
            }
            if (!Files.isWritable(f)) {
                throw new EXistException("Cannot write to journal output directory: " + f.toAbsolutePath().toString());
            }
            this.dir = f;
        } else {
            this.dir = directory;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Using directory for the journal: " + this.dir.toAbsolutePath().toString());
        }
        this.journalSizeLimit = 0x100000 * pool.getConfiguration().getProperty(PROPERTY_RECOVERY_SIZE_LIMIT, 10);
    }

    public void initialize() throws EXistException, ReadOnlyException {
        Path lck = this.dir.resolve(LCK_FILE);
        this.fileLock = new FileLock(this.pool, lck);
        boolean locked = this.fileLock.tryLock();
        if (!locked) {
            String lastHeartbeat = DateFormat.getDateTimeInstance(2, 2).format(this.fileLock.getLastHeartbeat());
            throw new EXistException("The journal log directory seems to be locked by another eXist process. A lock file: " + lck.toAbsolutePath().toString() + " is present in the log directory. Last access to the lock file: " + lastHeartbeat);
        }
    }

    public synchronized void writeToLog(Loggable loggable) throws JournalException {
        if (this.currentBuffer == null) {
            throw new JournalException("Database is shut down.");
        }
        SanityCheck.ASSERT(!this.inRecovery, "Write to log during recovery. Should not happen!");
        int size = loggable.getLogSize();
        int required = size + 13;
        if (required > this.currentBuffer.remaining()) {
            this.flushToLog(false);
        }
        this.currentLsn = Lsn.create(this.currentFile, this.inFilePos + this.currentBuffer.position() + 1);
        loggable.setLsn(this.currentLsn);
        try {
            this.currentBuffer.put(loggable.getLogType());
            this.currentBuffer.putLong(loggable.getTransactionId());
            this.currentBuffer.putShort((short)loggable.getLogSize());
            loggable.write(this.currentBuffer);
            this.currentBuffer.putShort((short)(size + 11));
        }
        catch (BufferOverflowException e) {
            throw new JournalException("Buffer overflow while writing log record: " + loggable.dump(), e);
        }
        this.pool.getTransactionManager().trackOperation(loggable.getTransactionId());
    }

    public long lastWrittenLsn() {
        return this.lastLsnWritten;
    }

    public void flushToLog(boolean fsync) {
        this.flushToLog(fsync, false);
    }

    public synchronized void flushToLog(boolean fsync, boolean forceSync) {
        if (this.inRecovery) {
            return;
        }
        this.flushBuffer();
        if (forceSync || fsync && this.syncOnCommit && this.currentLsn > this.lastSyncLsn) {
            this.syncThread.triggerSync();
            this.lastSyncLsn = this.currentLsn;
        }
        try {
            if (this.channel != null && this.channel.size() >= (long)this.journalSizeLimit) {
                this.pool.triggerCheckpoint();
            }
        }
        catch (IOException e) {
            LOG.warn("Failed to trigger checkpoint!", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushBuffer() {
        if (this.currentBuffer == null || this.channel == null) {
            return;
        }
        Object object = this.latch;
        synchronized (object) {
            try {
                if (this.currentBuffer.position() > 0) {
                    this.currentBuffer.flip();
                    int size = this.currentBuffer.remaining();
                    while (this.currentBuffer.hasRemaining()) {
                        this.channel.write(this.currentBuffer);
                    }
                    this.inFilePos += size;
                    this.lastLsnWritten = this.currentLsn;
                }
            }
            catch (IOException e) {
                LOG.warn("Flushing log file failed!", (Throwable)e);
            }
            finally {
                this.currentBuffer.clear();
            }
        }
    }

    public void checkpoint(long txnId, boolean switchLogFiles) throws JournalException {
        LOG.debug("Checkpoint reached");
        this.writeToLog(new Checkpoint(txnId));
        if (switchLogFiles) {
            this.flushBuffer();
        } else {
            this.flushToLog(true, true);
        }
        try {
            if (switchLogFiles && this.channel != null && this.channel.position() > 0x100000L) {
                Path oldFile = this.getFile(this.currentFile);
                RemoveThread rt = new RemoveThread(this.channel, oldFile);
                try {
                    this.switchFiles();
                }
                catch (LogException e) {
                    LOG.warn("Failed to create new journal: " + e.getMessage(), (Throwable)e);
                }
                rt.start();
            }
            this.clearBackupFiles();
        }
        catch (IOException e) {
            LOG.warn("IOException while writing checkpoint", (Throwable)e);
        }
    }

    public void setCurrentFileNum(int fileNum) {
        this.currentFile = fileNum;
    }

    public void clearBackupFiles() {
        if (Files.exists(this.fsJournalDir, new LinkOption[0])) {
            try (Stream<Path> backupFiles = Files.list(this.fsJournalDir);){
                backupFiles.forEach(p -> {
                    LOG.info("Checkpoint deleting: " + p.toAbsolutePath().toString());
                    if (!FileUtils.deleteQuietly(p)) {
                        LOG.fatal("Cannot delete file '" + p.toAbsolutePath().toString() + "' from backup journal.");
                    }
                });
            }
            catch (IOException ioe) {
                LOG.error("Could not clear fs.journal backup files", (Throwable)ioe);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void switchFiles() throws LogException {
        ++this.currentFile;
        String fname = Journal.getFileName(this.currentFile);
        Path file = this.dir.resolve(fname);
        if (Files.exists(file, new LinkOption[0])) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Journal file " + file.toAbsolutePath() + " already exists. Copying it.");
            }
            try {
                Path renamed = Files.move(file, file.resolveSibling(FileUtils.fileName(file) + BAK_FILE_SUFFIX), StandardCopyOption.ATOMIC_MOVE);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Old file renamed from '" + file.toAbsolutePath().toString() + "' to '" + renamed.toAbsolutePath().toString() + "'");
                }
            }
            catch (IOException ioe) {
                LOG.warn((Object)ioe);
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Creating new journal: " + file.toAbsolutePath().toString());
        }
        Object object = this.latch;
        synchronized (object) {
            this.close();
            try {
                this.os = new FileOutputStream(file.toFile(), true);
                this.channel = this.os.getChannel();
                this.syncThread.setChannel(this.channel);
            }
            catch (FileNotFoundException e) {
                throw new LogException("Failed to open new journal: " + file.toAbsolutePath().toString(), e);
            }
        }
        this.inFilePos = 0;
    }

    public void close() {
        if (this.channel != null) {
            try {
                this.channel.close();
            }
            catch (IOException e) {
                LOG.warn("Failed to close journal", (Throwable)e);
            }
        }
        if (this.os != null) {
            try {
                this.os.close();
            }
            catch (IOException e) {
                LOG.warn("Failed to close journal", (Throwable)e);
            }
        }
    }

    private static int journalFileNum(Path path) {
        String fileName = FileUtils.fileName(path);
        int p = fileName.indexOf(46);
        String baseName = fileName.substring(0, p);
        return Integer.parseInt(baseName, 16);
    }

    public static final int findLastFile(Stream<Path> files) {
        return files.map(Journal::journalFileNum).max(Integer::max).orElse(-1);
    }

    public Stream<Path> getFiles() throws IOException {
        String suffix = ".log";
        String indexSuffix = "_index.log";
        return Files.find(this.dir, 1, (path, attrs) -> attrs.isRegularFile() && FileUtils.fileName(path).endsWith(".log") && !FileUtils.fileName(path).endsWith("_index.log"), new FileVisitOption[0]);
    }

    public Path getFile(int fileNum) {
        return this.dir.resolve(Journal.getFileName(fileNum));
    }

    public void shutdown(long txnId, boolean checkpoint) {
        if (this.currentBuffer == null) {
            return;
        }
        if (!BrokerPool.FORCE_CORRUPTION) {
            if (checkpoint) {
                LOG.info("Transaction journal cleanly shutting down with checkpoint...");
                try {
                    this.writeToLog(new Checkpoint(txnId));
                }
                catch (JournalException e) {
                    LOG.error("An error occurred while closing the journal file: " + e.getMessage(), (Throwable)e);
                }
            }
            this.flushBuffer();
        }
        this.fileLock.release();
        this.syncThread.shutdown();
        try {
            this.syncThread.join();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        this.currentBuffer = null;
    }

    public void setInRecovery(boolean value) {
        this.inRecovery = value;
    }

    private static String getFileName(int fileNum) {
        String hex = Integer.toHexString(fileNum);
        hex = "0000000000".substring(hex.length()) + hex;
        return hex + '.' + LOG_FILE_SUFFIX;
    }

    private static class RemoveThread
    extends Thread {
        final FileChannel channel;
        final Path path;

        RemoveThread(FileChannel channel, Path path) {
            super("exist-removeJournalThread");
            this.channel = channel;
            this.path = path;
        }

        @Override
        public void run() {
            try {
                if (this.channel != null) {
                    this.channel.close();
                }
            }
            catch (IOException e) {
                LOG.warn("Exception while closing journal file: " + e.getMessage(), (Throwable)e);
            }
            FileUtils.deleteQuietly(this.path);
        }
    }
}

