/*
 * Decompiled with CFR 0.152.
 */
package org.exist.storage.lock;

import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.exist.storage.lock.DeadlockDetection;
import org.exist.storage.lock.Lock;
import org.exist.storage.lock.LockInfo;
import org.exist.storage.lock.LockOwner;
import org.exist.storage.lock.WaitingThread;
import org.exist.util.DeadlockException;
import org.exist.util.LockException;

public class MultiReadReentrantLock
implements Lock {
    private static final Logger LOG = LogManager.getLogger(MultiReadReentrantLock.class);
    private final Object id;
    private int waitingForReadLock = 0;
    private final List<LockOwner> outstandingReadLocks = new ArrayList<LockOwner>(4);
    private Thread writeLockedThread;
    private int outstandingWriteLocks = 0;
    private final List<WaitingThread> waitingForWriteLock = new ArrayList<WaitingThread>(3);

    public MultiReadReentrantLock(Object id) {
        this.id = id;
    }

    @Override
    public String getId() {
        return this.id.toString();
    }

    @Override
    public boolean acquire() throws LockException {
        return this.acquire(Lock.LockMode.READ_LOCK);
    }

    @Override
    public boolean acquire(Lock.LockMode mode) throws LockException {
        switch (mode) {
            case NO_LOCK: {
                LOG.warn("Acquired with LockMode.NO_LOCK!");
                return true;
            }
            case READ_LOCK: {
                return this.readLock(true);
            }
            case WRITE_LOCK: {
                return this.writeLock(true);
            }
        }
        throw new IllegalStateException();
    }

    @Override
    public boolean attempt(Lock.LockMode mode) {
        try {
            switch (mode) {
                case NO_LOCK: {
                    LOG.warn("Attempted acquire with LockMode.NO_LOCK!");
                    return true;
                }
                case READ_LOCK: {
                    return this.readLock(false);
                }
                case WRITE_LOCK: {
                    return this.writeLock(false);
                }
            }
            throw new IllegalStateException();
        }
        catch (LockException e) {
            return false;
        }
    }

    private synchronized boolean readLock(boolean waitIfNecessary) throws LockException {
        Thread thisThread = Thread.currentThread();
        if (this.writeLockedThread == thisThread) {
            this.outstandingReadLocks.add(new LockOwner(thisThread));
            return true;
        }
        this.deadlockCheck();
        ++this.waitingForReadLock;
        if (this.writeLockedThread != null) {
            if (!waitIfNecessary) {
                return false;
            }
            WaitingThread waiter = new WaitingThread(thisThread, this, this, Lock.LockMode.READ_LOCK);
            DeadlockDetection.addResourceWaiter(thisThread, waiter);
            while (this.writeLockedThread != null) {
                waiter.doWait();
            }
            DeadlockDetection.clearResourceWaiter(thisThread);
        }
        --this.waitingForReadLock;
        this.outstandingReadLocks.add(new LockOwner(thisThread));
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean writeLock(boolean waitIfNecessary) throws LockException {
        WaitingThread waiter;
        Thread thisThread = Thread.currentThread();
        MultiReadReentrantLock multiReadReentrantLock = this;
        synchronized (multiReadReentrantLock) {
            if (this.writeLockedThread == thisThread) {
                ++this.outstandingWriteLocks;
                return true;
            }
            if (this.writeLockedThread == null && this.grantWriteLock()) {
                this.writeLockedThread = thisThread;
                ++this.outstandingWriteLocks;
                return true;
            }
            if (!waitIfNecessary) {
                return false;
            }
            this.deadlockCheck();
            waiter = new WaitingThread(thisThread, thisThread, this, Lock.LockMode.WRITE_LOCK);
            this.addWaitingWrite(waiter);
            DeadlockDetection.addResourceWaiter(thisThread, waiter);
        }
        List<WaitingThread> deadlockedThreads = null;
        LockException exceptionCaught = null;
        Iterator<WaitingThread> iterator = thisThread;
        synchronized (iterator) {
            if (thisThread != this.writeLockedThread) {
                while (thisThread != this.writeLockedThread && deadlockedThreads == null) {
                    if (LockOwner.DEBUG) {
                        StringBuilder buf = new StringBuilder("Waiting for write: ");
                        for (int i = 0; i < this.waitingForWriteLock.size(); ++i) {
                            buf.append(' ');
                            buf.append(this.waitingForWriteLock.get(i).getThread().getName());
                        }
                        LOG.debug(buf.toString());
                        this.debugReadLocks("WAIT");
                    }
                    if ((deadlockedThreads = this.checkForDeadlock(thisThread)) != null) continue;
                    try {
                        waiter.doWait();
                    }
                    catch (LockException e) {
                        exceptionCaught = e;
                        break;
                    }
                }
            }
            if (deadlockedThreads == null && exceptionCaught == null) {
                ++this.outstandingWriteLocks;
            }
        }
        iterator = this;
        synchronized (iterator) {
            DeadlockDetection.clearResourceWaiter(thisThread);
            this.removeWaitingWrite(waiter);
        }
        if (exceptionCaught != null) {
            throw exceptionCaught;
        }
        if (deadlockedThreads != null) {
            for (WaitingThread wt : deadlockedThreads) {
                wt.signalDeadlock();
            }
            throw new DeadlockException();
        }
        return true;
    }

    private void addWaitingWrite(WaitingThread waiter) {
        this.waitingForWriteLock.add(waiter);
    }

    private void removeWaitingWrite(WaitingThread waiter) {
        for (int i = 0; i < this.waitingForWriteLock.size(); ++i) {
            WaitingThread next = this.waitingForWriteLock.get(i);
            if (next.getThread() != waiter.getThread()) continue;
            this.waitingForWriteLock.remove(i);
            break;
        }
    }

    public void release() {
        this.release(Lock.LockMode.READ_LOCK);
    }

    @Override
    public void release(Lock.LockMode mode) {
        switch (mode) {
            case NO_LOCK: {
                LOG.warn("Released with LockMode.NO_LOCK!");
                break;
            }
            case READ_LOCK: {
                this.releaseRead(1);
                break;
            }
            case WRITE_LOCK: {
                this.releaseWrite(1);
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    @Override
    public void release(Lock.LockMode mode, int count) {
        switch (mode) {
            case NO_LOCK: {
                LOG.warn("Released with LockMode.NO_LOCK and count=" + count + "!");
                break;
            }
            case READ_LOCK: {
                this.releaseRead(count);
                break;
            }
            case WRITE_LOCK: {
                this.releaseWrite(count);
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void releaseWrite(int count) {
        if (Thread.currentThread() == this.writeLockedThread) {
            if (this.outstandingWriteLocks > 0) {
                this.outstandingWriteLocks -= count;
            }
            if (this.outstandingWriteLocks > 0) {
                return;
            }
            if (this.grantWriteLockAfterRead()) {
                WaitingThread waiter = this.waitingForWriteLock.get(0);
                this.removeWaitingWrite(waiter);
                DeadlockDetection.clearResourceWaiter(waiter.getThread());
                Thread thread = this.writeLockedThread = waiter.getThread();
                synchronized (thread) {
                    this.writeLockedThread.notifyAll();
                }
            } else {
                this.writeLockedThread = null;
                if (this.waitingForReadLock > 0) {
                    this.notifyAll();
                }
            }
        } else {
            LOG.warn("Possible lock problem: a thread released a write lock it didn't hold. Either the thread was interrupted or it never acquired the lock.", new Throwable());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private synchronized void releaseRead(int count) {
        if (!this.outstandingReadLocks.isEmpty()) {
            this.removeReadLock(count);
            if (this.writeLockedThread == null && this.grantWriteLockAfterRead()) {
                WaitingThread waiter = this.waitingForWriteLock.get(0);
                this.removeWaitingWrite(waiter);
                DeadlockDetection.clearResourceWaiter(waiter.getThread());
                Thread thread = this.writeLockedThread = waiter.getThread();
                synchronized (thread) {
                    this.writeLockedThread.notifyAll();
                }
            }
            return;
        }
        LOG.warn("Possible lock problem: thread " + Thread.currentThread().getName() + " released a read lock it didn't hold. Either the thread was interrupted or it never acquired the lock. Write lock: " + (this.writeLockedThread != null ? this.writeLockedThread.getName() : "null"), new Throwable());
        if (LockOwner.DEBUG) {
            this.debugReadLocks("ILLEGAL RELEASE");
        }
    }

    @Override
    public synchronized boolean isLockedForWrite() {
        return this.writeLockedThread != null || this.waitingForWriteLock != null && this.waitingForWriteLock.size() > 0;
    }

    @Override
    public synchronized boolean hasLock() {
        return !this.outstandingReadLocks.isEmpty() || this.isLockedForWrite();
    }

    @Override
    public synchronized boolean isLockedForRead(Thread owner) {
        for (int i = this.outstandingReadLocks.size() - 1; i > -1; --i) {
            if (this.outstandingReadLocks.get(i).getOwner() != owner) continue;
            return true;
        }
        return false;
    }

    private void removeReadLock(int count) {
        Thread owner = Thread.currentThread();
        for (int i = this.outstandingReadLocks.size() - 1; i > -1 && count > 0; --i) {
            LockOwner current = this.outstandingReadLocks.get(i);
            if (current.getOwner() != owner) continue;
            this.outstandingReadLocks.remove(i);
            --count;
        }
    }

    private void deadlockCheck() throws DeadlockException {
        for (LockOwner next : this.outstandingReadLocks) {
            Lock lock = DeadlockDetection.isWaitingFor(next.getOwner());
            if (lock == null) continue;
            lock.wakeUp();
        }
    }

    private List<WaitingThread> checkForDeadlock(Thread waiter) {
        ArrayList<WaitingThread> waiters = new ArrayList<WaitingThread>(10);
        if (DeadlockDetection.wouldDeadlock(waiter, this.writeLockedThread, waiters)) {
            LOG.warn("Potential deadlock detected on lock " + this.getId() + "; killing threads: " + waiters.size());
            return waiters.size() > 0 ? waiters : null;
        }
        return null;
    }

    private boolean grantWriteLock() {
        if (this.outstandingReadLocks.isEmpty()) {
            return true;
        }
        Thread waiter = Thread.currentThread();
        for (LockOwner next : this.outstandingReadLocks) {
            if (next.getOwner() == waiter || DeadlockDetection.isBlockedBy(waiter, next.getOwner())) continue;
            return false;
        }
        return true;
    }

    private boolean grantWriteLockAfterRead() {
        if (this.waitingForWriteLock != null && this.waitingForWriteLock.size() > 0) {
            int size = this.outstandingReadLocks.size();
            if (size > 0) {
                WaitingThread waiter = this.waitingForWriteLock.get(0);
                return this.isCompatible(waiter.getThread());
            }
            return true;
        }
        return false;
    }

    private boolean hasReadLock(Thread owner) {
        for (LockOwner next : this.outstandingReadLocks) {
            if (next.getOwner() != owner) continue;
            return true;
        }
        return false;
    }

    public Thread getWriteLockedThread() {
        return this.writeLockedThread;
    }

    @Override
    public boolean hasLock(Thread owner) {
        if (this.writeLockedThread == owner) {
            return true;
        }
        return this.hasReadLock(owner);
    }

    @Override
    public void wakeUp() {
    }

    private boolean isCompatible(Thread waiting) {
        for (LockOwner next : this.outstandingReadLocks) {
            if (next.getOwner() == waiting || DeadlockDetection.isBlockedBy(waiting, next.getOwner())) continue;
            return false;
        }
        return true;
    }

    @Override
    public synchronized LockInfo getLockInfo() {
        LockInfo info;
        String[] readers = new String[]{};
        if (this.outstandingReadLocks != null) {
            readers = new String[this.outstandingReadLocks.size()];
            for (int i = 0; i < this.outstandingReadLocks.size(); ++i) {
                LockOwner owner = this.outstandingReadLocks.get(i);
                readers[i] = owner.getOwner().getName();
            }
        }
        if (this.writeLockedThread != null) {
            info = new LockInfo("RESOURCE", "WRITE", this.getId(), new String[]{this.writeLockedThread.getName()});
            info.setReadLocks(readers);
        } else {
            info = new LockInfo("RESOURCE", "READ", this.getId(), readers);
        }
        if (this.waitingForWriteLock != null) {
            String[] waitingForWrite = new String[this.waitingForWriteLock.size()];
            for (int i = 0; i < this.waitingForWriteLock.size(); ++i) {
                waitingForWrite[i] = this.waitingForWriteLock.get(i).getThread().getName();
            }
            info.setWaitingForWrite(waitingForWrite);
        }
        return info;
    }

    private void debugReadLocks(String msg) {
        for (LockOwner owner : this.outstandingReadLocks) {
            LOG.debug(msg + ": " + owner.getOwner(), owner.getStack());
        }
    }

    @Override
    public void debug(PrintStream out) {
        this.getLockInfo().debug(out);
    }
}

