/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import net.xfra.qizxopen.util.Collations;

import java.text.Collator;
import java.io.PrintStream;

/**
 *	An Implementation of the Data Model above FONI documents, which can be
 *	used independently of XQuery.
 *	<p>Used for indexed and parsed documents.
 */
public class FONIDM
{
    protected FONIDocument dom;
    protected Node root;
    // rather dirty way of referencing an optional Library to which the document belongs:
    protected Object  owner;

    ISequence emptySequence = new ISequence(0);	// cant be static 

    public FONIDM( FONIDocument dom ) {
	this.dom = dom;
	root = newDmNode(dom.getRootNode());
    }

    /**
     *	Returns the document (or root) node.
     */
    public Node document() {
	return root;
    }

    /**
     *	Base URI of the document.
     */
    public String getDocumentURI() {
	return dom.getBaseURI();
    }

    /**
     *	For cache management.
     */
    public int estimateMemorySize() {
	return dom.estimateMemorySize();
    }

    /** Internal purpose. */
    public Node newDmNode( int id ) {	// TODO? cached allocation !not thread safe
	if(id == 0)
	    return null;
	return new BaseNode(id);	
    }

    public void setOwner(Object owner) {
        this.owner = owner;
    }

    // Basic Node (single id)
    public class BaseNode implements Node {

	protected int id;

	public BaseNode( int id ) {
	    this.id = id;
	}

	protected FONIDocument getDom() {
	    return dom;
	}

	public Object getOwner() {
	    return owner;
	}

	public int getDocId() {
	    return dom.getDocId();
	}

	public int getNodeId() {
	    return id;
	}

	public String getNodeKind() {
	    switch(dom.getKind(id)) {
		case FONIDocument.DOCUMENT:
		    return "document";
		case FONIDocument.ELEMENT:
		    return "element";
		case FONIDocument.ATTRIBUTE:
		    return "attribute";
		case FONIDocument.TEXT:
		    return "text";
		case FONIDocument.PROCESSING_INSTRUCTION:
		    return "processing-instruction";
		case FONIDocument.COMMENT:
		    return "comment";
		case FONIDocument.NAMESPACE:
		    return "namespace";
		default:
		    throw new RuntimeException("wrong FONInode, id="+id);
	    }
	}

	public QName  getNodeName() {
	    return dom.getName(id);
	}

	public Node   parent() {
	    return newDmNode(dom.getParent(id));
	}

	public String getStringValue() {
	    return dom.getStringValue(id);
	}

	public String getDocumentURI() {
	    return dom.getBaseURI();
	}

	public String getBaseURI() {
	    int nameid = dom.internOtherName(QName.get(Namespace.XML, "base"));
	    if(nameid >= 0)
		for(int pid = id; pid != 0; pid = dom.getParent(pid)) {
		    int attrId = dom.getAttribute(id, nameid);
		    if(attrId != 0)
			return dom.pnGetStringValue(attrId);
		}
	    return dom.getBaseURI();
	}

	public NodeSequence children() {
	    // OPTIMIZED by avoiding an inner iterator
	    //return dom.newChildrenIterator(id);
	    return new ISequence(- dom.getFirstChild(id));
	}

	public NodeSequence attributes() {
	    return new ASequence( id, dom.attrIterator(id) );
	}

	public int getDefinedNSCount() {
	    return dom.getDefinedNSCount(id);
	}

	public NodeSequence namespaces( boolean inScope ) {
	    return new ASequence( id, dom.namespaceIterator(id, inScope) ) {
		    ANode makeNode(int id) {
			return new NSNode(ownerId, id);
		    }
		};
	}

	

	public int docPosition() {
	    if(owner != null) {	// part of a library: use the docIds
		return owner.hashCode() + dom.getDocId();
	    }
	    return FONIDM.this.hashCode();	    
	}

	public int   getNature() {
	    
	    return dom.getKind(id);	// CAUTION: codes must be the same in FONIDocument and Node
	}

	public boolean isElement() {
	    return dom.getKind(id) == FONIDocument.ELEMENT;
	}

	public Node  document() {
	    return newDmNode(dom.getRootNode());
	}

	public String toString() {
	    return "FONI "+getNodeKind()+" id="+id+" "+
		(getNodeName() != null ? (" name="+getNodeName()) : getStringValue());
	}

	public Node  attribute( QName name ) {
	    return newDmNode(dom.getAttribute(id, dom.internOtherName(name)));
	}

	public String getNsPrefix( String nsuri ) {
	    for (int nodeId = id; nodeId != 0; nodeId = dom.getParent(nodeId)) {
		FONIDocument.NodeIterator ns = dom.namespaceIterator(nodeId, true);
		for( ; ns.next(); )
		    if(nsuri.equals(dom.pnGetStringValue(ns.currentId())))
			return dom.pnGetName(ns.currentId()).getLocalName();
	    }
	    return null;
	}

	public String getNsUri( String prefix ) {
	    for (int nodeId = id; nodeId != 0; nodeId = dom.getParent(nodeId)) {
		FONIDocument.NodeIterator ns = dom.namespaceIterator(nodeId, true);
		for( ; ns.next(); ) {
		    if(prefix.equals(dom.pnGetName(ns.currentId()).getLocalName()))
			return dom.pnGetStringValue(ns.currentId());
		}
	    }
	    return null;
	}

	public int orderCompare( Node node ) {
	    if( node instanceof BaseNode) {
		BaseNode inode = (BaseNode) node;
		if(inode.getDom() == dom ) {	    //
		    
		    return id < inode.id ? -1 : id > inode.id ? 1 : 0;
		}
	    }
	    int cmp = docPosition() - node.docPosition();
	    return cmp < 0 ? -1 : cmp > 0 ? 1 : 0;
	}

	public boolean contains( Node node ) {
	    if( !(node instanceof BaseNode) )
		return false;	// or exception?
	    int delta = id - ((BaseNode) node).id;
	    return delta >= 0 && delta < dom.getNodeSpan(id);
	}

	public int getNodeSpan() {
	    return dom.getNodeSpan(id);
	}

	public int getNodeDepth() {
	    int depth = 0;
	    for (int nodeId = id; nodeId != 0; nodeId = dom.getParent(nodeId))
		++ depth;
	    return depth;
	}

	public boolean equals( Object that ) {
	    if(!(that instanceof BaseNode))
		return false;
	    BaseNode thatNode = (BaseNode) that;
	    return thatNode.id == id && thatNode.getDom() == dom;
	}

	public int hashCode() {
	    return id;
	}

	boolean dumpNode( PrintStream out, int nodeId ) {

	    int type = dom.getKind(nodeId);
	    switch(type) {
	    case ELEMENT:
	    case DOCUMENT: {
		int gi = dom.getNameId(nodeId), attC = dom.getAttrCount(nodeId);
		QName name = dom.getElementName(gi);
		out.print("Element "+ gi +" "+ name + "  " + attC + " attr:"); 
		StringBuffer buf = new StringBuffer();
		FONIDocument.NodeIterator attrIter = dom.attrIterator(nodeId);
		while(attrIter.next()) {
		    int attrNode = attrIter.currentId();
		    name = dom.getOtherName(dom.pnGetNameId(attrNode));
		    buf.append(' ').append(name);
		    buf.append("='").append(dom.pnGetStringValue(attrNode)).append('\'');
		}
		FONIDocument.NodeIterator nsit = dom.namespaceIterator(nodeId, false);
		while(nsit.next()) {
		    int nsNode = nsit.currentId();
		    
		    name = dom.getOtherName(dom.pnGetNameId(nsNode));
		    buf.append("xmlns:").append(name).append("='").
			append(dom.pnGetStringValue(nsNode)).append('\'');
		}
		System.out.println(buf);
		return true;
	    }
	    case FONIDocument.TEXT:
		out.print("Text |"); out.print(dom.getStringValue(nodeId)); out.println('|');
		break;
	    case FONIDocument.PROCESSING_INSTRUCTION:
		out.print("PI "+dom.getName(nodeId)+" |");
		out.print(dom.getStringValue(nodeId)); out.println('|');
		break;
	    case FONIDocument.COMMENT:
		out.print("Comment |"); out.print(dom.getStringValue(nodeId));out.println('|');
		break;
	    }
	    return false;
	}

	void dumpTree(PrintStream out, int nodeId, int depth ) {
	    out.print(nodeId);
	    for(int d = 0; d < depth; d++)
		out.print("  ");
	    if(dumpNode(out, nodeId)) {
		int kid = dom.getFirstChild(nodeId), next;
		int rank = 0;
		for( ; kid != 0; kid = next, ++ rank) {
		    next = dom.getNextSibling(kid); 
		    dumpTree(out, kid, depth + 1);
		}
	    }
	}

	public boolean deepEqual( Node node, Collator collator ) {
	    return deepEq( id, node, collator);
	}

	protected boolean deepEq( int id, Node that, Collator collator ) {
	    int kind = dom.getKind(id);
	    
	    if(kind != that.getNature() || dom.getName(id) != that.getNodeName() )
		return false;
	    if(kind == Node.DOCUMENT)
		return contentEq( id, that, collator);
	    else if(kind == Node.ELEMENT)
		return attributesEq( id, that, collator)
		    && contentEq( id, that, collator);
	    else {	//TODO OPTIM
		return Collations.compare( dom.getStringValue(id), that.getStringValue(),
					   collator) == 0;
	    }
	}

	boolean attributesEq( int id, Node that, Collator collator ) {
	    NodeSequence oattr = that.attributes();
	    int acnt = 0;
	    for(; oattr.nextNode(); ++ acnt) {
		Node oatt = oattr.currentNode();
		int attr = dom.getAttribute(id, dom.internOtherName(oatt.getNodeName()));
		if( attr == 0 || Collations.compare( dom.pnGetStringValue(attr),
						     oatt.getStringValue(), collator) != 0)
		    return false;
	    }
	    return dom.getAttrCount(id) == acnt;
	}

	boolean contentEq( int id, Node that, Collator collator ) {
	    
	    NodeSequence okids = that.children();
	    for(int kid = dom.getFirstChild(id); kid != 0; kid = dom.getNextSibling(kid)) {
		int kidk = dom.getKind(kid), okidk;
		if(kidk == Node.COMMENT || kidk == Node.PROCESSING_INSTRUCTION)
		    continue;
		Node okid;
		for( ; ; ) {
		    if(!okids.nextNode())
			return false;
		    okid = okids.currentNode();
		    if( (okidk = okid.getNature()) != Node.COMMENT &&
			 okidk != Node.PROCESSING_INSTRUCTION)
			break;
		}
		// we have 2 comparable kids:
		if(!deepEq( kid, okid, collator))
		   return false;
	    }
	    // just check that there are no more kids in 'that'
	    return !okids.nextNode();
	}

	public int compareStringValues(Node node, Collator collator) {
	    String s1 = this.getStringValue(), s2 = node.getStringValue();
	    return collator != null ? collator.compare(s1, s2) : s1.compareTo(s2);
	}

	

	public NodeSequence ancestors( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(id, nodeTest);
	}

	public NodeSequence ancestorsOrSelf( NodeTest nodeTest ) {
	    return new AncestorsOrSelf(id, nodeTest);
	}

	public NodeSequence parent( NodeTest nodeTest ) {
	    return new Parent(id, nodeTest);
	}

	public NodeSequence children( NodeTest nodeTest ) {
	    return new Children(id, nodeTest);
	}

	public NodeSequence descendants( NodeTest nodeTest ) {
	    return new Descendants(id, nodeTest);
	}

	public NodeSequence descendantsOrSelf( NodeTest nodeTest ) {
	    return new DescendantsOrSelf(id, nodeTest);
	}

	public NodeSequence attributes( NodeTest nodeTest ) {
	    return new Attributes(id, nodeTest);
	}

	public NodeSequence followingSiblings( NodeTest nodeTest ) {
	    return new FollowingSiblings(id, nodeTest);
	}

	public NodeSequence precedingSiblings( NodeTest nodeTest ) {
	    return new PrecedingSiblings(id, nodeTest);
	}

	public NodeSequence following( NodeTest nodeTest ) {
	    return new Following(id, nodeTest);
	}

	public NodeSequence preceding( NodeTest nodeTest ) {
	    return new Preceding(id, nodeTest);
	}

	public void addChild( Node child ) { 
	    throw new RuntimeException(getClass() + ".addChild!");
	}

	public void addAttribute( Node attribute ) { 
	    throw new RuntimeException(getClass() + ".addAttribute!");
	}

	public void addText( String value ) {
	    throw new RuntimeException(getClass() + ".setStringValue!");
	}

	

	public char[]  getChars() {
	    return dom.getCharValue(id, 0);
	}

	public Object  getAtomValue() {	// TODO
	    return getStringValue();
	}

	public int  getAtomType() {
	    return TYPE_STRING;
	}

	public long getIntegerValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Long)
		return ((Long) obj).longValue();
	    try {
		return Long.parseLong(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to integer", e);
	    }
	}

	public double getDoubleValue() throws DataModelException {
	    Object obj = getAtomValue();
	    if(obj instanceof Double)
		return ((Double) obj).doubleValue();
	    try {
		return Double.parseDouble(getStringValue());
	    }
	    catch (NumberFormatException e) {
		throw new DataModelException("atom cannot be converted to double", e);
	    }
	}
    }
 
    // Attribute and namespace nodes: 
    // the id is the owner node's, an additional offset points the value
    class ANode extends BaseNode
    {
	int offset;

	ANode( int id, int offset ) {
	    super(id);
	    this.offset = offset;
	}

	public boolean equals( Object that ) {
	    if(!(that instanceof ANode))
		return false;
	    ANode thatNode = (ANode) that;
	    return thatNode.id == id && thatNode.offset == offset &&
		   thatNode.getDom() == dom;
	}

	public int hashCode() {
	    return id ^ offset;
	}

	public int orderCompare( Node node ) {
	    int cmp = super.orderCompare(node);
	    if(cmp != 0)
		return cmp;
	    if( !(node instanceof ANode) )
		return 1;	// after owner
	    // attributes of the same node:
	    ANode anode = (ANode) node;
	    cmp = offset - anode.offset;
	    return cmp < 0 ? -1 : cmp > 0 ? 1 : 0;
	}

	public int getNature() {
	    return FONIDocument.ATTRIBUTE;
	}

	public String getNodeKind() {
	    return "attribute";
	}

	public QName  getNodeName() {
	    return dom.getOtherName(dom.pnGetNameId(offset));
	}

	public String getStringValue() {
	    return dom.pnGetStringValue(offset);
	}

	public Node   parent() {
	    return newDmNode(id);
	}

	public NodeSequence children() {
	    return emptySequence;
	}

	public NodeSequence attributes() {
	    return emptySequence;
	}

	public NodeSequence namespaces() {
	    return emptySequence;
	}
    }

    // namespace Node:
    class NSNode extends ANode {
	NSNode( int id, int offset ) {
	    super(id, offset);
	}
	public String getNodeKind() {
	    return "namespace";
	}
	public int getNature() {
	    return FONIDocument.NAMESPACE;
	}
    }

    // Sequence on BaseNodes:
    class ISequence implements NodeSequence
    {
	int startId, curId;

	ISequence( int startId ) {
	    curId = this.startId = startId;
	}

	public NodeSequence reborn() {
	    return new ISequence( startId );
	}

	public boolean nextNode() {
	    if(curId < 0)
		 curId = -curId;	// trick for children iterator
	    else curId = dom.getNextSibling(curId);
	    return curId != 0;
	}

	public int     currentId() {
	    return curId;
	}

	public Node    currentNode() {
	    return curId == 0 ? null : newDmNode(curId);
	}
    }

    abstract class TypedSequence extends ISequence {

	NodeTest nodeTest;
	int nameId = -1;
	boolean started = false;

	TypedSequence( int id, NodeTest nodeTest ) {
	    super(id);
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    if( curId == 0)
		return false;
	    return ( nodeTest == null ||
		     (nodeTest.needsNode() ? nodeTest.accepts(currentNode())
		      : nodeTest.accepts( dom.getKind(curId), dom.getName(curId) ) ));
	}
    }

    class Parent extends TypedSequence
    {

	Parent( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public boolean nextNode() {
	    if(started)
		return false;
	    started = true;
	    curId = dom.getParent(curId);
	    return checkNode();
	}
	
	public NodeSequence reborn() {
	    return new Parent( startId, nodeTest );
	}
    }

    class AncestorsOrSelf extends TypedSequence
    {
	AncestorsOrSelf( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public boolean next() {
	    for(; curId != 0; ) {
		if(started)
		    curId = dom.getParent(curId);
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}

	public NodeSequence reborn() {
	    return new AncestorsOrSelf( startId, nodeTest );
	}
    }

    class Ancestors extends AncestorsOrSelf
    {
	Ancestors( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    started = true;
	}

	public NodeSequence reborn() {
	    return new Ancestors( startId, nodeTest );
	}
    }

    class Children extends TypedSequence
    {
	Children( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    curId = dom.getFirstChild( id );
	}

	public NodeSequence reborn() {
	    return new Children( startId, nodeTest );
	}

	public boolean nextNode() {
	    for(; curId != 0; ) {
		if(started)
		     curId = dom.getNextSibling( curId );
		started = true;
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class DescendantsOrSelf extends TypedSequence
    {
	int lastNode;

	DescendantsOrSelf( int id, NodeTest nodeTest ) {
	    super( id, nodeTest);
	    lastNode = dom.getNodeAfter(Math.abs(id));
	    if(lastNode == 0)
		lastNode = Integer.MAX_VALUE;
	}

	public NodeSequence reborn() {
	    return new DescendantsOrSelf( startId, nodeTest );
	}

	public boolean nextNode() {
	    for(; curId != 0; ) {
		if(started)
		    curId = dom.getNodeNext(curId);
		started = true;
		if(curId >= lastNode)
		    curId = 0;
		else if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Descendants extends DescendantsOrSelf
    {
	Descendants( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    started = true;
	}

	public NodeSequence reborn() {
	    return new Descendants( startId, nodeTest );
	}
    }

    class FollowingSiblings extends TypedSequence
    {
	FollowingSiblings( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public NodeSequence reborn() {
	    return new FollowingSiblings( startId, nodeTest );
	}

	public boolean nextNode() {
	    for( ; curId != 0; ) {
		curId = dom.getNextSibling(curId);
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Following extends TypedSequence
    {
	Following( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	}

	public NodeSequence reborn() {
	    return new Following( startId, nodeTest );
	}

	public boolean nextNode() {
	    for( ; curId != 0; ) {
		curId = dom.getNodeNext(curId);
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class PrecedingSiblings extends TypedSequence
    {
	PrecedingSiblings( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    curId = dom.getFirstChild( dom.getParent(id) );
	}

	public NodeSequence reborn() {
	    return new PrecedingSiblings( startId, nodeTest );
	}

	public boolean nextNode() {
	    for( ; curId != 0; ) {
		if(started)
		    curId = dom.getNextSibling(curId);
		started = true;
		if(curId == startId) {
		    curId = 0; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }

    class Preceding extends PrecedingSiblings
    {
	Preceding( int id, NodeTest nodeTest ) {
	    super(id, nodeTest);
	    curId = dom.getRootNode();
	}

	public NodeSequence reborn() {
	    return new Preceding( startId, nodeTest );
	}

	public boolean nextNode() {
	    for( ; curId != 0; ) {
		if(started)
		    curId = dom.getNodeNext(curId);
		started = true;
		if(curId == startId) {
		    curId = 0; return false;
		}
		if(checkNode())
		    return true;
	    }
	    return false;
	}
    }


    class ASequence implements NodeSequence {
	FONIDocument.NodeIterator domIter;
	int ownerId;
	int curId;

	ASequence( int ownerId, FONIDocument.NodeIterator iter ) {
	    domIter = iter;
	    this.ownerId = ownerId;
	}

	public boolean nextNode() {
	    if( !domIter.next())
		return false;
	    curId = domIter.currentId();
	    
	    return true;
	}

	public Node    currentNode() {
	    curId = domIter.currentId();
	    return curId == 0 ? null : makeNode(curId);
	}

	ANode makeNode(int id) {
	    return new ANode(ownerId, id);
	}

	public NodeSequence reborn() {
	    return new ASequence( ownerId, domIter.reborn() );
	}
    }

    class Attributes extends ASequence
    {
	NodeTest nodeTest;
	int nameId = -1;

	Attributes( int ownerId, NodeTest nodeTest ) {
	    super(ownerId, dom.attrIterator(ownerId));
	    this.nodeTest = nodeTest;
	}

	boolean checkNode() {
	    if( curId == 0)
		return false;
	    return ( nodeTest == null ||
		     (nodeTest.needsNode() ? nodeTest.accepts(makeNode(curId))
		      : nodeTest.accepts( Node.ATTRIBUTE, dom.pnGetName(curId) ) ));
	}

	public NodeSequence reborn() {
	    return new Attributes( ownerId, nodeTest );
	}

	public boolean nextNode() {
	    for( ; super.nextNode() ; )
		if(checkNode())
		    return true;
	    return false;
	}
    }
}
