/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.QName;
import net.xfra.qizxopen.util.Namespace;
import net.xfra.qizxopen.util.NSPrefixMapping;

/**
 *	An abstract receiver of SAX-like events. 
 *	<p>Manages namespaces. Used as a base for serialization and SAX output.
 */
public abstract class XMLEventReceiverBase implements XMLEventReceiver
{
    protected NSPrefixMapping prefixes = new NSPrefixMapping();
    protected NSPrefixMapping prefixHints;	// optional: to help choosing a nice prefix
    protected int    depth = 0;
    protected long   maxVolume = -1, volume = 0;
    protected final static String VOLUME_LIMIT = "volume limit reached";

    // number of NS defined by the current element:
    protected int    nsCnt;
    // nsCnt saved in a stack for proper removal of prefix mappings:
    protected int[]  nsCounts = new int[16];

    protected boolean elementStarted, spaceNeeded, docStarted;
    protected boolean startDocumentDone, endDocumentDone;
    protected boolean trace = false;
    protected QName   tagName;

    // buffer attributes:
    protected QName[] attrNames;
    protected String[] attrValues;
    protected int  attrCnt;

    static final char[] blank = new char[] { ' '};

    protected abstract void flushElement( boolean isEmpty ) throws DataModelException;

    public void definePrefixHints( NSPrefixMapping prefixes ) {
	prefixHints = prefixes;
    }

    public void setTrace(boolean value) {
        trace = value;
    }

    public void setMaxVolume(int volume) {
	maxVolume = volume;
    }

    public boolean maxVolumeReached() {
	return volume >= maxVolume;
    }

    public void reset() {
	elementStarted = docStarted = false;
	startDocumentDone = endDocumentDone = false;
	spaceNeeded = false;
	depth = 0;
	attrCnt = 0;
	prefixes = new NSPrefixMapping();
	nsCounts = new int[16];
	attrNames = new QName[8];
	attrValues = new String[8];
    }

    public void terminate() throws DataModelException {
	if(!endDocumentDone)
	    endDocument();
    }

    public void startDocument() throws DataModelException {
	if(attrNames == null)
	    reset(); 	// for absent-minded people (eg me)
	startDocumentDone = true;
    }

    public void endDocument() throws DataModelException {
	endDocumentDone = true;
    }

    public void startElement(QName name) throws DataModelException {
	if(!startDocumentDone) {
	    startDocument();
	    startDocumentDone = true;
	}
	if(trace) System.err.println("--- start elem "+name);
	// TODO: whether 'tis allowed or not ?
	//if(depth == 0 && docStarted)
	//    throw new DataModelException("element after end of document");
	docStarted = true;
	if(elementStarted)
	    flushElement(false);

	if(depth >= nsCounts.length) {
            int[] old = nsCounts;
            nsCounts = new int[ old.length * 2 ];
            System.arraycopy(old, 0, nsCounts, 0, old.length);
        }
	nsCounts[depth++] = nsCnt;

	nsCnt = 0;
	attrCnt = 0;
	tagName = name;
	elementStarted = true;
	spaceNeeded = false;
    }

    public void  namespace(String prefix, String uri) throws DataModelException {
	if(trace) System.err.println("--- namespace "+prefix+" = "+uri);
	prefixes.addMapping( prefix, uri );
	++ nsCnt;
	if(!elementStarted) {  // stray namespace
	    text(prefix+" = "+uri);
	}
    }

    public void  attribute(QName name, String value) throws DataModelException {
	if(trace) System.err.println("--- attribute "+name+" = "+value);

	if(!elementStarted) {	    // stray attribute
	    text(name+" = "+value);
	    return;
	}
	// check duplicates (no error, just overwrite)
	for(int a = 0; a < attrCnt; a++)
	    if(attrNames[a] == name) {
		attrValues[a] = value;
		return;
	    }
	if(attrNames == null) {
	    System.err.println("***** oh le null");
	    return;
	}
	if(attrCnt >= attrNames.length) {
            QName[] oldn = attrNames;
            attrNames = new QName[ oldn.length * 2 ];
            System.arraycopy(oldn, 0, attrNames, 0, oldn.length);
	    String[] oldv = attrValues;
	    attrValues = new String[ oldv.length * 2 ];
            System.arraycopy(oldv, 0, attrValues, 0, oldv.length);
        }
	attrNames[attrCnt] = name;
	attrValues[attrCnt] = value;
	++ attrCnt;
    }

    public void endElement(QName name) throws DataModelException {
	if(trace) System.err.println("--- end elem "+name);
	//if(elementStarted)
	//    flushElement(true);
	prefixes.removeMappings(nsCnt);
	if(depth == 0)
	    throw new RuntimeException("no open element "+name); //TODO
	nsCnt = nsCounts[ -- depth ];
	spaceNeeded = false;
    }

    public String resolvePrefix( String prefix ) {
	Namespace ns = prefixes.convertToNamespace(prefix);
	return ns == null ? null : ns.getURI();
    }

    /**
     *	Traverse and generate a subtree.  TODO optimize for FONIDoc
     */
    public void traverse( Node node, boolean inScopeNS ) throws DataModelException {

	NodeSequence seq;
	switch( node.getNature() )
	{
	    case Node.DOCUMENT:
		startDocument();
		for(seq = node.children(); seq.nextNode(); )
		    traverse(seq.currentNode(), inScopeNS);
		endDocument();
		break;

	    case Node.ELEMENT:
		startElement( node.getNodeName() );

		// locally defined NS or all:
		if( node.getDefinedNSCount() > 0 || inScopeNS)
		    for( seq = node.namespaces( inScopeNS ); seq.nextNode(); ) {
			Node nsNode = seq.currentNode();
			namespace( nsNode.getNodeName().getLocalName(),
				   nsNode.getStringValue() );
		    }

		for( seq = node.attributes(); seq.nextNode(); ) {
		    Node attr = seq.currentNode();
		    attribute( attr.getNodeName(), attr.getStringValue());
		}

		seq = node.children();
		for( ; seq.nextNode(); )
		    traverse( seq.currentNode(), false );

		endElement( node.getNodeName() );
		break;

	    case Node.TEXT:
		text( node.getStringValue() );
		break;

	    case Node.PROCESSING_INSTRUCTION:
		pi( node.getNodeName().toString(), node.getStringValue() );
		break;

	    case Node.COMMENT:
		comment( node.getStringValue() );
		break;

	    case Node.ATTRIBUTE: 
		//text( node.getNodeName().toString() );
		//text("= "); text(node.getStringValue());
		attribute( node.getNodeName(), node.getStringValue());
		break;

	    case Node.NAMESPACE:
		//text("xmlns:"); text( node.getNodeName().getLocalName() );
		//text("= "); text(node.getStringValue());
		namespace( node.getNodeName().getLocalName(), node.getStringValue());
		break;

	    default:
		System.err.println("illegal node kind "+node.getNature()+" "+node);
	}
    }

}
