/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.dt;

import net.xfra.qizxopen.xquery.Item;
import net.xfra.qizxopen.xquery.Value;
import net.xfra.qizxopen.xquery.XQueryException;

/**
 *	A sequence that stores Integer items in an array.
 */
public class IntegerArraySequence extends GenericValue
{
    protected long[] items;
    protected int size;
    protected int index = -1;

    public IntegerArraySequence( long[] items, int size ) {
        this.items = items;
	this.size = size;
    }

    public IntegerArraySequence( int[] items, int size ) {
	long[] ritems = new long[size];
	for(int i = size; --i >= 0; )
	    ritems[i] = items[i];
        this.items = ritems;
	this.size = size;
    }

    public IntegerArraySequence( short[] items, int size ) {
	long[] ritems = new long[size];
	for(int i = size; --i >= 0; )
	    ritems[i] = items[i];
        this.items = ritems;
	this.size = size;
    }

    public IntegerArraySequence( byte[] items, int size ) {
	long[] ritems = new long[size];
	for(int i = size; --i >= 0; )
	    ritems[i] = items[i];
        this.items = ritems;
	this.size = size;
    }

    public IntegerArraySequence( char[] items, int size ) {
	long[] ritems = new long[size];
	for(int i = size; --i >= 0; )
	    ritems[i] = items[i];
        this.items = ritems;
	this.size = size;
    }

    public boolean next() throws XQueryException {
	if(++ index >= size)
	    return false;
	item = new SingleInteger(items[index]);	// Hmmm
	return true;
    }

    public Value  bornAgain() {
	return new IntegerArraySequence( items, size );
    }

    static long[] unroll(Value value) throws XQueryException {
	long[] items = new long[8];
	int ptr = 1;	// items[0] stores the length
	for(; value.next(); ) {
	    if(ptr >= items.length) {
		long[] old = items;
		items = new long[ old.length * 2 ];
		System.arraycopy(old, 0, items, 0, old.length);
	    }
	    items[ptr ++] = value.asInteger();
        }
	items[0] = ptr - 1;
	return items;
    }

    /**
     *	Converts a Value representing a sequence of integers into an integer array.
     */
    public static long[] expandIntegers(Value value) throws XQueryException {
	long[] items = unroll(value);
	int len = (int) items[0];
	long[] ritems = new long[len];
	System.arraycopy(items, 1, ritems, 0, len);
	return ritems;
    }

    /**
     *	Converts a Value representing a sequence of ints into an int array.
     */
    public static int[] expandInts(Value value) throws XQueryException {
	long[] items = unroll(value);
	int len = (int) items[0];
	int[] ritems = new int[len];
	for(int i = 0; i < len; i++)
	    ritems[i] = (int) items[i + 1];
	return ritems;
    }

    /**
     *	Converts a Value representing a sequence of shorts into an short array.
     */
    public static short[] expandShorts(Value value) throws XQueryException {
	long[] items = unroll(value);
	int len = (int) items[0];
	short[] ritems = new short[len];
	for(int i = 0; i < len; i++)
	    ritems[i] = (short) items[i + 1];
	return ritems;
    }

    /**
     *	Converts a Value representing a sequence of bytes into an byte array.
     */
    public static byte[] expandBytes(Value value) throws XQueryException {
	long[] items = unroll(value);
	int len = (int) items[0];
	byte[] ritems = new byte[len];
	for(int i = 0; i < len; i++)
	    ritems[i] = (byte) items[i + 1];
	return ritems;
    }

    /**
     *	Converts a Value representing a sequence of chars into an char array.
     */
    public static char[] expandChars(Value value) throws XQueryException {
	long[] items = unroll(value);
	int len = (int) items[0];
	char[] ritems = new char[len];
	for(int i = 0; i < len; i++)
	    ritems[i] = (char) items[i + 1];
	return ritems;
    }
} 
