/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.impl;

import net.xfra.qizxopen.dm.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.dm.DocumentTest;
import net.xfra.qizxopen.xquery.op.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.fn.UserFunction;
import net.xfra.qizxopen.xquery.dt.*;
import net.xfra.qizxopen.util.*;

import java.io.*;
import java.math.BigDecimal;
import java.util.HashMap;

/**
 *	Parser for XPath 2.0 / XQuery 1.0 expressions.
 *	
 */
public class Parser extends Lexer
{
    int     curToken = -1;
    int     prevTokenLoc;
    Module  predefinedModule = PredefinedModule.BASE;
    NSPrefixMapping predefinedNS;
    HashMap collations;
    ModuleManager moduleManager;
    Pragma[] pragmas = new Pragma[0];
    boolean  preserveSpace = false;

    public final static String XQUERY_VERSION = "1.0";

    public Parser( ModuleManager man ) {
	moduleManager = man;
    }

    /**
     *	Parses and returns a Query (aka Main Module).
     *	@param input the actual text to parse
     *	@param uri of the query source (for messages), or null if not applicable.
     *	@param log message collector
     *	@return a parsed Query (static analysis is not performed).
     *	@throws SyntaxException on the first lexical or syntactic error (no recovery)
     */
    public Query parseQuery( CharSequence input, String uri, Log log )
	throws XQueryException {
	this.log = log;
	startLexer( input );
	nextToken();
	Query q = new Query();
	currentModule = q; 
	setupModule( input, uri );

	parseOptVersion();
	parseProlog();
	q.body = parseExpr();
	if(curToken != T_SemiColon && curToken != T_END)
	    syntax("unrecognized characters at end of query");
	if(stateSP != 0) System.err.println("*** " + stateSP+" unpopped lexical states");
	currentModule.storePragmas(pragmas);
	return q;
    }

    /**
     *	Parses a library module (used by the Module Manager).
     *  <p>Uses the ModuleManager specified on creation of this parser.
     *  @param module being parsed (created by the calling Module Manager to cope
     *  with cyclical references.)
     *	@param input the actual text to parse
     *	@param uri of the query source (for messages), or null if not applicable.
     *	@param log message collector
     *	@return a parsed Module (static analysis is not performed).
     */
    public Module parseLibraryModule( Module module,
				      CharSequence input, String uri, Log log )
	throws XQueryException 
    {
	this.log = log;
	startLexer( input );
	nextToken();
	currentModule = module;
	setupModule( input, uri );

	parseOptVersion();
	// module namespace <prefix> = <StringLiteral> ;
	wantToken(T_Module);
	if(curToken != T_NCName)
	    syntax("expecting prefix");
	String prefix = makeString();
	wantToken(T_AssignEquals);
	if(curToken != T_URLLiteral) checkToken(T_StringLiteral);
	String moduleURI = makeString();
	
	currentModule.setDeclaredURI( moduleURI );
	currentModule.addNamespaceDecl( prefix, moduleURI );
	pickToken(T_SemiColon);	// optional!

	parseProlog();
	if(curToken != T_END)
	    syntax("unrecognized characters at end of query");
	currentModule.storePragmas(pragmas);
	return currentModule;
    }

    public void setPredefinedModule( Module predefined ) {
	this.predefinedModule = predefined;
    }

    public void setPredefinedNamespaces( NSPrefixMapping predefined ) {
	this.predefinedNS = predefined;
    }

    public void setCollations( HashMap collations) {
	this.collations = collations;
    }

    private void setupModule( CharSequence input, String uri ) {
	currentModule.setPredefinedModule(predefinedModule);
	currentModule.setLog(log);
	currentModule.setSource(input, uri);
	currentModule.setCollations(collations);
	if(predefinedNS != null)
	    for(int p = predefinedNS.getMappingCount(); p >= 1; --p)
		currentModule.addNamespaceDecl( predefinedNS.getLastPrefix(p),
						predefinedNS.getLastNamespace(p).getURI());
		
    }

    //  This code was generated from the XML specification, then manually completed and cleaned.

    private void parseOptVersion() throws XQueryException {
	if(curToken != T_XQueryVersion)
	    return;
	String version = makeString();
	if(version.compareTo(XQUERY_VERSION) > 0)	// mmm
	    log.error( currentModule, prevTokenLoc,
		       "XML Query version %1 not supported, the current version is %2",
		       version, XQUERY_VERSION);
	wantToken(T_SemiColon);	// useless!
    }

    void parseProlog() throws XQueryException {
	// declarations:
        loop: for(;;) {
	    int here = tokenStart;
	    switch(curToken) {
	    case T_DeclareNamespace:
		nextToken(); parseNamespaceDecl();
		break;

            case T_DefaultElement:
	    case T_DefaultFunction:
		parseDefaultNamespaceDecl();
		break;

	    case T_DeclareXMLSpace:
		nextToken(); wantToken(T_XMLSpaceEquals);
		if( pickToken(T_XMLSpacePreserve) )
		    preserveSpace = true;
		else if( pickToken(T_XMLSpaceStrip) )
		    preserveSpace = false;
		else syntax("expecting preserve or strip");
		break;

	    case T_DefaultCollationEquals:
		nextToken(); 
		checkToken(T_URLLiteral);
		currentModule.setDefaultCollation( makeString() );
		break;

	    case T_DeclareVariable:
		nextToken(); 
		checkToken(T_VarName);
		here = tokenStart;
		Type varType = null;
		QName name = resolveVarName();
		if(name.getNamespace() != currentModule.getNamespace() &&
		   !(name.getNamespace() == Namespace.NONE &&
		     currentModule.getNamespace() == Module.LOCAL_NS))
		   log.error( currentModule, prevTokenLoc,
			      "namespace of variable name %1 should match",
			      currentModule.prefixedName(name));

		if(pickToken(T_As))
		    varType = parseSequenceType();
		Expression init = null;
		if(!pickToken(T_External))
		    init = parseEnclosedExpr(); 
		GlobalVariable global = new GlobalVariable( name, varType, init );
		currentModule.addDeclaration( locate(global, here) );
		break;

	    case T_DeclareFunction:
		parseFunctionDefn();
		break;

	    case T_ImportModule:
		nextToken(); 
		String prefix = null, uri, loc = null;
		if(pickToken(T_Namespace)) {
		    checkToken(T_NCName);
		    prefix = makeString();
		    wantToken(T_AssignEquals);
		}
		if(curToken != T_StringLiteral) checkToken(T_URLLiteral);
		uri = makeString();
		if(curToken == T_AtStringLiteral) {
		    loc = makeString();
		}
		moduleImport( here, uri, loc );
		if(prefix != null)
		    currentModule.addNamespaceDecl( prefix, uri );
		break;

	    case T_ImportSchemaToken:
		nextToken(); 
		switch(curToken) {
		    case T_DefaultElement:
		    case T_DefaultFunction:
			parseDefaultNamespaceDecl(); break;
		    case T_Namespace:
			nextToken(); parseNamespaceDecl(); break;
		    case T_StringLiteral:
			nextToken();  break;
		    default:
			syntax("improper schema import");
		}
		if( pickToken(T_AtStringLiteral)) {
		    makeString();
		}
		log.error( currentModule, here, "schema import is not supported");
		break;

	    case T_ValidationStrict:
	    case T_ValidationSkip:
	    case T_ValidationLax:
		log.warning( currentModule, here, "validation directive ignored", "");
		nextToken();
		break;

	    //// EXTENSION: templates similar to XSLT
	    case T_Template:
		parseTemplate();
		break;

	    default:
		break loop;
	    }
	    pickToken(T_SemiColon);	// aug2003 new BS: made optional
        }
    }

    void parseNamespaceDecl( ) throws XQueryException  {
	if(curToken != T_NCName)
	    syntax("expecting prefix");
	String prefix = makeString();
	wantToken(T_AssignEquals);
	if( curToken != T_URLLiteral ) wantToken(T_URLLiteral);
	currentModule.addNamespaceDecl( prefix, makeString() );
    }

    void parseDefaultNamespaceDecl( ) throws XQueryException  {
	boolean forFun = curToken == T_DefaultFunction;
	nextToken();
	wantToken(T_Namespace); wantToken(T_AssignEquals);
	if( curToken != T_URLLiteral && curToken != T_StringLiteral)
	    wantToken(T_URLLiteral);
	currentModule.addDefaultNamespace( forFun, makeString() );
    }

    // SequenceExpr ::= Expr ( Comma Expr )*  
    Expression parseExpr( ) throws XQueryException 
    {
        Expression e = parseExprSingle();
	if(curToken != T_Comma)
	    return e;
	SequenceExpr seq = new SequenceExpr();
	locate(seq);
	seq.addExpr(e);
	while( pickToken(T_Comma) )
	    seq.addExpr( parseExprSingle() );
	return seq;
    }

    Expression parseExprSingle( ) throws XQueryException  {
	switch(curToken) {
	    case T_ForVariable:
	    case T_LetVariable:
		return parseFLWRExpr();
	    case T_Every:
	    case T_Some:
		return parseQuantifiedExpr();
	    case T_TypeswitchLpar:
		return parseTypeswitchExpr();
	    case T_IfLpar:
		return parseIfExpr();
	    default:
		return parseOrExpr();
	}
    }

    // FLWRExpr ::=
    //    ( (ForClause | LetClause)+ WhereClause? OrderByClause?  Return )*  QuantifiedExpr 
    Expression parseFLWRExpr( ) throws XQueryException 
    {
        if( curToken != T_ForVariable && curToken != T_LetVariable )
	    return parseQuantifiedExpr();
	FLWRExpr flower = new FLWRExpr();
	locate(flower);
	while( curToken == T_ForVariable || curToken == T_LetVariable )
	    if( pickToken(T_ForVariable) ) {
		flower.addClause( parseForClause(true) );
		while(pickToken(T_Comma)) {
		    wantToken(T_VariableIndicator);
		    flower.addClause( parseForClause(true) );
		}
	    }
	    else if( pickToken(T_LetVariable) ) {
		flower.addClause( parseLetClause() );
		while(pickToken(T_Comma)) {
		    wantToken(T_VariableIndicator);
		    flower.addClause( parseLetClause() );
		}
	    }
	if( pickToken(T_Where) )
	    flower.where = parseExpr();
	if( pickToken(T_OrderBy) ) {
	    flower.addOrderSpec( parseOrderSpec() );
	    while( pickToken(T_Comma) )
		flower.addOrderSpec( parseOrderSpec() );
	}
	else if( pickToken(T_OrderByStable) ) {
	    flower.stableOrder = true;
	    flower.addOrderSpec( parseOrderSpec() );
	    while( pickToken(T_Comma) )
		flower.addOrderSpec( parseOrderSpec() );
	}
	wantToken(T_Return);
        flower.expr = parseFLWRExpr();
	return flower;
    }

    // used for variable groups in Let 
    LetClause parseLetClause( ) throws XQueryException  {
	int here = prevTokenLoc;
	checkToken(T_VarName);
	LetClause clause = new LetClause( resolveVarName() );
	locate(clause, here);
	if(pickToken(T_As))
	    clause.varType = parseSequenceType();
	wantToken(T_ColonEquals);
	clause.expr = parseExprSingle();
	return clause;
    }

    // used for variable groups in For Some Every 
    ForClause parseForClause( boolean withPos ) throws XQueryException {
	int here = prevTokenLoc;
	checkToken(T_VarName);
	ForClause clause = new ForClause( resolveVarName() );
	locate(clause, here);
	if(pickToken(T_As))
	    clause.varType = parseSequenceType();
	if(withPos && pickToken(T_AtWord)) {
	    wantToken(T_VariableIndicator);
	    clause.position = resolveVarName();
	}
	wantToken(T_In);
	clause.expr = parseExprSingle();
	return clause;
    }

    OrderSpec parseOrderSpec( ) throws XQueryException 
    {
	int here = tokenStart;
        OrderSpec spec = new OrderSpec( parseExprSingle() );
	locate(spec, here);
	if( pickToken(T_Descending) )
	    spec.descending = true;
	else pickToken(T_Ascending);
	if( pickToken(T_EmptyGreatest) )
	    spec.emptyGreatest = true;
	else pickToken(T_EmptyLeast);
	if( pickToken(T_Collation) )
	    spec.collation = makeString();
	return spec;
    }

    // QuantifiedExpr ::=
    //	    ( (Some | Every) VarClause ( Comma VarClause )*  Satisfies )* ExprSingle
    // VarClause ::= VarName ( TypeDeclaration )?  In Expr
    Expression parseQuantifiedExpr( ) throws XQueryException 
    {
        if( curToken != T_Some && curToken != T_Every )
	    return parseTypeswitchExpr();
	QuantifiedExpr q = new QuantifiedExpr(curToken == T_Every);
	locate(q);
	nextToken();
	q.addVarClause( parseForClause(false) );
	while(pickToken(T_Comma)) {
	    wantToken(T_VariableIndicator);
	    q.addVarClause( parseForClause(false) );
	}
	wantToken(T_Satisfies);
        q.cond = parseExprSingle();
	return q;
    }

    // TypeswitchExpr ::= 
    //  (TypeswitchLpar Expr Rpar CaseClause+ Default ( VariableIndicator VarName )?  Return  )*  IfExpr 
    // CaseClause ::= Case (
    Expression parseTypeswitchExpr( ) throws XQueryException 
    {
	int here = tokenStart;
        if(!pickToken(T_TypeswitchLpar))
	    return parseIfExpr();
	TypeswitchExpr sw = new TypeswitchExpr( parseExpr() );
	locate(sw, here);
        wantToken(T_Rpar);
        while( pickToken(T_Case) ) {
	    CaseClause cc = new CaseClause();
	    locate2(cc);
	    sw.addCaseClause(cc);
	    if( pickToken(T_VariableIndicator) ) {
		checkToken(T_VarName);
		cc.variable = resolveVarName();
		wantToken(T_As);
	    }
	    cc.varType = parseSequenceType();
	    wantToken(T_Return);
	    cc.expr = parseExprSingle();
	}
        wantToken(T_Default);
	CaseClause defc = new CaseClause();
	locate2(defc);
	sw.addCaseClause(defc);
	if( pickToken(T_VariableIndicator) ) {
	    checkToken(T_VarName);
	    defc.variable = resolveVarName();
	    wantToken(T_As);
	}
	wantToken(T_Return);
	defc.expr = parseExpr();
	return sw;
    }

    Expression parseIfExpr( ) throws XQueryException 
    {
	int here = tokenStart;
        if(!pickToken(T_IfLpar))
	    return parseInstanceofExpr();
        Expression e1 = parseExpr();
        wantToken(T_Rpar);
        wantToken(T_Then);
        Expression e2 = parseExpr();
        wantToken(T_Else);
	return locate(new IfExpr( e1, e2, parseExprSingle() ), here);
    }

    Expression parseOrExpr( ) throws XQueryException 
    {
        Expression e = parseAndExpr();
	if(curToken != T_Or)
	    return e;
	OrExpr or = new OrExpr(e);
	locate(or);
	while(pickToken(T_Or))
	    or.addExpr(parseAndExpr());
	return or;
    }

    Expression parseAndExpr( ) throws XQueryException 
    {
        Expression e = parseInstanceofExpr();
	if(curToken != T_And)
	    return e;
	AndExpr and = new AndExpr(e);
	locate(and);
	while(pickToken(T_And))
	    and.addExpr(parseInstanceofExpr());
	return and;
    }

    // InstanceofExpr ::= TreatExpr ( Instanceof SequenceType  )?   //CHANGED  
    //
    Expression parseInstanceofExpr( ) throws XQueryException 
    {
        Expression e = parseTreatExpr();
	int here = tokenStart;
	if(pickToken(T_Instanceof)) {
	    e = new InstanceofExpr(e, parseSequenceType());
	    locate(e, here);
	}
        return e;
    }

    // TreatExpr ::= CastableExpr ( <TreatAs> SequenceType )?
    //
    Expression parseTreatExpr( ) throws XQueryException 
    {
        Expression e = parseCastableExpr();
	int here = tokenStart;
	if( pickToken(T_TreatAs) ) {
	    locate(e = new TreatExpr(e, parseSequenceType()), here);
	}
        return e;
    }

    // CastableExpr ::= CastExpr ( <Castable As> SingleType  )*  
    //
    Expression parseCastableExpr( ) throws XQueryException 
    {
        Expression e = parseCastExpr();
	int here = tokenStart;
	if(pickToken(T_CastableAs))
	    locate(e = new CastableExpr(e, parseSingleType()), here);
	return e;
    }

    // CastExpr ::= ComparisonExpr ( <Cast As> SingleType  )*  
    //
    Expression parseCastExpr( ) throws XQueryException 
    {
        Expression e = parseComparisonExpr();
	int here = tokenStart;
	if(pickToken(T_CastAs))
	    locate(e = new CastExpr(e, parseSingleType()), here);
	return e;
    }

    // associativity is not clear here, anyway it does not make sense...
    Expression parseComparisonExpr( ) throws XQueryException 
    {
        Expression e = parseRangeExpr();
      loop:
	for(;;) {
	    int here = tokenStart;
	    switch(curToken) {
	    case T_FortranLt:
		nextToken(); e = new ValueLtOp(e, parseRangeExpr() ); break;
	    case T_FortranEq:
		nextToken(); e = new ValueEqOp(e, parseRangeExpr() ); break;
	    case T_FortranLe:
		nextToken(); e = new ValueLeOp(e, parseRangeExpr() ); break;
	    case T_FortranNe:
		nextToken(); e = new ValueNeOp(e, parseRangeExpr() ); break;
	    case T_FortranGt:
		nextToken(); e = new ValueGtOp(e, parseRangeExpr() ); break;
	    case T_FortranGe:
		nextToken(); e = new ValueGeOp(e, parseRangeExpr() ); break;
	    case T_Gt:
		nextToken(); e = new GtOp(e, parseRangeExpr() ); break;
	    case T_GtEquals:
		nextToken(); e = new GeOp(e, parseRangeExpr() ); break;
	    case T_Lt:
		nextToken(); e = new LtOp(e, parseRangeExpr() ); break;
	    case T_LtEquals:
		nextToken(); e = new LeOp(e, parseRangeExpr() ); break;
	    case T_Equals:
		nextToken(); e = new EqOp(e, parseRangeExpr() ); break;
	    case T_NotEquals:
		nextToken(); e = new NeOp(e, parseRangeExpr() ); break;
	    case T_IsNot:
		nextToken(); e = new IsNotOp(e, parseRangeExpr() ); break;
	    case T_Is:
		nextToken(); e = new IsOp(e, parseRangeExpr() ); break;
	    case T_LtLt:
		nextToken(); e = new BeforeOp(e, parseRangeExpr() ); break;
	    case T_GtGt:
		nextToken(); e = new AfterOp(e, parseRangeExpr() ); break;
	    default: break loop;
	    }
	    locate(e, here);
	}
	return e;
    }

    Expression parseRangeExpr( ) throws XQueryException 
    {
        Expression e = parseAdditiveExpr();
	int here = tokenStart;
	if(pickToken(T_To)) {
	    e = new RangeExpr( e, parseAdditiveExpr() );
	    locate(e, here);
	}
	return e;
     }

    // AdditiveExpr ::= MultiplicativeExpr ( ( Plus | Minus) MultiplicativeExpr )*  
    Expression parseAdditiveExpr( ) throws XQueryException 
    {
        Expression e = parseMultiplicativeExpr();
	for( ;; ) {
	    int here = tokenStart;
	    if( pickToken(T_Plus))
		e = new PlusOp(e, parseMultiplicativeExpr() );
	    else if(pickToken(T_Minus))
		e = new MinusOp(e, parseMultiplicativeExpr() );
	    else
		break;
	    locate(e, here);
	}
	return e;
    }

    // MultiplicativeExpr ::= UnaryExpr ( ( Multiply | Div | Idiv | Mod) UnaryExpr )*  
    Expression parseMultiplicativeExpr( ) throws XQueryException 
    {
        Expression e = parseUnaryExpr();
	for( ;; ) {
	    int here = tokenStart;
	    if( pickToken(T_Multiply))
		e = new MulOp(e, parseUnaryExpr() );
	    else if( pickToken(T_Div) )
		e = new DivOp(e, parseUnaryExpr() );
	    else if( pickToken(T_Idiv) )
		e = new IDivOp(e, parseUnaryExpr() );
	    else if( pickToken(T_Mod) )
		e = new ModOp(e, parseUnaryExpr() );
	    else
		break;
	    locate(e, here);
	}
	return e;
    }

    // UnaryExpr ::= ( ( Minus | Plus) )*  UnionExpr 
    Expression parseUnaryExpr( ) throws XQueryException 
    {
	if( pickToken(T_Minus)) {
	    int here = prevTokenLoc;
	    return locate( new NegateOp(parseUnaryExpr()), here );
	}
	else if( pickToken(T_Plus))
	    return parseUnaryExpr();
	else
	    return parseUnionExpr();
    }

    // UnionExpr ::= IntersectExceptExpr ( ( Union | Vbar) IntersectExceptExpr )*  
    Expression parseUnionExpr( ) throws XQueryException 
    {
        Expression e = parseIntersectExceptExpr();
	while( pickToken(T_Union) || pickToken(T_Vbar) ) {
	    int here = prevTokenLoc;
	    e = new UnionOp(e, parseIntersectExceptExpr() );
	    locate(e, here);
	}
	return e;
    }

    // IntersectExceptExpr ::= ValueExpr ( ( Intersect | Except) ValueExpr )*  
    Expression parseIntersectExceptExpr( ) throws XQueryException 
    {
        Expression e = parseValueExpr();
	for( ;; ) {
	    int here = tokenStart;
	    if( pickToken(T_Intersect))
		e = new IntersectOp(e, parseMultiplicativeExpr() );
	    else if(pickToken(T_Except))
		e = new ExceptOp(e, parseMultiplicativeExpr() );
	    else
		break;
	    locate(e, here);
	}
	return e;
    }

    // ValueExpr ::= ( ValidateExpr | PathExpr) 
    Expression parseValueExpr( ) throws XQueryException 
    {
	if(curToken == T_Validate)
            return parseValidateExpr();
	else
	    return parsePathExpr();
    }


    // PathExpr ::= Root RelativePathExpr? |
    // 		    RootDescendants RelativePathExpr | RelativePathExpr 
    // RelativePathExpr ::= StepExpr ( ( Slash | SlashSlash) StepExpr )*  
    Expression parsePathExpr( ) throws XQueryException 
    {
	int here = tokenStart;
	boolean atRoot = false, atRootDesc = false;
	if( pickToken(T_Root) )
	    atRoot = true;
	else if( pickToken(T_RootDescendants) )
	    atRoot = atRootDesc = true;
	// first expression
	Expression step = parseStepExpr();
	PathExpr p = null;
	// need a PathExpr even with only one step, depending on the kind of step
	if(atRoot || step instanceof ReverseStep) {
	    p = new PathExpr();
	    locate(p, here);
	    if(atRoot)
		p.addStep( locate(new RootStep(null), here) );
	    if( atRootDesc )
		if( step != null )
		    p.addStep( locate(new DescendantOrSelfStep(null), here) );
		else syntax("unterminated path '//'");
	    if(step != null)
		p.addStep(step);
	}
	else if(step == null)
	    syntax("expecting expression"); 
	// if there is no following / or //, finish here:
	if( curToken != T_Slash && curToken != T_SlashSlash )
	    return p != null ? p : step;
	// definitely a path expr:
	if( p == null ) {
	    locate(p = new PathExpr(), here);
	    p.addStep(step);
	}
	while( curToken == T_Slash || curToken == T_SlashSlash ) {
	    if( pickToken(T_SlashSlash) )
		p.addStep( locate(new DescendantOrSelfStep(null), here) );
	    else nextToken();
	    step = parseStepExpr();
	    if( step == null )
		syntax("unterminated path expression");
	    p.addStep(step);
	}
	locate(p, here);
	return p;
    }

    // StepExpr ::= ( ForwardStep | ReverseStep | PrimaryExpr) Predicates 
    Expression parseStepExpr( ) throws XQueryException 
    {
	Expression step = null;
	NodeTest nt;
	int here = tokenStart;
        switch(curToken) {
	    case T_DotDot:
		nextToken(); step = new AncestorStep( null );
		break;
	    case T_AxisAncestor:
		nextToken(); step = new AncestorStep( parseNodeTest() );
		break;
	    case T_AxisAncestorOrSelf:
		nextToken(); step = new AncestorOrSelfStep( parseNodeTest() );
		break;
	    case T_AxisAttribute:
		nextToken(); step = new AttributeStep( parseNodeTest() );
		break;
	    case T_At:
		nextToken(); step = new AttributeStep( parseNameTest(true) );
		break;
	    case T_AxisChild:
		nextToken(); step = new ChildStep( parseNodeTest() );
		break;
	    case T_AxisDescendant:
		nextToken(); step = new DescendantStep( parseNodeTest() );
		break;
	    case T_AxisDescendantOrSelf:
		nextToken(); step = new DescendantOrSelfStep( parseNodeTest() );
		break;
	    case T_AxisFollowing:
		nextToken(); step = new FollowingStep( parseNodeTest() );
		break;
	    case T_AxisFollowingSibling:
		nextToken(); step = new FollowingSiblingStep( parseNodeTest() );
		break;
// 	    case T_AxisNamespace:
// 		nextToken(); step = new NamespaceStep( parseNodeTest() );
// 		break;
	    case T_AxisSelf:
		nextToken(); step = new SelfStep( parseNodeTest() );
		break;
	    case T_AxisParent:
		nextToken(); step = new ParentStep( parseNodeTest() );
		break;
	    case T_AxisPrecedingSibling:
		nextToken(); step = new PrecedingSiblingStep( parseNodeTest() );
		break;
	    case T_AxisPreceding:
		nextToken(); step = new PrecedingStep( parseNodeTest() );
		break;
	    case T_Dot:
		nextToken(); step = new SelfStep( null );
		break;
	    case T_AttributeLpar:
		step = new AttributeStep( parseNodeTest(true) );
		break;
	    case T_DocumentNodeLpar:
	    case T_ElementLpar:
	    case T_CommentLpar:
	    case T_TextLpar:
	    case T_NodeLpar:
	    case T_ProcessingInstructionLpar:
	    case T_NCNameColonStar:
	    case T_QName:
	    case T_Star:
	    case T_StarColonNCName:
		step = new ChildStep( parseNodeTest() );
		break;
	    case T_Lbrack:
		syntax("unexpected '['");
	    default:
		step = parsePrimaryExpr();
		break;
        }
	if(step != null)
	    locate(step, here);
	// predicates:
	FilterExpr filtered = (step instanceof FilterExpr)? (FilterExpr) step : null;
	while( pickToken(T_Lbrack) ) {
	    here = prevTokenLoc;
	    if(filtered == null)
		filtered = new FilterExpr( step );
	    filtered.addPredicate(parseExpr());
	    locate(step, here);
	    wantToken(T_Rbrack);
	}
	return filtered == null ? step : filtered;
    }

    // ValidateExpr ::= 'validate' SchemaMode ? SchemaContext? '{' Expr '}'
    //
    Expression parseValidateExpr( ) throws XQueryException //TODO
    {
	int here = tokenStart;
	wantToken(T_Validate);
	int mode = -1;
	if( pickKeyword("lax") )
	    mode = ValidateExpr.LAX_MODE;
	else if( pickKeyword("strict") )
	    mode = ValidateExpr.STRICT_MODE;
	else if( pickKeyword("skip") )
	    mode = ValidateExpr.SKIP_MODE;
	Expression sc = null;
	if( !pickToken(T_LbraceExprEnclosure) ) {
	    sc = parseSchemaContext();
	    wantToken(T_LbraceExprEnclosure);
	}
	Expression e = parseExpr();
	wantToken(T_Rbrace);
	e = new ValidateExpr(mode, sc, e);
	locate(e, here);
	return e;
    }


    // PrimaryExpr ::= ( Literal | FunctionCall | VariableIndicator VarName  | ParenthesizedExpr) 
    //
    Expression parsePrimaryExpr( ) throws XQueryException 
    {
        switch(curToken) {
	    case T_DecimalLiteral:
		return locate2( new DecimalLiteral(makeDecimal()) );
	    case T_DoubleLiteral:
		return locate2( new DoubleLiteral(makeNumber()) );
	    case T_IntegerLiteral:
		return locate2( new IntegerLiteral(makeInteger()) );
	    case T_StringLiteral:
		return locate2( makeStringLiteral() );
	    case T_QNameLpar:
		return parseFunctionCall();
	    case T_VariableIndicator:
		nextToken(); checkToken(T_VarName);
		return locate2(new VarReference( resolveVarName()));
	    case T_Lpar:
		return parseParenthesizedExpr();
	    case T_StartTagOpenRoot:
	    case T_StartTagOpen:
	    case T_XmlComment:
	    case T_ProcessingInstruction:
	    case T_CdataSectionStart:

	    case T_DocumentLbrace:
	    case T_ElementQNameLbrace:
	    case T_ElementLbrace:
	    case T_AttributeQNameLbrace:
	    case T_AttributeLbrace:
	    case T_TextLbrace:
	    case T_CommentLbrace:
	    case T_NamespaceLbrace:
	    case T_PILbrace:
	    case T_PINameLbrace:
		return parseConstructor( );
	    default:	// dont protest immediately
		return null;
        }
    }

    Expression parseParenthesizedExpr( ) throws XQueryException 
    {
        wantToken(T_Lpar);
	if( pickToken(T_Rpar) )
	    return new SequenceExpr();
        Expression e = parseExpr();
        wantToken(T_Rpar);
	return e;
    }

    Expression parseFunctionCall( ) throws XQueryException 
    {
	int here = tokenStart;
	checkToken(T_QNameLpar);
	QName fname = resolveQName( currentModule.getDefaultFunctionNS() );
	FunctionCall call = new FunctionCall(fname);
	locate(call, here);
	if(!pickToken(T_Rpar)) {
	    call.addArgument( parseExprSingle() );
	    while( pickToken(T_Comma) )
		call.addArgument( parseExprSingle() );
	    wantToken(T_Rpar);
        }
	return call;
    }

    ItemType checkTypeName( QName name ) {
	ItemType itype = currentModule.lookForType(name);
	if(itype == null) {
	    log.error( currentModule, prevTokenLoc,
		       "unknown type %1", currentModule.prefixedName(name));
	    return Type.ITEM;
	}
	else if(!Type.ATOM.accepts(itype))
	    log.error( currentModule, prevTokenLoc,
		       "non atomic type %1", currentModule.prefixedName(name));
	return itype;
    }

    // SingleType ::= AtomicType ( QMark )?  
    Type parseSingleType( ) throws XQueryException
    {
	checkToken(T_QName);
	ItemType itype = checkTypeName( resolveElementName() );
	return pickToken(T_QMark) ? (Type) itype.opt : itype;
    }

    // SequenceType ::=  ItemType OccurrenceIndicator  |  Empty 
    Type parseSequenceType( ) throws XQueryException 
    {
	if(pickToken(T_Empty))
	   return Type.NONE;
	ItemType itemType = parseItemType();
	if(pickToken(T_Star) || pickToken(T_Multiply))
	    return itemType.star;
	else if(pickToken(T_Plus))
	    return new SequenceType(itemType, Type.MULTI_OCC);
	else if(pickToken(T_QMark))
	    return new SequenceType(itemType, Type.OPT_OCC);
	return itemType;
    }

    // ItemType ::= ( ( ElementType | AttributeType) ( ElemOrAttrType )?  
    //		  | Node | ProcessingInstruction | Comment | Text | Document
    //		  | Item | AtomicType | Untyped | AtomicValue
    //
    ItemType parseItemType( ) throws XQueryException 
    {
        switch(curToken) {
	    case T_DocumentNodeLpar:
	    case T_ElementLpar:
	    case T_TextLpar:
	    case T_ProcessingInstructionLpar:
	    case T_CommentLpar:
	    case T_NodeLpar:
		return new NodeType(parseNodeTest());
	    case T_AttributeLpar:
		return new NodeType(parseNodeTest(true));
	    case T_QName:
		ItemType itype = checkTypeName( resolveElementName() );
		// wish it were "default type NS" resolveQName(Namespace.XSD)
		return itype;
	    case T_ItemLparRpar:
		nextToken(); return Type.ITEM;
 	    default:
		syntax("expecting item type"); return null;
        }
    }

    NodeTest parseNodeTest( ) throws XQueryException {
	return parseNodeTest(false);
    }

    NodeTest parseNodeTest( boolean forAttr ) throws XQueryException 
    {
	int here = tokenStart;
        switch(curToken) {
	    case T_DocumentNodeLpar:
		nextToken();
		NodeTest nt = curToken == T_Rpar ? null : parseElemAttrTest(false);
		wantToken(T_Rpar);
		if(nt == null)
		    nt = new BaseNodeTest( Node.ELEMENT, null, null );
		return locate(here, new DocumentTest(nt));
	    case T_ElementLpar:
		return locate(here, parseElemAttrTest(false));
	    case T_AttributeLpar:
		return locate(here, parseElemAttrTest(true));
	    case T_TextLpar:
		nextToken(); wantToken(T_Rpar);
		return locate(here, new BaseNodeTest( Node.TEXT, null, null ) );
	    case T_ProcessingInstructionLpar:
		nextToken(); 
		String target = null;
		if(curToken == T_StringLiteral)
		    target = makeString();
		else if(curToken == T_QName)
		    target = makeString();	// ignore NS!!
		wantToken(T_Rpar);
		return locate( here, new BaseNodeTest( Node.PROCESSING_INSTRUCTION,
						       null, target));
	    case T_CommentLpar:
		nextToken(); wantToken(T_Rpar);
		return locate(here, new BaseNodeTest( Node.COMMENT, null, null ));
	    case T_NodeLpar:
		nextToken(); wantToken(T_Rpar);
		return null;
	    default:
		return parseNameTest(forAttr);
        }
    }

    NodeTest parseElemAttrTest( boolean forAttr ) throws XQueryException 
    {
	wantToken(forAttr ? T_AttributeLpar : T_ElementLpar);
	Namespace ns = null;
	String name = null;
	SchemaContext path = null;
	if( curToken != T_Rpar && !pickToken(T_Star) ) {
	    path = parseSchemaContextPath();
	    if(path.isSimpleName()) {
		QName qname = path.getStep(0); path = null;
		ns = qname.getNamespace();
		name = qname.getLocalName();
	    }
	}
	if(path == null && pickToken(T_Comma)) {
	    QName typeName = parseStarName();
	    locate2(path = new SchemaContext(true));
	    path.addStep(typeName);
	}
	wantToken(T_Rpar);
	if(path != null)
	    log.error(path.module, path.location, "Schema Type test not supported");
	return new BaseNodeTest( forAttr ? Node.ATTRIBUTE : Node.ELEMENT, ns, name );
    }


    QName parseStarName() throws XQueryException  {
	if(curToken == T_QName)
	    return resolveElementName();
	else if(!pickToken(T_Star))
	    syntax("expecting name or '*'");
	return null;
    }

    NodeTest parseNameTest( boolean forAttr ) throws XQueryException 
    {
	int here = tokenStart;
	Namespace dns = forAttr? Namespace.NONE : currentModule.getDefaultElementNS();
	int kind = forAttr? Node.ATTRIBUTE : Node.ELEMENT;
        switch(curToken) {
	    case T_QName:
		return locate(here, new BaseNodeTest( kind,
					 expandPrefix(prefixValue, dns), makeString() ));
	    case T_StarColonNCName:
		return locate(here, new BaseNodeTest( kind, null, makeString() ));
	    case T_Star:
		nextToken(); return locate(here, new BaseNodeTest( kind, null, null ));
	    case T_NCNameColonStar:
		return locate(here,
			      new BaseNodeTest(kind, expandPrefix(makeString(), dns), null));
	    default:
		syntax("expecting name test");
		return null;
        }
    }

    // SchemaContext ::= 'global' | 'context' SchemaContextLocation 
    SchemaContext parseSchemaContext( ) throws XQueryException 
    {
	if(pickKeyword("global"))
	    return null;
	if(pickKeyword("context"))
	    return parseSchemaContextPath();	// TODO reject simple name
	syntax( "expecting schema context");
	return null;
    }

    // SchemaContextPath ::= ( QName | TypeQName) ( Slash QName )*  
    // This is different than the standard, because the grammar is largely ambiguous
    // this can accept a simple QName, or no final '/', which is not OK for a SchemaContextPath:
    // this is tested afterwards by SchemaContext.length() and SchemaContext.endsWithSlash
    SchemaContext parseSchemaContextPath( ) throws XQueryException 
    {
	SchemaContext sc = new SchemaContext( curToken == T_TypeQName );
	if( curToken != T_TypeQName && curToken != T_QName)
	    wantToken(T_QName);
	sc.addStep( resolveElementName() );
        while( pickToken(T_Slash) )
	    if(curToken == T_QName)
		sc.addStep( resolveElementName() );
	    else {
		sc.endsWithSlash = true;
		break;
	    }
	return sc;
    }

    Expression parseConstructor( ) throws XQueryException 
    {
	int here = tokenStart;
	Expression e = null;
	NamedConstructor nc;

        switch(curToken) {
	

        case T_StartTagOpenRoot:
	case T_StartTagOpen:
            return parseElementConstructor();
        case T_XmlComment:
            return locate(new AtomConstructor( Node.COMMENT, makeStringLiteral() ), here);
        case T_ProcessingInstruction:
            return locate(new PIConstructor( makeString() ), here);
        case T_TextLbrace:
	    nextToken();
	    if(!pickToken(T_Rbrace)) {
		e = parseExpr();
		wantToken(T_Rbrace);
	    }
	    return new AtomConstructor(Node.TEXT, e);

	

        case T_DocumentLbrace:
	    nextToken();
            e = new DocumentConstructor(parseExpr());
	    wantToken(T_Rbrace);
            return e;

        case T_ElementLbrace:
        case T_AttributeLbrace:
	    nc = curToken == T_ElementLbrace ? (NamedConstructor) new ElementConstructor(null)
		: new AttributeConstructor(null);
	    nextToken();
	    nc.name = parseExpr();
	    wantToken(T_Rbrace);
	    wantToken(T_LbraceExprEnclosure);
	    if(!pickToken(T_Rbrace)) {
		nc.addItem( parseExpr() );
		wantToken(T_Rbrace);
	    }
	    return locate(nc, here);

        case T_ElementQNameLbrace:
        case T_AttributeQNameLbrace:
	    int prevToken = curToken;
	    Expression qname = new QNameLiteral(resolveElementName());
	    nc = prevToken == T_ElementQNameLbrace?
		(NamedConstructor) new ElementConstructor(qname) : new AttributeConstructor(qname);
	    if(!pickToken(T_Rbrace)) {
		nc.addItem( parseExpr() );
		wantToken(T_Rbrace);
	    }
	    return locate(nc, here);

	case T_CommentLbrace:
	    nextToken(); e = parseExpr(); wantToken(T_Rbrace);
            return locate(new AtomConstructor( Node.COMMENT, e ), here);
	    
	case T_NamespaceLbrace:
	    String prefix = makeString();
	    e = parseExpr(); wantToken(T_Rbrace);
	    return locate( new NamespaceConstructor( new StringLiteral(prefix), e ), here);

	case T_PILbrace:
	    nextToken();
	    Expression name = parseExpr();
	    wantToken(T_Rbrace);  wantToken(T_LbraceExprEnclosure);
	    if(!pickToken(T_Rbrace)) {
		e = parseExpr();
		wantToken(T_Rbrace);
	    }
	    return locate( new PIConstructor( name, e), here);

	case T_PINameLbrace:
	    String target = makeString();
	    e = parseExpr(); wantToken(T_Rbrace);
	    return locate( new PIConstructor( new StringLiteral(target), e ), here);

	default:
	    syntax("illegal constructor"); return null;
        }
    }


    // ElementConstructor ::= ( StartTagOpenRoot | StartTagOpen) TagQName AttributeList 
    //     ( EmptyTagClose |
    //	     StartTagClose ElementContent*  EndTagOpen TagQName S?  EndTagClose ) 
    //
    ElementConstructor parseElementConstructor( ) throws XQueryException 
    {
	int here = tokenStart, nsCnt = 0;
	if(!pickToken(T_StartTagOpenRoot))
	    wantToken(T_StartTagOpen);
	checkToken(T_TagQName);
	String tagPrefix = prefixValue, tagName = makeString();
	ElementConstructor c = new ElementConstructor( null );
	locate(c, here);
	pickToken(T_S);
	while(curToken == T_TagQName) {
	    AttributeConstructor ac = new AttributeConstructor( null );
	    ac.prefix = prefixValue;	// temporary
	    ac.value = makeString();
	    locate2(ac);
	    pickToken(T_S);
	    wantToken(T_ValueIndicator);
	    pickToken(T_S);
	    parseAttributeValue( ac );

	    // detect and process xmlns attributes
	    String nsdecl = null;
	    if(ac.prefix.equals("xmlns"))
		nsdecl = ac.value;
	    else if(ac.prefix.length() == 0 && ac.value.equals("xmlns"))
		nsdecl = "";
	    if(nsdecl == null)
		c.addAttribute(ac);
	    else {
		//new ExprDump().display("attr value", ac.contents);
		if(ac.contents.length != 1 || !(ac.contents[0] instanceof StringLiteral))
		    log.error( ac.module, ac.location,
			       "namespace attribute must have a literal value");
		else {
		    String nsuri = ((StringLiteral) ac.contents[0]).value;
		    currentModule.getInScopeNS().addMapping(nsdecl, nsuri);
		    ++ nsCnt;
		    c.addAttribute( new NamespaceConstructor(nsdecl, nsuri) );
		}
	    }
	}
	// now the qnames can be resolved
	Namespace ens = expandPrefix(tagPrefix);
	QName stag = QName.get(ens == null ? Namespace.NONE : ens, tagName);
	c.name = new QNameLiteral(stag);
	locate(c.name, here);

	for(int a = 0, AN = c.attributes.size(); a < AN; a++) {
	    AttributeConstructor ac = c.getAttribute(a);
	    if(ac instanceof NamespaceConstructor)
		continue;
	    // no default NS for attributes:
	    Namespace ns = ac.prefix.length() == 0 ? Namespace.NONE : expandPrefix(ac.prefix);
	    ac.name = new QNameLiteral(QName.get(ns, ac.value));
	    ac.prefix = ac.value = null;	// recover memory...
	}

	if( pickToken(T_EmptyTagClose) ) {
	    currentModule.getInScopeNS().removeMappings(nsCnt);
	    return c;
	}
	wantToken(T_StartTagClose);
	StringBuffer textBuf = new StringBuffer();
	// parse contents:
    loop: 
	for( ;; ) {
	    textBuf.setLength(0);
	    here = tokenStart;
	    boolean keepAnyway = false;
	    while( curToken == T_Char || curToken == T_CharRef ) {
		textBuf.append( saveBuffer );
		if( curToken == T_CharRef )
		    keepAnyway = true;
		else if(!keepAnyway && !preserveSpace)
		    for(int ic = saveBuffer.length(); -- ic >= 0; )
			if(!Character.isWhitespace(saveBuffer.charAt(ic))) {
			    keepAnyway = true;
			    break;
			}
		nextToken();
	    }
	    if(textBuf.length() > 0)
		if( preserveSpace || keepAnyway )
		    locate( c.addTextItem(textBuf.toString()), here );
	    switch(curToken) {
	    case T_Lbrace:
	    case T_LbraceExprEnclosure:
		c.addItem( parseEnclosedExpr() );
		break;
	    case T_StartTagOpen:
		c.addItem( parseElementConstructor() );
		break;
	    case T_XmlComment:
		c.addItem( locate2( new AtomConstructor( Node.COMMENT, makeStringLiteral() )) );
		break;
	    case T_ProcessingInstruction:
		c.addItem( locate2( new PIConstructor( makeString() )));
		break;
	    case T_EndTagOpen:
	    case T_END:
		break loop;
	    }
	}
	wantToken(T_EndTagOpen);
	checkToken(T_TagQName);
	tagPrefix = prefixValue; tagName = makeString();
	ens = expandPrefix(tagPrefix);
	QName etag = QName.get(ens == null ? Namespace.NONE : ens, tagName);

	if(etag != stag)
	    syntax("tag mismatch: "+etag+" encountered when expecting "+stag);
        pickToken(T_S);
        wantToken(T_EndTagClose);
	currentModule.getInScopeNS().removeMappings(nsCnt);
	return c;
    }

    // AttributeValue ::=  OpenQuot ( ( EscapeQuot | AttributeValueContent) )*  CloseQuot 
    //			 | OpenApos ( ( EscapeApos | AttributeValueContent) )*  CloseApos 
    void parseAttributeValue( AttributeConstructor c ) throws XQueryException 
    {
	int delimiter = curToken == T_OpenQuot ? T_CloseQuot : T_CloseApos;
	if(!pickToken(T_OpenQuot) && !pickToken(T_OpenApos))
	    syntax("bad attribute delimiter");
	for( ;; )
	    if( curToken == T_Char || curToken == T_CharRef ) {
		locate2( c.addTextItem( makeString() ));
	    }
	    else if( curToken == T_Lbrace || curToken == T_LbraceExprEnclosure )
		c.addItem( parseEnclosedExpr() );
	    else
		break;
	wantToken(delimiter);
	if(c.contents.length == 0)
	    c.addTextItem("");
    }

    Expression parseEnclosedExpr( ) throws XQueryException 
    {
        if(!pickToken(T_LbraceExprEnclosure))
	    wantToken(T_Lbrace);
        Expression e = parseExpr();
        wantToken(T_Rbrace);
	return e;
    }

    // FunctionDefn ::=
    //   DeclareFunction QNameLpar ParamList? (Rpar | RparAs SequenceType) EnclosedExpr
    //
    UserFunction parseFunctionDefn( ) throws XQueryException 
    {
        wantToken(T_DeclareFunction);
	checkToken(T_QNameLpar);
	QName name = resolveQName( null );
	if(name.getNamespace() != currentModule.getNamespace()) {
	    boolean loc = (currentModule.getNamespace() == Module.LOCAL_NS);
	    log.error( currentModule, prevTokenLoc,
		       loc ? "function %1 should be declared in 'local' namespace"
		       : "namespace of function name %1 does not match module namespace",
		       currentModule.prefixedName(name));
	}
	UserFunction fun = (UserFunction) currentModule.localFunctionLookup(name);
	if( fun == null ) {
	    fun = new UserFunction(currentModule, prevTokenLoc);
	    currentModule.addDeclaration( fun );
	}
	else if(false)	// OBSOLETE ?or add 'pedantic' option?
	    log.error( currentModule, prevTokenLoc,
		       "function %1 is already defined in this module",
		       currentModule.prefixedName(name));
	UserFunction.Signature proto = fun.addPrototype(name);
	// the function is immediately visible (forward references accepted):	
	// NB: done here because we need the name in proto. Doesnt harm to be put twice in map.
	currentModule.declareFunction( fun );

	if( curToken != T_Rpar && curToken != T_RparAs ) {
	    wantToken(T_VariableIndicator);
	    checkToken(T_VarName);
	    QName argName = resolveVarName();
	    proto.arg( argName, pickToken(T_As) ? parseSequenceType() : Type.ANY );
	    while( pickToken(T_Comma) ) {
		wantToken(T_VariableIndicator);
		checkToken(T_VarName);
		argName = resolveVarName();
		proto.arg( argName, pickToken(T_As) ? parseSequenceType() : Type.ANY );
	    }
	}
	if(pickToken(T_RparAs))
	    proto.returnType = parseSequenceType();
	else wantToken(T_Rpar);
	Expression body = null;
	if(!pickToken(T_External))
	    body = parseEnclosedExpr();
	proto.body = body;
	return fun;
    }

    //EXTENSION
    //	template := 'template' named_template? key* '{' expr '}'
    //  named_template := qname '(' (parameter (',' parameter)* )? ')'
    //  parameter := '$' VariableName ('as' sequenceType)? ( ':=' defaultValue )
    //  key := name '=' Expr
    //		match = pattern
    //
    void parseTemplate() throws XQueryException  {
	int here = tokenStart;
	nextToken();
	

    }

    /*
        
         for group $grp, [$grp-key ] in Expr <grouping> order by ... return <expr>

	 grouping := 'by' <key> | 'adjacent' | 'first' pattern | 'last' pattern

    */


    // ------ utilities: --------------------------------------------------

    private void moduleImport( int location, String logicalURI, String physicalURI ) {
	try {
	    Module imported = moduleManager.loadModule( predefinedModule, 
							logicalURI, physicalURI, log );
	    if(! logicalURI.equals( imported.getDeclaredURI() ))
		log.error( currentModule, location,
			   "module imported from '%1' declares a different URI (%2)",
			   logicalURI, imported.getDeclaredURI());
	    currentModule.addDeclaration(imported);
	}
	catch (Exception e) {
	    log.error( currentModule, location,
		       "module %1 cannot be imported: %2", logicalURI,
		       e.getMessage() );
	}
    }

    private Expression makeStringLiteral() throws XQueryException {
	return locate2(new StringLiteral( makeString() ));
    }

    private Namespace expandPrefix( String prefix ) {
	Namespace ns = currentModule.getInScopeNS().convertToNamespace( prefix );
	if( ns == null && prefix.length() > 0 ) {
	    log.error(currentModule, tokenStart, "unknown prefix '%1'", prefix);
	    ns = Namespace.NONE;
	}
	return ns;
    }

    private Namespace expandPrefix( String prefix, Namespace defaultNS ) {
	if(prefix == null || prefix.length() == 0)
	    return defaultNS;
	return expandPrefix(prefix);
    }

    // assuming that the current token is QName, return the QName after moving to next token.
    private QName resolveQName( Namespace defaultNS ) throws XQueryException {
	if(defaultNS == null)
	    defaultNS = Namespace.NONE;
	return QName.get( expandPrefix(prefixValue, defaultNS), makeString());
    }

    private QName resolveElementName() throws XQueryException {
	return resolveQName( currentModule.getDefaultElementNS() );
    }

    private QName resolveVarName() throws XQueryException {
	// normally blank NS, should be module NS
	return resolveQName( null );
	//return resolveQName( currentModule.getNamespace() );
    }

    void parsedExtension(int location, String prefix, String localName, String body) {
	log.error( currentModule, location, "extension not supported: %1:%2 %3",
		   new String[]{ prefix, localName, body });
    }

    void parsedPragma(int location, String prefix, String localName, String body) {
	Pragma[] old = pragmas;
	pragmas = new Pragma[old.length+1];
	System.arraycopy(old, 0, pragmas, 0, old.length);
	pragmas[old.length] = new Pragma(QName.get(prefix+':'+localName), body);
    }

    private String makeString( int asToken ) throws XQueryException {
	if(curToken != asToken) wantToken(asToken);
	return makeString();
    }

    private String makeString() throws XQueryException {
	String v = saveBuffer.toString();
	nextToken();
	return v;
    }

    private BigDecimal makeDecimal() throws XQueryException {
	BigDecimal v;
	try {
	    v = Conversion.toDecimal(saveBuffer.toString());
	}
	catch( TypeException e ) {
	    log.error( currentModule, tokenStart, "invalid value of decimal literal '%1'",
		       saveBuffer.toString());
	    v = new BigDecimal(0);
	}
	nextToken();
	return v;
    }

    private double makeNumber() throws XQueryException {
	double v = 0;
	try {
	    v = Double.parseDouble(saveBuffer.toString());
	}
	catch( java.lang.NumberFormatException e) {
	    log.error( currentModule, tokenStart, "invalid value of double literal '%1'",
		       saveBuffer.toString());
	}
	nextToken();
	return v;
    }

    private long makeInteger() throws XQueryException {
	long v = 0;
	try {
	    v = Conversion.toInteger(saveBuffer.toString());
	}
	catch( TypeException e) {
	    log.error( currentModule, tokenStart, "value of integer literal '%1' out of bounds",
		       saveBuffer.toString());
	}
	nextToken();
	return v;
    }

    boolean pickKeyword( String kw ) throws XQueryException {
	if(curToken != T_QName || prefixValue.length() > 0 || !kw.equals(saveBuffer.toString()) )
	    return false;
	nextToken();
	return true;
    }

    boolean pickToken( int token ) throws XQueryException {
	if(curToken != token)
	    return false;
	nextToken();
	return true;
    }

    void wantToken( int token ) throws XQueryException {
	checkToken(token);
	nextToken();
    }

    void checkToken( int token ) throws XQueryException {
	if(curToken != token)
	    syntax( "expecting "+ tokenName(token) );
    }

    void nextToken() throws XQueryException {
	prevTokenLoc = tokenStart;
	curToken = getToken();
    }

    Expression locate(Expression e, int where) {
	e.location = where;
	e.module = currentModule;
	return e;
    }
 
    Expression locate(Expression e) {
	return locate(e, tokenStart);
    }

    Expression locate2(Expression e) {
	return locate(e, prevTokenLoc);
    }

    NodeTest locate(int where, NodeTest nt) {
	////nt.location = where;		//TODO
	////nt.module = currentModule;
	return nt;
    }

    String tokenName(int token) {
	String tname = tokenNames[token];
	return (tname.length() > 1 && tname.charAt(0) == '<') ?  tname  :  '\'' +tname+ '\'';
    }

    void syntax( String message ) throws XQueryException {
	log.error( currentModule, tokenStart, "syntax error, near %1 : %2",
		   tokenName(curToken), message);
	throw new SyntaxException(message, tokenStart);
    }

} // end of class XQueryParser
