/*
 *	Qizx/Open version 0.3
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.xquery.op;

import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.dt.SingleDecimal;
import net.xfra.qizxopen.xquery.dt.Conversion;
import net.xfra.qizxopen.xquery.fn.Prototype;

/**
 *   Implementation of operator '*'. 
 * 
 */
public class MulOp extends NumericOp {

    public MulOp( Expression expr1, Expression expr2 ) {
        super( expr1, expr2 );
    }

    static Prototype[] protos = { 
        Prototype.op("*", Type.DOUBLE, ExecD.class).hidden()
	  .arg("op1", Type.DOUBLE).arg("op2", Type.UNTYPED_ATOMIC),
        Prototype.op("*", Type.DOUBLE, ExecD.class).hidden()
	  .arg("op1", Type.UNTYPED_ATOMIC).arg("op2", Type.DOUBLE),
        Prototype.op("*", Type.INTEGER, ExecI.class)
	  .arg("op1", Type.INTEGER).arg("op2", Type.INTEGER),
        Prototype.op("*", Type.DECIMAL, ExecDec.class)
	  .arg("op1", Type.DECIMAL).arg("op2", Type.DECIMAL),
        Prototype.op("*", Type.FLOAT, ExecF.class)
	  .arg("op1", Type.FLOAT).arg("op2", Type.FLOAT),
        Prototype.op("*", Type.DOUBLE, ExecD.class)
	  .arg("op1", Type.DOUBLE).arg("op2", Type.DOUBLE),
    };
    public Prototype[] getProtos() { return protos; }

    public void dump( ExprDump d ) {
	d.header( this, "Op *" );
        d.display("expr1", operands[0]);
        d.display("expr2", operands[1]);
    }

    public static class ExecI extends Function.IntegerCall {

	public long evalAsInteger( Focus focus, EvalContext context ) throws XQueryException {
	    long e1 = args[0].evalAsInteger(focus, context);
	    long e2 = args[1].evalAsInteger(focus, context);
	    context.at(this);
	    if(!Conversion.isIntegerRange( e1 * (double) e2 ))
		context.error(this, "integer overflow");
	    return e1 * e2;
	}
    }

    public static class ExecDec extends Function.Call {

	public Value eval( Focus focus, EvalContext context ) throws XQueryException {
	    Value e1 = args[0].eval(focus, context);
	    Value e2 = args[1].eval(focus, context);
	    context.at(this);
	    return new SingleDecimal( e1.asDecimal().multiply( e2.asDecimal() ));
	}
    }

    public static class ExecF extends Function.FloatCall {

	public float evalAsFloat( Focus focus, EvalContext context ) throws XQueryException {
	    float e1 = args[0].evalAsFloat(focus, context);
	    float e2 = args[1].evalAsFloat(focus, context);
	    context.at(this);
	    
	    return e1 * e2;
	}
    }

    public static class ExecD extends Function.DoubleCall {

	public double evalAsDouble( Focus focus, EvalContext context ) throws XQueryException {
	    double e1 = args[0].evalAsDouble(focus, context);
	    double e2 = args[1].evalAsDouble(focus, context);
	    context.at(this);
	    
	    return e1 * e2;
	}
    }
}
