/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */

package net.xfra.qizxopen.dm;

import net.xfra.qizxopen.util.Util;

import java.io.InputStream;
import java.io.File;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.Vector;
import java.net.URL;
import org.xml.sax.InputSource;
import org.xml.sax.EntityResolver;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import javax.xml.parsers.*;

/**
 *  Provides access to the data-model of XML documents.
 *  <p>Resolves the URI of a document and parses it or opens it according to its
 *  representation.
 *  <p>Parsed documents are managed in a cache with configurable size
 *  (see {@link #setCacheSize}).
 *  <p>Supports XML catalogs through
 *  Norman Walsh's <code>com.sun.resolver.tools.CatalogResolver</code>.
 */
public class DocumentManager extends DocumentParser
{
    private static final int MIN_CACHE_SIZE = 128 * 1024;
    private static final String CACHE_SIZE_PROP =
	"net.xfra.qizxopen.docman.cachesize";
    // 8Mb default, configurable by system prop:
    private long cacheSize = 8 * 1048576;
    private Vector cache = new Vector();

    protected URL baseURL;

    /**
     *	Creation with a single base location.
     *	@param baseURL the base URL used for resolving a relative document URI.
     *	For parsed documents, any Java-supported URL is suitable. 
     */
    public DocumentManager( URL baseURL ) {
	init(baseURL);
    }

    /**
     *	Creation with a single base location.
     */
    public DocumentManager( String baseURI ) throws IOException {
	init( Util.uriToURL(baseURI) );
    }

    /**
     *	Returns the current baseURI.
     */
    public String getBaseURI() {
	return baseURL.toString();
    }

    /**
     *	Defines the maximal memory size of the document cache. This size is
     *	otherwise defined by the system property
     *	"net.xfra.qizxopen.docman.cachesize".
     *	@param size in bytes (hint).
     */
    public synchronized void setCacheSize( long size ) {
	cacheSize = Math.max( size, MIN_CACHE_SIZE );
    }

    /**
     *	Overridable method for resolving a document URI to an actual URL.
     */
    protected URL resolveLocation( String uri ) throws IOException {
	return new URL(baseURL, uri);
    }

    private void init( URL baseURL ) {
	super.init();
	this.baseURL = baseURL;
	String sysp = System.getProperty(CACHE_SIZE_PROP);
	if(sysp != null) {
	    try {
		setCacheSize( Long.parseLong(sysp) );
	    } catch(Exception ignored) { }
	}
    }

    /**
     *	Cached access by URI.
     *	@return the root node (or document node) of the document.
     */
    public Node findDocumentNode( String uri ) throws DataModelException {
	FONIDocument doc = findDocument( uri );
	FONIDM dm = new FONIDM(doc);
	return dm.documentNode();
    }

    /**
     *	Cached access by URI.
     */
    public FONIDocument findDocument( String uri ) throws DataModelException {
	try {
	    URL rloc = resolveLocation(uri);
	    FONIDocument dm = getCachedDocument(rloc.toString());
	    if(dm == null)
		try {
		    dm = parseDocument(rloc);
		}
		catch (org.xml.sax.SAXException sax) {
		    if(false) {
			sax.printStackTrace();
			if(sax.getException() != null) {
			    System.err.println("caused by: ");
			    sax.getException().printStackTrace();
			}
		    }
		    throw new DataModelException( "XML parsing error in "+ uri +
						  ": "+ sax.getMessage(),
						  sax.getException());
		}
	    if(dm == null)
		throw new DataModelException(uri+" (document cannot be located)");
	    cacheDocument(dm);
	    return dm;
	}
	catch(IOException io) {
	   throw new DataModelException("document cannot be opened: "+uri, io);
	}
    }

    protected synchronized FONIDocument getCachedDocument(String uri) {
	// linear search: never mind!
	for(int d = 0, D = cache.size(); d < D; d++) {
	    FONIDocument doc = (FONIDocument) cache.get(d);
	    if(uri.equals(doc.getBaseURI())) {
		// put at head:
		cache.remove(d);
		cache.insertElementAt(doc, 0);
		return doc;
	    }
	}
	return null;
    }

    protected synchronized void cacheDocument(FONIDocument doc) {
	if(!cache.contains(doc))
	    cache.insertElementAt(doc, 0);
	int cumulatedSize = 0;
	for(int d = 0, D = cache.size(); d < D; d++) {
	    FONIDocument doc2 = (FONIDocument) cache.get(d);
	    int size = doc2.estimateMemorySize();
	    if(cumulatedSize + size > cacheSize) {
		cache.setSize(d);
		break;
	    }
	    cumulatedSize += size;
	}
    }
}
