/*
 *	Qizx/Open version 0.4
 *
 *	Copyright (c) 2003-2004 Xavier C. FRANC -- All rights reserved.
 *
 *	This program is free software; you can redistribute it  and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation (see LICENSE.txt).
 */
package net.xfra.qizxopen.xquery.impl;

import net.xfra.qizxopen.util.*;
import net.xfra.qizxopen.xquery.*;
import net.xfra.qizxopen.xquery.op.Expression;
import net.xfra.qizxopen.xquery.op.GlobalVariable;
import net.xfra.qizxopen.xquery.op.LocalVariable;
import net.xfra.qizxopen.xquery.op.VarReference;
import net.xfra.qizxopen.xquery.op.Pragma;
import net.xfra.qizxopen.xquery.fn.Function;
import net.xfra.qizxopen.xquery.fn.UserFunction;
import net.xfra.qizxopen.xquery.fn.Prototype;
import net.xfra.qizxopen.xquery.fn.Template;
import net.xfra.qizxopen.xquery.fn.TemplateMode;
import net.xfra.qizxopen.xquery.dt.SingleDouble;
import net.xfra.qizxopen.xquery.dt.SingleInteger;
import net.xfra.qizxopen.xquery.dt.SingleString;
import net.xfra.qizxopen.xquery.dt.SingleBoolean;

import java.util.Vector;
import java.util.HashMap;
import java.util.HashSet;
import java.text.Collator;

/**
 *  Compiled Module and base class of Main Query.
 *  <p>This is essentially an opaque object.
 *  Global variables can be initialized through the diverse {@link #initGlobal} methods.
 */
public class Module  implements StaticContext
{
    protected Log      log;	// current message reporter
    protected NSPrefixMapping nsTable = new NSPrefixMapping();
    protected NSPrefixMapping predefNSTable = new NSPrefixMapping();
    protected Namespace defaultFunctionNS = Namespace.FN;
    protected Namespace defaultElementNS = Namespace.NONE;
    protected String baseURI = "";

    protected boolean checked = false;
    protected Module  predefined = PredefinedModule.BASE;
    protected String  physicalURI;	// null for query
    protected Namespace  moduleNS;	// null for query
    protected CharSequence source;

    protected Vector  importedModules = new Vector();
    protected HashMap functionMap = new HashMap();	// quick access
    protected HashMap globalMap = new HashMap();

    protected HashMap registeredCollations;
    protected String defaultCollation = Collations.CODEPOINT;
    protected Collator defaultCollator = null;	// ie codepoints

    // accumulates globals, functions and imported modules in declaration order
    protected Vector  declarations = new Vector();

    Pragma[] pragmas;

    // Local variable declarations and bindings:
    protected LocalVariable locals, lastLocal;
    protected int   maxLocalSize = 0;
    protected int   localCnt;
    protected int   localIntCnt, localDoubleCnt, localStringCnt, localItemCnt;

    //TODO ? zap
    Expression[] exprStack = new Expression[40];
    int exprDepth = 0;

   
    public static Namespace LOCAL_NS =
        Namespace.get("http://www.w3.org/2003/11/xquery-local-functions");

    public Module( ) {
        predefNamespaceDecl( "xml", Namespace.XML );
        predefNamespaceDecl( "xs", Namespace.XSD );
        predefNamespaceDecl( "xsi", Namespace.XSI );
        predefNamespaceDecl( "fn", Namespace.FN );
        predefNamespaceDecl( "xdt", Namespace.XDT );
	predefNamespaceDecl( "local", LOCAL_NS );
    }
    
    public void dump( ExprDump d ) {
        d.println( "Module" );
        d.display( "declarations", declarations);
    }
    
    public void staticCheck( ModuleManager modman, Log log ) {
        if( checked )
            return;
        
        // we synchronize on the ModuleManager, not on module, to avoid a deadlock
        // on cyclical module import. May sound a bit paranoid, but why take risks ?
        synchronized (modman) {
            // block cyclical module import:
            checked = true;
            // process declarations in order:
            for(int d = 0; d < declarations.size(); d++) {
                Object decl = declarations.get(d);
                
                if(decl instanceof GlobalVariable) {
                    GlobalVariable global = (GlobalVariable) decl;
                    
                    Expression ginit = global.init;
                    if(ginit != null) {
                        global.init = ginit = ginit.staticCheck(this);
                        if(global.getType() == null)
                            global.setType( ginit.getType() );
                        else if(!global.getType().accepts(ginit.getType()))
                            log.error( this, ginit.location,
                        "incompatible value type %2 for variable %1",
                        prefixedName(global.name),
                        ginit.getType().toString(this));
                    }
                    else if(global.getType() == null)
                        global.setType(Type.ANY);
                    GlobalVariable g = (GlobalVariable) globalMap.get(global.name);
                    if(g != null)
                        log.error(this, global.location,
				  "global variable %1 already defined",
				  prefixedName(global.name));
                    else {
                        globalMap.put( global.name, global );
                    }
                }
                else if(decl instanceof UserFunction)
                {
                    UserFunction fun = (UserFunction) decl;
                    
                    resetLocals();
                    fun.staticCheck( this );
		    allocateLocalAddress(locals);
                    fun.setLocalSize(maxLocalSize);
                }
                else if(decl instanceof Module)
                {
                    Module module = (Module) decl;
                    module.staticCheck( modman, log );
                    // add to imported module list: the module's declarations become visible
                    importedModules.add(module);
                }
            }
        }
        
    }

    // called on startup of a XQueryProcessor
    public void initGlobals( DefaultEvalContext context, HashMap globals )
	throws XQueryException
    {
        if(predefined != null)
            predefined.initGlobals( context, globals );
        
        for(int g = 0, G = declarations.size(); g < G; g++) 
            if(declarations.get(g) instanceof GlobalVariable)
            {
                GlobalVariable global = (GlobalVariable) declarations.get(g);
                if(global.init != null) {
                    Value v = global.init.eval(null, context);
                    context.setGlobal(global, v);
                }
                // is there a value specified externally ? use the fully q-name of
                // the variable (possible issue if $local:v allowed in modules)
                Value init = (Value) globals.get(global.name);

		// glitch for compatibility: if $local:var, look for $var
                if(init == null && global.name.getNamespace() == LOCAL_NS)
		    init = (Value) globals.get( QName.get(global.name.getLocalName()) );

                if(init == null) {
                    
                    continue;
                }
                // check type:
                if(!global.getType().test(init)) {
                    try { // to cast:
                        init = init.bornAgain();
                        if(!init.next())
                            throw new EvalException("cannot cast empty initial value of "
                        + global.name);
                        init = global.getType().getItemType().cast( init, context );
                    } catch(TypeException tex) {
                        context.error(global, tex);
                    }
                }
                context.setGlobal(global, init);
            }
            else if (declarations.get(g) instanceof Module) {
                Module module = (Module) declarations.get(g);
                // recursive descent to imported modules.
                // CAUTION lock modules to avoid looping forever on cyclic imports
                if(globals.get(module) == null) {	// not that beautiful but 
                    globals.put(module, module);
                    module.initGlobals( context, globals );
                    globals.remove(module);
                }
            }
    }

    public int getLocalSize() {
        return maxLocalSize;
    };

    public Log getLog() {
        return log;
    }

    void setLog(Log log) {
        this.log = log;
    }
    
    void setSource( CharSequence source, String uri ) {
        this.source = source;
        this.physicalURI = uri;
    }
    
    public CharSequence getSource( ) {
        return source;
    }

    public String printLocation( int location ) {
        return physicalURI + (source == null
			      ? (", character offset " + location)
			      : ", line "+ Log.getLineNumber(source, location));
    }
    
    public void error( Expression place,
		       String msg, String argument1, String argument2 ) {
        log.error( place.module, place.location, msg, argument1, argument2 );
    }
    
    public Expression staticCheck(Expression expr, int flags) {
        // push expr on context stack:
        if(exprDepth >= exprStack.length) {
            Expression[] old = exprStack;
            exprStack = new Expression[ old.length * 2 ];
            System.arraycopy(old, 0, exprStack, 0, old.length);
        }
        exprStack[ exprDepth++ ] = expr;
        Expression sc = expr.staticCheck(this);
        --exprDepth;
        return  sc;
    }
    
    public Expression getEnclosing( int levels ) {
        if(levels < 0 || levels > exprDepth - 1)
            return null;
        return exprStack[ exprDepth - 1 - levels ];
    }
    
    public boolean check( Prototype proto, int rank, Expression actualArgument ) {
        if(!proto.checkArg(rank, actualArgument)) {
            log.error( actualArgument.module, actualArgument.location,
            "type %3 not acceptable for argument '%2' in %1",
            new String[] { proto.toString(this), proto.getArgName(rank, this),
            actualArgument.getType().toString(this) } );
            return false;
        }
        return true;
    }
    
    public Expression resolve( Prototype[] protos, Expression[] actualArguments,
			       Expression subject ) {
	boolean errored = false;
        int pr = 0;
        for(; pr < protos.length; pr++) {
            if(protos[pr].check(actualArguments))
                break;
        }
        if(pr >= protos.length) {
	    errored = true;
            String msg = protos[0].displayName(this)+ " cannot be applied to (";
            for(int a = 0; a < actualArguments.length; a++) {
                if(a > 0) msg += ',';
                msg += ' ';
                msg += actualArguments[a].getType().toString(this);
            }
            msg += " )";
            log.error( subject.module, subject.location, msg, "");
            log.info("known signature(s):");
            for(int p = 0; p < protos.length; p++)
                if(!protos[p].hidden)
                    log.info( "  "+ protos[p].toString(this) );
            
            pr = 0;	// arbitrarily take 1st proto
        }
        Function.Call call = protos[pr].newInstance(this, actualArguments);
        call.setType( protos[pr].returnType );
        call.module = subject.module;
        call.location = subject.location;
	if(!errored)
	    call.compilationHook();	// optimizations
        return call;
    }
    
    public void checkType( Expression expr, Type expected, String message ) {
        if( !expected.accepts(expr.getType()) )
            log.error( expr.module, expr.location,
		       "type error for %1: expecting %2",
		       message, expected.toString(this) );	    
    }
    
    public void setPredefinedModule( Module predefined ) {
        this.predefined = predefined;
    }
    
    
    
    public Namespace getDefaultFunctionNS() {
        return defaultFunctionNS;
    }
    
    public Namespace getDefaultElementNS() {
        return defaultElementNS;
    }
    
    public void addDefaultNamespace( boolean forFun, String uri ) {
        if(forFun)
            defaultFunctionNS = Namespace.get(uri);
        else defaultElementNS = Namespace.get(uri);
    }

    public void predefNamespaceDecl( String prefix, Namespace ns ) {
	predefNSTable.addMapping( prefix, ns );
	nsTable.addMapping( prefix, ns );
    }

    public boolean addNamespaceDecl( int location, String prefix, String uri ) {
        if( nsTable.convertToNamespace(prefix) != null &&
	    predefNSTable.convertToNamespace(prefix) == null)
            log.error(this, location, "namespace prefix %1 is already declared",
		      prefix, "");
        nsTable.addMapping( prefix, uri );
        return true;
    }
    
    public NSPrefixMapping getInScopeNS() {
        return nsTable;
    }

    public Namespace convertPrefixToNamespace( String prefix ) {
	Namespace ns = nsTable.convertToNamespace(prefix);
	return ns;
    }

    public String prefixedName( QName name ) {
        String prefix = nsTable.convertToPrefix(name.getNamespace());
        return (prefix == null) ? name.toString()
        : prefix + ":" + name.getLocalName();
    }
    
    /**
    *	Gets the physical URI of the module (optionally defined in import).
    */
    public String  getPhysicalURI() {
        return physicalURI;
    }
    
    public void setDeclaredURI( String uri ) {
        moduleNS = Namespace.get(uri);
        defaultFunctionNS = moduleNS;
    }
    
    /**
    *	Gets the namespace URI of the module.
    */
    public String getDeclaredURI() {
        return moduleNS.getURI();
    }
    
    /**
    *	Gets the namespace of the module.
    */
    public Namespace getNamespace() {
        return moduleNS;
    }
    
    public Collator setDefaultCollation( String coll ) {
        defaultCollation = coll;
        return defaultCollator = getCollator(defaultCollation);
    }
    
    public String getDefaultCollation() {
        return defaultCollation;
    }
    
    public void setCollations( HashMap collations) {
        this.registeredCollations = collations;
    }
    
    public synchronized Collator  getCollator( String uri ) {
        if(uri == null)
            return defaultCollator;
        Collator col = registeredCollations == null ? 
                          null : (Collator) registeredCollations.get(uri);
        if(col == null)
            col = Collations.getInstanceWithStrength(uri);
        return col;
    }
    
    public void setBaseURI( String baseURI ) {
        this.baseURI = baseURI;
    }
    
    public String getBaseURI() {
        return baseURI;
    }
    
    public void addDeclaration( Object declaration ) {
        declarations.add(declaration);
    }
    
    public void storePragmas( Pragma[] pragmas ) {
        this.pragmas = pragmas;
    }
    
    public Pragma[] getPragmas() {
        return pragmas;
    }
    
    public void evalContextModule( Module imp ) {
	importedModules.add(imp);
	nsTable = imp.nsTable;	// should not declare NS
    }

    Module importedModule(int m) {
        return (Module) importedModules.get(m);
    }

    ItemType lookForType( QName typeName ) {
	// until implementation of schema import:
	return Type.findItemType(typeName);
    }

    public Function simpleFunctionLookup( QName name ) throws XQueryException {
        Function fun = (Function) functionMap.get(name);
        return fun;
    }

    public Function localFunctionLookup( QName name ) throws XQueryException {
        return simpleFunctionLookup(name);
    }

    public Function functionLookup( QName name ) throws XQueryException {
        Function fun = localFunctionLookup(name);
        if(fun != null)
            return fun;
        // search in imported modules: (shallow search)
        for(int m = importedModules.size(); --m >= 0; ) {
            fun = importedModule(m).localFunctionLookup( name );
            if(fun != null)
                return fun;
        }
        // finally search in predefined module:
        return predefined.localFunctionLookup(name);
    }

    public void  declareFunction( Function function ) {
        functionMap.put(function.getName(), function);
    }

    void declareTemplate(Template tem) { //TODO
    }

    TemplateMode declareTemplateMode(QName modeName) {
	return null; //TODO
    }


    public GlobalVariable lookforGlobalVariable(QName name)
    {
        GlobalVariable g = (GlobalVariable) globalMap.get(name);
        if(g != null)
            return g;
        for(int m = importedModules.size(); --m >= 0; ) {
            g = importedModule(m).lookforGlobalVariable(name);
            if(g != null)
                return g;
        }
        return predefined == null ? null
	                          : predefined.lookforGlobalVariable(name);
    }

    /**
     *	Defines a global variable in this module.
     */
    public GlobalVariable defineGlobal( QName qname, Type type ) {
	GlobalVariable g = new GlobalVariable(qname, type, null);
	globalMap.put(qname, g );
	addDeclaration(g);	// otherwise cant be initialized
	return g;
    }
 
    

    public LocalVariable lookforLocalVariable(QName name) {
	
	for( LocalVariable v = lastLocal; v != null; v = v.before) {
	    if(v.name == name)
		return v;
	}
	return null;
    }

    public LocalVariable defineLocalVariable(QName name, Type type,
					     Expression declaring) {
	// TODO no more check for duplicates !?
	LocalVariable decl = new LocalVariable(name, type, declaring);
	lastLocal.addAfter(decl);
	lastLocal = decl;
	return decl;
    }

    public LocalVariable markLocalVariables() {
	return lastLocal;
    }

    public void popLocalVariables(LocalVariable mark) {
	lastLocal = mark;
    }

    public void resetLocals() {
	maxLocalSize = 0;
	localCnt =
	    localIntCnt = localDoubleCnt = localStringCnt = localItemCnt = 0;
	locals = lastLocal = new LocalVariable(null, null, null);
    }

/*
    public void dumpLocals(String message) {
	System.err.println(message);
	for(LocalVariable loc = locals; loc != null; loc = loc.after) {
	    System.err.print("  local "+loc.name);
	    for(LocalVariable rep = loc.replacer; rep != null; rep = rep.replacer) {
		System.err.print(" plus "+rep.name);
		if(rep.after != null) System.err.println(" followed by "+rep.after.name);
	    }
	    System.err.println("");
	}
    }
*/

    protected void allocateLocalAddress( LocalVariable root ) {
	int addrType = -1;
	Type type = root.getType();
	if( root.name != null)
	    if(root.address < 0 || type.getOccurrence() != Type.ONE_OCC) {
		addrType = EvalContext.LAST_REGISTER;	// in stack 
		root.address = EvalContext.LAST_REGISTER + localCnt++;
		maxLocalSize = Math.max(root.address, maxLocalSize);
	    }
	    else if(type == Type.INTEGER) {
		addrType = EvalContext.INT_REGISTER;
		root.address = EvalContext.INT_REGISTER + localIntCnt ++;
		if(localIntCnt > EvalContext.MAX_REGISTER)
		    throw new RuntimeException("(internal) too many int registers");
	    }
	    else if(type == Type.STRING) {
		addrType = EvalContext.STRING_REGISTER;
		root.address = EvalContext.STRING_REGISTER + localStringCnt ++;
	    }
	    else if(type == Type.DOUBLE) {
		addrType = EvalContext.DOUBLE_REGISTER;
		root.address = EvalContext.DOUBLE_REGISTER + localDoubleCnt ++;
	    }
	    else {
		addrType = EvalContext.ITEM_REGISTER;
		root.address = EvalContext.ITEM_REGISTER + localItemCnt ++;
		if( localItemCnt >
		    EvalContext.LAST_REGISTER - EvalContext.ITEM_REGISTER)
		    throw new RuntimeException("(internal) too many Item registers");
	    }
	
	LocalVariable simult = root.after;
	for(; simult != null; simult = simult.replacer) {
	    allocateLocalAddress(simult);
	}
	switch(addrType) {
	case EvalContext.LAST_REGISTER:
	    -- localCnt; break;
	case EvalContext.INT_REGISTER:
	    -- localIntCnt; break;
	case EvalContext.STRING_REGISTER:
	    -- localStringCnt; break;
	case EvalContext.DOUBLE_REGISTER:
	    -- localDoubleCnt; break;
	case EvalContext.ITEM_REGISTER:
	    -- localItemCnt; break;
	}
    }	

    public void pushDotType(Type type) {
	
	dotTypeStack = new DotType(type.getItemType(), dotTypeStack);
    }

    public void popDotType() {
	dotTypeStack = dotTypeStack.up;
    }

    public ItemType getDotType() {
	return dotTypeStack == null ? null : dotTypeStack.type;
    }

    DotType dotTypeStack;
    private static class DotType {
	DotType up;
	ItemType type;

	DotType( ItemType ty, DotType uptype ) {
	    type = ty; up = uptype;
	}
    }
}
