

--
-- Copyright (C) 2019  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- Sokoban Reverse-BFS solver : 
-- a brute-force breadth-first-search in reverse
--
-- ibox3:  BoxInertia...[fixed-urgency] tunnel skipper+relent.
-- Not purely BFS so...after first solution is found, 
-- the longer search tree branches are skipped.  
-- Now exits only after entire queue is searched.
--
-- Inertial Box-centric version (for larger/lishout puzzles)...
-- choose a box, then direction to move it as far as possible
-- in that same direction, while saving critical intermediate 
-- positions. Ignores exact puller position but saves puller-corral.
--
-- An article by Frank Takes shows clear advantages to working from
-- a solved position backwards to the start position, which prevents
-- deadlocked positions from taking up space in the search tree.
-- I am aware that puller-deadlocks are still possible, but they are
-- less problematic because they self-terminate fairly quickly in a BFS.
--
-- This version attempts to detect tunnels
-- and avoids placing configs onto the priority queue that represent
-- partial traversals thru them.  The only exceptions are a) if pulling
-- and the box lands on a box-target;  b) if the puller lands on a
-- puller-target = initial pusher position.
--
-- Uses a splaytree, to test whether a given config was seen before.  
-- Extremely fast access, but can only solve relatively small puzzles
-- due to memory constraints.
--
-- Uses a fullproof Key-type, for which we must define
-- operators "<" and ">".







with splaylist;
with text_io;

with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;

with ada.command_line;
with ada.calendar;

with utils;



procedure ibox3r is


	use Ada.Strings.Unbounded;
	use Ada.Strings.Unbounded.Text_IO;

	use text_io;
	use utils;
	use mysplay;








-- these test box-moves




-----------------------------------------------------------










procedure pullup(
	okey: keytype;  olp,xr,xc : ushort; changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br-1;
	pc: ushort := bc;
	boxmoves: ushort := 0;
	prc,brc: ushort;
begin --pullup
	changed:=false;
	if dppathexists(pr,pc) and then btestup(br,bc) then
		changed:=true;
		loop
			vf(indx(pr+1,pc)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pr:=pr-1;
			boxmoves:=boxmoves+1;
			brc := indx(br,bc);
			prc := indx(pr,pc);

			exit when not btestup(br,bc);
			exit when bnexus(brc);
			--exit when enexus(prc);

			exit when not vtunl(prc);

		end loop;
		bsaveifnew(okey,0,olp,boxmoves, pr,pc, br,bc);
	end if;
end pullup;

-- enexus > bnexus > nexus contains goals [ff(brc)=2]
-- NOTE:  if ibox3r ever fails to find a solution,
--        simply uncomment the final exit statements.




procedure pulldown(
	okey: keytype;  olp,xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br+1;
	pc: ushort := bc;
	boxmoves: ushort := 0;
	prc,brc: ushort;
begin
	changed:=false;
	if dppathexists(pr,pc) and then btestdown(br,bc) then
		changed:=true;
		loop
			vf(indx(pr-1,pc)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pr:=pr+1;
			boxmoves:=boxmoves+1;
			brc := indx(br,bc);
			prc := indx(pr,pc);

			exit when not btestdown(br,bc);
			exit when bnexus(brc);
			--exit when enexus(prc);

			exit when not vtunl(prc);

		end loop;
		bsaveifnew(okey,1,olp,boxmoves, pr,pc, br,bc);
	end if;
end pulldown;






procedure pullleft(
	okey: keytype;  olp,xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br;
	pc: ushort := bc-1;
	boxmoves: ushort := 0;
	prc,brc: ushort;
begin
	changed:=false;
	if dppathexists(pr,pc) and then btestleft(br,bc) then
		changed:=true;
		loop
			vf(indx(pr,pc+1)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pc:=pc-1;
			boxmoves:=boxmoves+1;
			brc := indx(br,bc);
			prc := indx(pr,pc);

			exit when not btestleft(br,bc);
			exit when bnexus(brc);
			--exit when enexus(prc);

			exit when not htunl(prc);

		end loop;
		bsaveifnew(okey,3,olp,boxmoves, pr,pc, br,bc);
	end if;
end pullleft;






procedure pullright(
	okey: keytype;  olp,xr,xc : ushort;	changed: out boolean ) is

	br: ushort := xr;
	bc: ushort := xc;
	pr: ushort := br;
	pc: ushort := bc+1;
	boxmoves: ushort := 0;
	prc,brc: ushort;
begin
	changed:=false;
	if dppathexists(pr,pc) and then btestright(br,bc) then
		changed:=true;
		loop
			vf(indx(pr,pc-1)):=0;
			vf(indx(pr,pc)):=1;
			br:=pr; bc:=pc;
			pc:=pc+1;
			boxmoves:=boxmoves+1;
			brc := indx(br,bc);
			prc := indx(pr,pc);

			exit when not btestright(br,bc);
			exit when bnexus(brc);
			--exit when enexus(prc);

			exit when not htunl(prc);

		end loop;
		bsaveifnew(okey,2,olp,boxmoves, pr,pc, br,bc);
	end if;
end pullright;





















procedure trymove is
	it,odiff,diff, newstop, oldstop, avg2: integer := 0;
	okey: keytype;
	orec: hashrectype;
	olp, opr, opc, ii : ushort;
	bxfrac : float;
	difference : boolean;
begin --trymove

	newstop:=0;

	outer:
	loop


		oldstop:=newstop;
		newstop:=mysplay.length(mytree);
		diff := newstop-oldstop;
		exit outer when diff=0;


		bxfrac := float(bestnk*100)/float(gngoals+1);
		-- in order to win, we must end up with puller in the
		-- proper corral, in addition to positioning all boxes.


		if not winner then
			put("R=");
			put(ushort'image(relenting));
			put(" NewCfg="&integer'image(newstop-oldstop));
			put(", ");
			myfloat_io.put(item=>bxfrac,fore=>2,aft =>1,exp=>0);
			put("%");
			if newstop<2000 then
				put(" TotCfg="&integer'image(newstop));
			else
				put(" TotCfg(k)="&integer'image(newstop/1000));
			end if;
			new_line;
		end if;


		for it in 1..diff loop

			--NOTE: at each reentry into trymove, oldstop=0.
			if oldstop=0 and it=1 then
				mysplay.head( mytree, status ); --put iterator @ list-head
				--myassert( status=Ok, 101, "head error" );
			else
				mysplay.next( mytree, status ); --move iterator to next
				--myassert( status=Ok, 102, "next error" );
			end if;


			-- get data from iterator's current position:
			mysplay.data( mytree, okey, orec, status ); --get okey, orec
			--myassert( status=Ok, 103, "splay.data error" );


		if 
			(orec.ngoals>=ubyte(bestnk/relenting)) --greediness
			and ( orec.xlevel<1 ) --yet unexpanded
		then


			-- mark as expanded...
			-- prevents wasted effort at next relenting.
			orec.xlevel:=1;
			mysplay.modifynode(okey,orec,mytree,status);

			brestore(orec, opr,opc); --,opr,opc);
			dppathprep(opr,opc);

			olp:=orec.totpulz;


			-- do a lexicographic search for boxes,
			-- then try to move it in 4 directions:
			for br in 2..nrows-1 loop
			for bc in 2..ncols-1 loop


				ii:=indx(br,bc);

				if vf(ii)=1 and ee(ii)<256 then --process this box

					pullright(okey,olp,br,bc,difference);
					if difference then
						brestore(orec,opr,opc);
					end if;


					pullleft(okey,olp,br,bc, difference);
					if difference then
						brestore(orec,opr,opc);
					end if;


					pullup(okey,olp,br,bc, difference);
					if difference then
						brestore(orec,opr,opc);
					end if;


					pulldown(okey,olp,br,bc, difference);
					if difference then
						brestore(orec,opr,opc);
					end if;


				end if;

				exit outer when winner;

			end loop; --bc
			end loop; --br

		end if;

		end loop; --it


	end loop outer; -- while




end trymove;



	density: ushort;
	Ok: boolean;

begin -- ibox3

	put_line("Bytes per hashrec:" & integer'image(hashrectype'size/8));
	put_line("Bytes per keytype:" & integer'image(keytype'size/8));
	put_line("Bytes per ushort:" & integer'image(ushort'size/8));
	put_line("Bytes per ubyte:" & integer'image(ubyte'size/8));

	put_line("Bytes per pooltype:" & integer'image(pooltype'size/8));
	put_line("Bytes per VFS:" & integer'image(vfstype'size/8));
	put_line("Bytes per vftype:" & integer'image(vftype'size/8));


	checkForUserFile(Ok);
	-- defines:  infilname, level, maxlevel

	if Ok then

		winner:=false;

		readPuzzle(level);
		density := 100*gngoals/ushort(nbvalid);

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(gngoals));

		myassert( mysplay.length(mytree) = 0, 99999, "initialSplay" );

		bsave0;

		findnexii;

		tsec0:=ada.calendar.seconds(ada.calendar.clock);


		relenting:=2;
		--if density>20 then relenting:=4; end if; --31jul19

		trymove;
		while not winner loop
			relenting:=relenting*2;
			exit when relenting>gngoals*4;
			trymove;
		end loop;



		if not winner then
			new_line;
			put_line("Failure to find solution.");
		else
			put_line("Winner=========================================");
		end if;

		--put_line(" minBoxPulls="&ushort'image(minBoxPulls));

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(gngoals));
		put(" ibox3r: tunnel-skipper + relenting=");
		put_line(ushort'image(relenting));

		bdump(0,0); --show nexii on screen
		dumpvalid;
		put_line(" Density="&ushort'image(density));
		put(" Winning value of relenting="&ushort'image(relenting));
		new_line;

	end if;

exception
	when storage_error =>
		put_line("Memory insufficient to solve this problem with this algorithm!");
		raise;


end ibox3r;
