

--
-- Copyright (C) 2018  <fastrgv@gmail.com>
--
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
--
-- You may read the full text of the GNU General Public License
-- at <http://www.gnu.org/licenses/>.
--


-- Sokoban Reverse-BFS solver : 
-- a brute-force breadth-first-search in reverse
--
-- ipuller2...
-- adjustable uni-dir-inertia for pulls;
-- uni-dir-inertia for non-pulls, 
-- but no saving in tunnels, unless on puller-goal;
--
-- Puller-centric version... (good for small,dense puzzles)
-- chooses puller direction {no,so,ea,we} to try
-- and then whether or not to pull any adjacent box.
--
-- An article by Frank Takes shows clear advantages to working from
-- a solved position backwards to the start position, which prevents
-- deadlocked positions from taking up space in the search tree.
-- I am aware that puller-deadlocks are still possible, but they are
-- less problematic because they self-terminate fairly quickly in a BFS.
--
-- This version attempts to detect tunnels
-- and avoids placing configs onto the priority queue that represent
-- partial traversals thru them.  The only exceptions are a) if pulling
-- and the box lands on a box-target;  b) if the puller lands on a
-- puller-target = initial pusher position.
--
-- Uses a splaytree, to test whether a given config was seen before.  
-- Extremely fast access, but can only solve relatively small puzzles
-- due to memory constraints.
--
-- Uses a fullproof Key-type, for which we must define
-- operators "<" and ">".








with system;
with splaylist;
with text_io;

with Ada.Strings.Unbounded;
with Ada.Strings.Unbounded.Text_IO;

with ada.command_line;
with ada.calendar;
with utils;




procedure ipuller3 is


	use Ada.Strings.Unbounded;
	use Ada.Strings.Unbounded.Text_IO;

	use text_io;
	use utils;
	use mysplay;





--///////////////////////////////////////////////////////////////////////
-- the following 8 procs all attempt to exit "tunnels" prior to saving,
-- where "tunnel" means turns are not possible:
--///////////////////////////////////////////////////////////////////////

procedure moveup(okey: keytype) is -- without pulling
	moves,boxmoves: ushort := 0;
begin
	pr:=pr-1;
	moves:=1;
	while 
		ptestup and 
		not ptestright and 
		not ptestleft and
		(pr/=gpr or pc/=gpc) 
	loop 
		pr:=pr-1; 
		moves:=moves+1;
	end loop;
	psaveifnew(okey,0,boxmoves,moves);
end moveup;

-- only called if testup=true
procedure pullup(okey: keytype;  changed: out boolean) is
	boxmoves: ushort := 0;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr+1,pc))=1 then -- box to pull
		changed:=true;
		loop
			vf(indx(pr+1,pc)):=0;
			vf(indx(pr,pc)):=1;
			pr:=pr-1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr+1,pc); --box index
			exit when not ptestup;        --puller blocked
			exit when ff(ib)=2; --box on goal
			exit when bnexus(ib); -- Bvalid+Enexus
			exit when enexus( indx(pr,pc) ); --key puller pos
			exit when pr=gpr and pc=gpc; --puller on puller goal
		end loop;
		psaveifnew(okey,0,boxmoves,boxmoves);
	end if;
end pullup;






procedure movedown(okey: keytype) is -- without pulling
	moves,boxmoves: ushort := 0;
begin
	pr:=pr+1;
	moves:=1;
	while 
		ptestdown and 
		not ptestright and 
		not ptestleft and
		(pr/=gpr or pc/=gpc) 
	loop 
		pr:=pr+1; 
		moves:=moves+1;
	end loop;
	psaveifnew(okey,1,boxmoves,moves);
end movedown;

-- only called if testdown=true
procedure pulldown(okey: keytype;  changed: out boolean) is
	boxmoves: ushort := 0;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr-1,pc))=1 then -- box to pull
		changed:=true;
		loop
			vf(indx(pr-1,pc)):=0;
			vf(indx(pr,pc)):=1;
			pr:=pr+1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr-1,pc); --box index
			exit when not ptestdown;
			exit when ff(ib)=2; --box on goal
			exit when bnexus(ib); -- 
			exit when enexus( indx(pr,pc) ); --key puller pos
			exit when pr=gpr and pc=gpc;
		end loop;
		psaveifnew(okey,1,boxmoves,boxmoves);
	end if;
end pulldown;







procedure moveleft(okey: keytype) is -- without pulling
	moves,boxmoves: ushort := 0;
begin
	pc:=pc-1;
	moves:=1;
	while 
		ptestleft and 
		not ptestup and 
		not ptestdown and
		(pr/=gpr or pc/=gpc) 
	loop 
		pc:=pc-1; 
		moves:=moves+1;
	end loop;
	psaveifnew(okey,3,boxmoves,moves);
end moveleft;

-- only called when testleft=true
procedure pullleft(okey: keytype;  changed: out boolean) is
	boxmoves: ushort := 0;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr,pc+1))=1 then -- box to pull
		changed:=true;
		loop
			vf(indx(pr,pc+1)):=0;
			vf(indx(pr,pc)):=1;
			pc:=pc-1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr,pc+1); --box index
			exit when not ptestleft;
			exit when ff(ib)=2; --box on goal
			exit when bnexus(ib); -- slower
			exit when enexus( indx(pr,pc) ); --key puller pos
			exit when pr=gpr and pc=gpc;
		end loop;
		psaveifnew(okey,3,boxmoves,boxmoves);
	end if;
end pullleft;






procedure moveright(okey: keytype) is -- without pulling
	moves,boxmoves: ushort := 0;
begin
	pc:=pc+1;
	moves:=1;
	while 
		ptestright and 
		not ptestup and 
		not ptestdown and
		(pr/=gpr or pc/=gpc) 
	loop 
		pc:=pc+1; 
		moves:=moves+1;
	end loop;
	psaveifnew(okey,2,boxmoves,moves);
end moveright;

-- only called when testright=true
procedure pullright(okey: keytype;  changed: out boolean) is
	boxmoves: ushort := 0;
	ib: ushort;
begin
	changed:=false;
	if vf(indx(pr,pc-1))=1 then -- box to pull
		changed:=true;
		loop
			vf(indx(pr,pc-1)):=0;
			vf(indx(pr,pc)):=1;
			pc:=pc+1;
			boxmoves:=boxmoves+1;
			ib:=indx(pr,pc-1); --box index
			exit when not ptestright;
			exit when ff(ib)=2; --box on goal
			exit when bnexus(ib); -- 
			exit when enexus( indx(pr,pc) ); --key puller pos
			exit when pr=gpr and pc=gpc;
		end loop;
		psaveifnew(okey,2,boxmoves,boxmoves);
	end if;
end pullright;









procedure trymove is
	newstop, oldstop: integer := 0;
	okey: keytype;
	orec: hashrectype;
	prev, bp : ubyte;
	opr, opc : ushort;
	pch: character;
	lbox, rbox, ubox, dbox, changed : boolean;
	bxfrac : float;
begin --trymove


	if relent then
		put_line("################### relented Mode");
	else
		put_line("################### UNrelented Mode");
	end if;

	newstop:=0;

	loop

		--exit when winner;

		countbest(nk,ng);
		if bestnk<nk then bestnk:=nk; end if;
		bxfrac := float(bestnk*100)/float(ng);


		depth:=depth+1;

		oldstop:=newstop;
		newstop:=mysplay.length(mytree);

		exit when oldstop=newstop;


		if not winner then
		put(" NewCfg="&integer'image(newstop-oldstop));
		put(" depth="&integer'image(depth)&", %=");
		myfloat_io.put(item=>bxfrac,fore=>2,aft =>1,exp=>0);
		if newstop<2000 then
			put(" TotCfg="&integer'image(newstop));
		else
			put(" TotCfg(k)="&integer'image(newstop/1000));
		end if;
		new_line;
		end if;


		for it in 1 .. newstop-oldstop loop

			--exit when winner;

			if oldstop=0 and it=1 then
				mysplay.head( mytree, status ); --put iterator @ list-head
				myassert( status=Ok, 101, "head error" );
			else
				mysplay.next( mytree, status ); --move iterator to next
				myassert( status=Ok, 102, "next error" );
			end if;

			-- get data from iterator's current position:
			mysplay.data( mytree, okey, orec, status ); --get okey, orec
			myassert( status=Ok, 103, "splay.data error" );


			olpulz:=orec.totpulz;
			olmovz:=orec.totmovz;

		if 
			(olpulz+1<minBoxPulls) 
			and (olmovz+1<minMoves) --only works AFTER 1st soln found
			and (relent or (orec.ngoals>=ubyte(bestnk/2)))
		then


			prestore(orec);
			prev:= orec.prevmove;
			bp:= orec.boxpull; -- # [straight-line] pulls of this box


			if bp>0 then -- was a pull

				case prev is
					when 0 => pch:='D';
					when 1 => pch:='U';
					when 2 => pch:='L';
					when 3 => pch:='R';
					when others => pch:='X';
				end case;

			else -- was a move with no pull

				case prev is
					when 0 => pch:='d';
					when 1 => pch:='u';
					when 2 => pch:='l';
					when 3 => pch:='r';
					when others => pch:='x';
				end case;

			end if;


			opr:=ushort(orec.prsave);
			opc:=ushort(orec.pcsave);
			lbox:=(vf(indx(opr,opc-1))=1);
			rbox:=(vf(indx(opr,opc+1))=1);
			ubox:=(vf(indx(opr-1,opc))=1);
			dbox:=(vf(indx(opr+1,opc))=1);

			if ptestright then
				if pch/='r' then
					moveright(okey);
					prestore(orec);
				end if;
				changed:=false;
				if lbox then pullright(okey,changed); end if;
				if changed then prestore(orec); end if;
			end if;


			if ptestleft then
				if pch/='l' then
					moveleft(okey);
					prestore(orec);
				end if;
				changed:=false;
				if rbox then pullleft(okey,changed); end if;
				if changed then prestore(orec); end if;
			end if;


			if ptestup then
				if pch/='u' then
					moveup(okey);
					prestore(orec);
				end if;
				changed:=false;
				if dbox then pullup(okey,changed); end if;
				if changed then prestore(orec); end if;
			end if;


			if ptestdown then
				if pch/='d' then
					movedown(okey);
					prestore(orec);
				end if;
				changed:=false;
				if ubox then pulldown(okey,changed); end if;
				if changed then prestore(orec); end if;
			end if;


		end if;

		end loop; --it::944


	end loop; -- while::940



end trymove;




	Ok: boolean;

begin -- ipuller3

	checkForUserFile(Ok);
	-- defines:  infilname, level, maxlevel

	if Ok then

		readPuzzle(level);

		psave0;

		countbest(nk,gngoals);
		if bestnk<nk then bestnk:=nk; end if;

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(gngoals));



		findnexii;

	tsec0:=ada.calendar.seconds(ada.calendar.clock);


		depth:=0;


		trymove;

		if not winner then 
			relent:=true; 
			trymove; 
			new_line; 
		end if;



		if not winner then
			put_line("Failure to find solution.");
		else
			put_line("Winner=====================================");
		end if;


		put_line(" minMoves="&ushort'image(minMoves));
		put_line(" minBoxPulls="&ushort'image(minBoxPulls));

		put_line(" nrows="&ushort'image(nrows));
		put_line(" ncols="&ushort'image(ncols));
		put_line(" pfmax="&ushort'image(pfmax));
		put_line(" nBox="&ushort'image(gngoals));
		put_line(" ipuller3 with uni-dir-inertia");

		pdump; --show nexii on screen

-- ipuller3...
-- uni-dir-inertia for pulls;
-- uni-dir-inertia for non-pulls, 
-- but no saving in tunnels, unless on puller-goal;

	end if;

	--put_line(integer'image(system.storage_pools.storage_size(pool)));

	--put_line(long_long_integer'image(system.memory_size/1_000_000/1_000_000));
	--put_line(long_long_integer'image(system.memory_size/1_000_000/system.storage_unit/1_000_000));


exception
	when storage_error =>
		put_line("Memory insufficient to solve this problem with this algorithm!");
		raise;

end ipuller3;
