package gqclient

import (
	"errors"
	"io"
	prand "math/rand"
	"net"
	"time"
)

// BtoInt converts a byte slice into int in Big Endian order
// Uint methods from binary package can be used, but they are messy
func BtoInt(b []byte) int {
	var mult uint = 1
	var sum uint
	length := uint(len(b))
	var i uint
	for i = 0; i < length; i++ {
		sum += uint(b[i]) * (mult << ((length - i - 1) * 8))
	}
	return int(sum)
}

// PsudoRandBytes returns a byte slice filled with psudorandom bytes generated by the seed
func PsudoRandBytes(length int, seed int64) []byte {
	prand.Seed(seed)
	ret := make([]byte, length)
	prand.Read(ret)
	return ret
}

// ReadTillDrain reads TLS data according to its record layer
func ReadTillDrain(conn net.Conn, buffer []byte) (n int, err error) {
	// TCP is a stream. Multiple TLS messages can arrive at the same time,
	// a single message can also be segmented due to MTU of the IP layer.
	// This function guareentees a single TLS message to be read and everything
	// else is left in the buffer.
	i, err := io.ReadFull(conn, buffer[:5])
	if err != nil {
		return
	}

	dataLength := BtoInt(buffer[3:5])
	left := dataLength
	readPtr := 5

	conn.SetReadDeadline(time.Now().Add(3 * time.Second))
	for left != 0 {
		if readPtr > len(buffer) || readPtr+left > len(buffer) {
			err = errors.New("Reading TLS message: message size greater than buffer")
			return
		}
		// If left > buffer size (i.e. our message got segmented), the entire MTU is read
		// if left = buffer size, the entire buffer is all there left to read
		// if left < buffer size (i.e. multiple messages came together),
		// only the message we want is read
		i, err = io.ReadFull(conn, buffer[readPtr:readPtr+left])
		if err != nil {
			return
		}
		left -= i
		readPtr += i
	}
	conn.SetReadDeadline(time.Time{})
	n = 5 + dataLength
	return
}
