/**
 * Module for AWS IoT configuration and connection establishment
 *
 * @packageDocumentation
 */
import { MqttConnectionConfig, MqttWill } from "./mqtt";
import * as io from "./io";
import { TlsContextOptions } from "./io";
import { HttpProxyOptions } from "./http";
import { WebsocketOptionsBase } from "../common/auth";
import { AwsCredentialsProvider } from "./auth";
/**
 * Websocket-specific mqtt connection configuration options
 *
 * @category IoT
 */
export interface WebsocketConfig extends WebsocketOptionsBase {
    /** Sources the AWS Credentials used to sign the websocket connection handshake */
    credentials_provider: AwsCredentialsProvider;
    /** (Optional) http proxy configuration */
    proxy_options?: HttpProxyOptions;
    /** AWS region the websocket connection is being established in.  Must match the region embedded in the
     * endpoint.
     */
    region: string;
    /** (Optional)  TLS configuration to use when establishing the connection */
    tls_ctx_options?: TlsContextOptions;
}
/**
 * Builder functions to create a {@link MqttConnectionConfig} which can then be used to create
 * a {@link MqttClientConnection}, configured for use with AWS IoT.
 *
 * @category IoT
 */
export declare class AwsIotMqttConnectionConfigBuilder {
    private tls_ctx_options;
    private params;
    private is_using_custom_authorizer;
    private constructor();
    /**
     * Create a new builder with mTLS file paths
     * @param cert_path - Path to certificate, in PEM format
     * @param key_path - Path to private key, in PEM format
     */
    static new_mtls_builder_from_path(cert_path: string, key_path: string): AwsIotMqttConnectionConfigBuilder;
    /**
     * Create a new builder with mTLS cert pair in memory
     * @param cert - Certificate, in PEM format
     * @param private_key - Private key, in PEM format
     */
    static new_mtls_builder(cert: string, private_key: string): AwsIotMqttConnectionConfigBuilder;
    /**
     * Create a new builder with mTLS using a PKCS#11 library for private key operations.
     *
     * NOTE: This configuration only works on Unix devices.
     * @param pkcs11_options - PKCS#11 options.
     */
    static new_mtls_pkcs11_builder(pkcs11_options: TlsContextOptions.Pkcs11Options): AwsIotMqttConnectionConfigBuilder;
    /**
     * Create a new builder with mTLS using a certificate in a Windows certificate store.
     *
     * NOTE: This configuration only works on Windows devices.
     * @param certificate_path - Path to certificate in a Windows certificate store.
     *      The path must use backslashes and end with the certificate's thumbprint.
     *      Example: `CurrentUser\MY\A11F8A9B5DF5B98BA3508FBCA575D09570E0D2C6`
     */
    static new_mtls_windows_cert_store_path_builder(certificate_path: string): AwsIotMqttConnectionConfigBuilder;
    /**
     * Creates a new builder with default Tls options. This requires setting the connection details manually.
     */
    static new_default_builder(): AwsIotMqttConnectionConfigBuilder;
    static new_websocket_builder(...args: any[]): AwsIotMqttConnectionConfigBuilder;
    private static configure_websocket_handshake;
    /**
     * Configures the connection to use MQTT over websockets. Forces the port to 443.
     */
    static new_with_websockets(options?: WebsocketConfig): AwsIotMqttConnectionConfigBuilder;
    /**
     * Overrides the default system trust store.
     * @param ca_dirpath - Only used on Unix-style systems where all trust anchors are
     * stored in a directory (e.g. /etc/ssl/certs).
     * @param ca_filepath - Single file containing all trust CAs, in PEM format
     */
    with_certificate_authority_from_path(ca_dirpath?: string, ca_filepath?: string): this;
    /**
     * Overrides the default system trust store.
     * @param ca - Buffer containing all trust CAs, in PEM format
     */
    with_certificate_authority(ca: string): this;
    /**
     * Configures the IoT endpoint for this connection
     * @param endpoint The IoT endpoint to connect to
     */
    with_endpoint(endpoint: string): this;
    /**
     * The port to connect to on the IoT endpoint
     * @param port The port to connect to on the IoT endpoint. Usually 8883 for MQTT, or 443 for websockets
     */
    with_port(port: number): this;
    /**
     * Configures the client_id to use to connect to the IoT Core service
     * @param client_id The client id for this connection. Needs to be unique across all devices/clients.
     */
    with_client_id(client_id: string): this;
    /**
     * Determines whether or not the service should try to resume prior subscriptions, if it has any
     * @param clean_session true if the session should drop prior subscriptions when this client connects, false to resume the session
     */
    with_clean_session(clean_session: boolean): this;
    /**
     * Configures MQTT keep-alive via PING messages. Note that this is not TCP keepalive.
     * @param keep_alive How often in seconds to send an MQTT PING message to the service to keep the connection alive
     */
    with_keep_alive_seconds(keep_alive: number): this;
    /**
     * Configures the TCP socket timeout (in milliseconds)
     * @param timeout_ms TCP socket timeout
     * @deprecated
     */
    with_timeout_ms(timeout_ms: number): this;
    /**
     * Configures the PINGREQ response timeout (in milliseconds)
     * @param ping_timeout PINGREQ response timeout
     */
    with_ping_timeout_ms(ping_timeout: number): this;
    /**
     * Configures the protocol operation timeout (in milliseconds)
     * @param protocol_operation_timeout protocol operation timeout
     */
    with_protocol_operation_timeout_ms(protocol_operation_timeout: number): this;
    /**
     * Configures the will message to be sent when this client disconnects
     * @param will The will topic, qos, and message
     */
    with_will(will: MqttWill): this;
    /**
     * Configures the common settings for the socket to use when opening a connection to the server
     * @param socket_options The socket settings
     */
    with_socket_options(socket_options: io.SocketOptions): this;
    /**
     * Configures AWS credentials (usually from Cognito) for this connection
     * @param aws_region The service region to connect to
     * @param aws_access_id IAM Access ID
     * @param aws_secret_key IAM Secret Key
     * @param aws_sts_token STS token from Cognito (optional)
     */
    with_credentials(aws_region: string, aws_access_id: string, aws_secret_key: string, aws_sts_token?: string): AwsIotMqttConnectionConfigBuilder;
    /**
     * Configure the http proxy options to use to establish the connection
     * @param proxy_options proxy options to use to establish the mqtt connection
     */
    with_http_proxy_options(proxy_options: HttpProxyOptions): this;
    /**
     * Sets the custom authorizer settings. This function will modify the username, port, and TLS options.
     *
     * @param username The username to use with the custom authorizer. If an empty string is passed, it will
     *                 check to see if a username has already been set (via WithUsername function). If no
     *                 username is set then no username will be passed with the MQTT connection.
     * @param authorizerName The name of the custom authorizer. If an empty string is passed, then
     *                       'x-amz-customauthorizer-name' will not be added with the MQTT connection.
     * @param authorizerSignature The signature of the custom authorizer. If an empty string is passed, then
     *                            'x-amz-customauthorizer-signature' will not be added with the MQTT connection.
     * @param password The password to use with the custom authorizer. If null is passed, then no password will
     *                 be set.
     */
    with_custom_authorizer(username: string, authorizer_name: string, authorizer_signature: string, password: string): this;
    /**
     * Sets username for the connection
     *
     * @param username the username that will be passed with the MQTT connection
     */
    with_username(username: string): this;
    /**
     * Sets password for the connection
     *
     * @param password the password that will be passed with the MQTT connection
     */
    with_password(password: string): this;
    /**
     * Configure the max reconnection period (in second). The reonnection period will
     * be set in range of [reconnect_min_sec,reconnect_max_sec].
     * @param reconnect_max_sec max reconnection period
     */
    with_reconnect_max_sec(max_sec: number): this;
    /**
     * Configure the min reconnection period (in second). The reonnection period will
     * be set in range of [reconnect_min_sec,reconnect_max_sec].
     * @param reconnect_min_sec min reconnection period
     */
    with_reconnect_min_sec(min_sec: number): this;
    /**
     * Returns the configured MqttConnectionConfig.  On the first invocation of this function, the TLS context is cached
     * and re-used on all subsequent calls to build().
     * @returns The configured MqttConnectionConfig
     */
    build(): MqttConnectionConfig;
}
