// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.Client = void 0;
const dbg = require("debug");
const debug = dbg('azure-iot-device:DeviceClient');
const debugErrors = dbg('azure-iot-device:DeviceClient:Errors');
const azure_iot_common_1 = require("azure-iot-common");
const internal_client_1 = require("./internal_client");
const blob_upload_1 = require("./blob_upload");
const sas_authentication_provider_1 = require("./sas_authentication_provider");
const x509_authentication_provider_1 = require("./x509_authentication_provider");
const sak_authentication_provider_1 = require("./sak_authentication_provider");
function safeCallback(callback, error, result) {
    if (callback)
        callback(error, result);
}
/**
 * IoT Hub device client used to connect a device with an Azure IoT hub.
 *
 * Users of the SDK should call one of the factory methods,
 * {@link azure-iot-device.Client.fromConnectionString|fromConnectionString}
 * or {@link azure-iot-device.Client.fromSharedAccessSignature|fromSharedAccessSignature}
 * to create an IoT Hub device client.
 */
class Client extends internal_client_1.InternalClient {
    /**
     * @constructor
     * @param {Object}  transport         An object that implements the interface
     *                                    expected of a transport object, e.g.,
     *                                    {@link azure-iot-device-http.Http|Http}.
     * @param {string}  connStr           A connection string (optional: when not provided, updateSharedAccessSignature must be called to set the SharedAccessSignature token directly).
     * @param {Object}  blobUploadClient  An object that is capable of uploading a stream to a blob.
     * @param {Object}  fileUploadApi     An object that is used for communicating with IoT Hub for Blob Storage related actions.
     */
    constructor(transport, connStr, blobUploadClient, fileUploadApi) {
        super(transport, connStr);
        this._blobUploadClient = blobUploadClient;
        this._c2dFeature = false;
        this._fileUploadApi = fileUploadApi;
        this.on('removeListener', (eventName) => {
            if (eventName === 'message' && this.listeners('message').length === 0) {
                /*Codes_SRS_NODE_DEVICE_CLIENT_16_005: [The client shall stop listening for messages from the service whenever the last listener unsubscribes from the `message` event.]*/
                debug('in removeListener, disabling C2D.');
                this._disableC2D((err) => {
                    if (err) {
                        debugErrors('in removeListener, error disabling C2D: ' + err);
                        this.emit('error', err);
                    }
                    else {
                        this._c2dFeature = false;
                        debug('removeListener successfully disabled C2D.');
                    }
                });
            }
        });
        this.on('newListener', (eventName) => {
            if (eventName === 'message') {
                /*Codes_SRS_NODE_DEVICE_CLIENT_16_004: [The client shall start listening for messages from the service whenever there is a listener subscribed to the `message` event.]*/
                debug('in newListener, enabling C2D.');
                this._enableC2D((err) => {
                    if (err) {
                        debugErrors('in newListener, error enabling C2D: ' + err);
                        this.emit('error', err);
                    }
                    else {
                        this._c2dFeature = true;
                        debug('in newListener, successfully enabled C2D');
                    }
                });
            }
        });
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_002: [The `message` event shall be emitted when a cloud-to-device message is received from the IoT Hub service.]*/
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_003: [The `message` event parameter shall be a `message` object.]*/
        this._transport.on('message', (msg) => {
            this.emit('message', msg);
        });
        this._deviceDisconnectHandler = (err) => {
            if (err) {
                debugErrors('transport disconnect event: ' + err);
            }
            else {
                debug('transport disconnect event: no error');
            }
            if (err && this._retryPolicy.shouldRetry(err)) {
                debugErrors('reconnect policy specifies a reconnect on error');
                /*Codes_SRS_NODE_DEVICE_CLIENT_16_097: [If the transport emits a `disconnect` event while the client is subscribed to c2d messages the retry policy shall be used to reconnect and re-enable the feature using the transport `enableC2D` method.]*/
                if (this._c2dFeature) {
                    // turn on C2D
                    debug('disconnectHandler re-enabling C2D');
                    this._enableC2D((err) => {
                        if (err) {
                            /*Codes_SRS_NODE_DEVICE_CLIENT_16_102: [If the retry policy fails to reestablish the C2D functionality a `disconnect` event shall be emitted with a `results.Disconnected` object.]*/
                            debugErrors('error on _enableC2D in _deviceDisconnectHandler. Failed to reestablish C2D functionality: ' + err);
                            this.emit('disconnect', new azure_iot_common_1.results.Disconnected(err));
                        }
                        else {
                            debug('_deviceDisconnectHandler has enabled C2D');
                        }
                    });
                }
                else {
                    debug('C2D has not been enabled on the device');
                }
            }
        };
        this._transport.on('disconnect', this._deviceDisconnectHandler);
    }
    setOptions(options, done) {
        if (!options)
            throw new ReferenceError('options cannot be falsy.');
        if (this._blobUploadClient) {
            /*Codes_SRS_NODE_DEVICE_CLIENT_99_103: [The `setOptions` method shall set `blobUploadClient` options.]*/
            this._blobUploadClient.setOptions(options);
        }
        if (this._fileUploadApi) {
            /* [The `setOptions` method shall set `fileUploadApi` options.]*/
            this._fileUploadApi.setOptions(options);
        }
        return super.setOptions(options, done);
    }
    close(closeCallback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            this._transport.removeListener('disconnect', this._deviceDisconnectHandler);
            super.close(_callback);
        }, closeCallback);
    }
    /**
     * Registers a callback for a method named `methodName`.
     *
     * @param methodName Name of the method that will be handled by the callback
     * @param callback Function that shall be called whenever a method request for the method called `methodName` is received.
     */
    onDeviceMethod(methodName, callback) {
        this._onDeviceMethod(methodName, callback);
    }
    uploadToBlob(blobName, stream, streamLength, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_DEVICE_CLIENT_16_037: [The `uploadToBlob` method shall throw a `ReferenceError` if `blobName` is falsy.]*/
            if (!blobName)
                throw new ReferenceError('blobName cannot be \'' + blobName + '\'');
            /*Codes_SRS_NODE_DEVICE_CLIENT_16_038: [The `uploadToBlob` method shall throw a `ReferenceError` if `stream` is falsy.]*/
            if (!stream)
                throw new ReferenceError('stream cannot be \'' + stream + '\'');
            /*Codes_SRS_NODE_DEVICE_CLIENT_16_039: [The `uploadToBlob` method shall throw a `ReferenceError` if `streamLength` is falsy.]*/
            if (!streamLength)
                throw new ReferenceError('streamLength cannot be \'' + streamLength + '\'');
            const retryOp = new azure_iot_common_1.RetryOperation('uploadToBlob', this._retryPolicy, this._maxOperationTimeout);
            retryOp.retry((opCallback) => {
                /*Codes_SRS_NODE_DEVICE_CLIENT_16_040: [The `uploadToBlob` method shall call the `_callback` callback with an `Error` object if the upload fails.]*/
                /*Codes_SRS_NODE_DEVICE_CLIENT_16_041: [The `uploadToBlob` method shall call the `_callback` callback no parameters if the upload succeeds.]*/
                this._blobUploadClient.uploadToBlob(blobName, stream, streamLength, opCallback);
            }, (err, result) => {
                safeCallback(_callback, err, result);
            });
        }, callback);
    }
    getBlobSharedAccessSignature(blobName, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_DEVICE_CLIENT_41_001: [The `getBlobSharedAccessSignature` method shall throw a `ReferenceError` if `blobName` is falsy.]*/
            if (!blobName)
                throw new ReferenceError('blobName cannot be \'' + blobName + '\'');
            const retryOp = new azure_iot_common_1.RetryOperation('getBlobSharedAccessSignature', this._retryPolicy, this._maxOperationTimeout);
            retryOp.retry((opCallback) => {
                /*Codes_SRS_NODE_DEVICE_CLIENT_41_002: [The `getBlobSharedAccessSignature` method shall call the `getBlobSharedAccessSignature` method in the instantiated `_fileUploadApi` class and pass in `blobName` as a parameter.]*/
                this._fileUploadApi.getBlobSharedAccessSignature(blobName, opCallback);
            }, (err, result) => {
                /*Codes_SRS_NODE_DEVICE_CLIENT_41_003: [The `getBlobSharedAccessSignature` method shall call the `_callback` callback with `err` and `result` from the call to `getBlobSharedAccessSignature`.]*/
                if (!err) {
                    debug('got blob storage shared access signature.');
                }
                else {
                    debugErrors('Could not obtain blob shared access signature: ' + err);
                }
                safeCallback(_callback, err, result);
            });
        }, callback);
    }
    notifyBlobUploadStatus(correlationId, isSuccess, statusCode, statusDescription, callback) {
        return (0, azure_iot_common_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_DEVICE_CLIENT_41_016: [The `notifyBlobUploadStatus` method shall throw a `ReferenceError` if `correlationId` is falsy.]*/
            /*Codes_SRS_NODE_DEVICE_CLIENT_41_005: [The `notifyBlobUploadStatus` method shall throw a `ReferenceError` if `isSuccess` is falsy but not the boolean false.]*/
            /*Codes_SRS_NODE_DEVICE_CLIENT_41_006: [The `notifyBlobUploadStatus` method shall throw a `ReferenceError` if `statusCode` is falsy but not the number 0.]*/
            /*Codes_SRS_NODE_DEVICE_CLIENT_41_007: [The `notifyBlobUploadStatus` method shall throw a `ReferenceError` if `statusDescription` is falsy but not an empty string.]*/
            if (!correlationId)
                throw new ReferenceError('correlationId cannot be \' ' + correlationId + ' \'');
            if (!isSuccess && typeof (isSuccess) !== 'boolean')
                throw new ReferenceError('isSuccess cannot be \' ' + isSuccess + ' \'');
            if (!statusCode && !(statusCode === 0))
                throw new ReferenceError('statusCode cannot be \' ' + statusCode + ' \'');
            if (!statusDescription && statusDescription !== '')
                throw new ReferenceError('statusDescription cannot be \' ' + statusDescription + ' \'.');
            const retryOp = new azure_iot_common_1.RetryOperation('notifyBlobUploadStatus', this._retryPolicy, this._maxOperationTimeout);
            retryOp.retry((opCallback) => {
                let uploadResult = { isSuccess: isSuccess, statusCode: statusCode, statusDescription: statusDescription };
                /*Codes_SRS_NODE_DEVICE_CLIENT_41_015: [The `notifyBlobUploadStatus` method shall call the `notifyUploadComplete` method via the internal `_fileUploadApi` class.]*/
                this._fileUploadApi.notifyUploadComplete(correlationId, uploadResult, opCallback);
            }, (err) => {
                /*Codes_SRS_NODE_DEVICE_CLIENT_41_008: [The `notifyBlobUploadStatus` method shall call the `_callback` callback with `err` if the notification fails.]*/
                /*Codes_SRS_NODE_DEVICE_CLIENT_41_009: [The `notifyBlobUploadStatus` method shall call the `_callback` callback with no parameters if the notification succeeds.]*/
                safeCallback(_callback, err);
            });
        }, callback);
    }
    _enableC2D(callback) {
        debug('enabling C2D');
        const retryOp = new azure_iot_common_1.RetryOperation('_enableC2D', this._retryPolicy, this._maxOperationTimeout);
        retryOp.retry((opCallback) => {
            this._transport.enableC2D(opCallback);
        }, (err) => {
            if (!err) {
                debug('enabled C2D');
            }
            else {
                debugErrors('Error while enabling C2D: ' + err);
            }
            callback(err);
        });
    }
    _disableC2D(callback) {
        debug('disabling C2D');
        this._transport.disableC2D((err) => {
            if (!err) {
                debug('disabled C2D');
            }
            else {
                debugErrors('Error while disabling C2D: ' + err);
            }
            callback(err);
        });
    }
    /**
     * Creates an IoT Hub device client from the given connection string using the given transport type.
     *
     * @param {String}    connStr        A connection string which encapsulates "device connect" permissions on an IoT hub.
     * @param {Function}  transportCtor  A transport constructor.
     *
     * @throws {ReferenceError}         If the connStr parameter is falsy.
     *
     * @returns {module:azure-iot-device.Client}
     */
    static fromConnectionString(connStr, transportCtor) {
        /*Codes_SRS_NODE_DEVICE_CLIENT_05_003: [The fromConnectionString method shall throw ReferenceError if the connStr argument is falsy.]*/
        if (!connStr)
            throw new ReferenceError('connStr is \'' + connStr + '\'');
        const cn = azure_iot_common_1.ConnectionString.parse(connStr);
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_087: [The `fromConnectionString` method shall create a new `SharedAccessKeyAuthorizationProvider` object with the connection string passed as argument if it contains a SharedAccessKey parameter and pass this object to the transport constructor.]*/
        let authenticationProvider;
        if (cn.SharedAccessKey) {
            authenticationProvider = sak_authentication_provider_1.SharedAccessKeyAuthenticationProvider.fromConnectionString(connStr);
        }
        else if (cn.SharedAccessSignature) {
            /*Codes_SRS_NODE_DEVICE_CLIENT_16_094: [The `fromConnectionString` method shall create a new `SharedAccessSignatureAuthenticationProvider` object with the connection string passed as argument if it contains a SharedAccessSignature parameter and pass this object to the transport constructor.]*/
            authenticationProvider = sas_authentication_provider_1.SharedAccessSignatureAuthenticationProvider.fromSharedAccessSignature(cn.SharedAccessSignature);
        }
        else {
            /*Codes_SRS_NODE_DEVICE_CLIENT_16_093: [The `fromConnectionString` method shall create a new `X509AuthorizationProvider` object with the connection string passed as argument if it contains an X509 parameter and pass this object to the transport constructor.]*/
            authenticationProvider = new x509_authentication_provider_1.X509AuthenticationProvider({
                host: cn.HostName,
                deviceId: cn.DeviceId
            });
        }
        /*Codes_SRS_NODE_DEVICE_CLIENT_05_006: [The fromConnectionString method shall return a new instance of the Client object, as by a call to new Client(new transportCtor(...)).]*/
        return new Client(new transportCtor(authenticationProvider), null, new blob_upload_1.BlobUploadClient(authenticationProvider), new blob_upload_1.DefaultFileUploadApi(authenticationProvider));
    }
    /**
     * @method            module:azure-iot-device.Client.fromSharedAccessSignature
     * @description       Creates an IoT Hub device client from the given
     *                    shared access signature using the given transport type.
     *
     * @param {String}    sharedAccessSignature      A shared access signature which encapsulates "device
     *                                  connect" permissions on an IoT hub.
     * @param {Function}  Transport     A transport constructor.
     *
     * @throws {ReferenceError}         If the connStr parameter is falsy.
     *
     * @returns {module:azure-iothub.Client}
     */
    static fromSharedAccessSignature(sharedAccessSignature, transportCtor) {
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_029: [The fromSharedAccessSignature method shall throw a ReferenceError if the sharedAccessSignature argument is falsy.] */
        if (!sharedAccessSignature)
            throw new ReferenceError('sharedAccessSignature is \'' + sharedAccessSignature + '\'');
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_088: [The `fromSharedAccessSignature` method shall create a new `SharedAccessSignatureAuthorizationProvider` object with the shared access signature passed as argument, and pass this object to the transport constructor.]*/
        const authenticationProvider = sas_authentication_provider_1.SharedAccessSignatureAuthenticationProvider.fromSharedAccessSignature(sharedAccessSignature);
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_030: [The fromSharedAccessSignature method shall return a new instance of the Client object] */
        return new Client(new transportCtor(authenticationProvider), null, new blob_upload_1.BlobUploadClient(authenticationProvider), new blob_upload_1.DefaultFileUploadApi(authenticationProvider));
    }
    /**
     * @method                        module:azure-iot-device.Client.fromAuthenticationMethod
     * @description                   Creates an IoT Hub device client from the given authentication method and using the given transport type.
     * @param authenticationProvider  Object used to obtain the authentication parameters for the IoT hub.
     * @param transportCtor           Transport protocol used to connect to IoT hub.
     */
    static fromAuthenticationProvider(authenticationProvider, transportCtor) {
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_089: [The `fromAuthenticationProvider` method shall throw a `ReferenceError` if the `authenticationProvider` argument is falsy.]*/
        if (!authenticationProvider) {
            throw new ReferenceError('authenticationMethod cannot be \'' + authenticationProvider + '\'');
        }
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_092: [The `fromAuthenticationProvider` method shall throw a `ReferenceError` if the `transportCtor` argument is falsy.]*/
        if (!transportCtor) {
            throw new ReferenceError('transportCtor cannot be \'' + transportCtor + '\'');
        }
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_090: [The `fromAuthenticationProvider` method shall pass the `authenticationProvider` object passed as argument to the transport constructor.]*/
        /*Codes_SRS_NODE_DEVICE_CLIENT_16_091: [The `fromAuthenticationProvider` method shall return a `Client` object configured with a new instance of a transport created using the `transportCtor` argument.]*/
        return new Client(new transportCtor(authenticationProvider), null, new blob_upload_1.BlobUploadClient(authenticationProvider), new blob_upload_1.DefaultFileUploadApi(authenticationProvider));
    }
}
exports.Client = Client;
//# sourceMappingURL=device_client.js.map