/*! Copyright (c) Microsoft. All rights reserved.
 *! Licensed under the MIT license. See LICENSE file in the project root for full license information.
 */
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityDeviceError = exports.DeviceRegistrationFailedError = exports.OperationCancelledError = exports.TwinRequestError = exports.TwinDetachedError = exports.DeviceMethodsDetachedError = exports.CloudToDeviceDetachedError = exports.DeviceTimeoutError = exports.GatewayTimeoutError = exports.BadDeviceResponseError = exports.TimeoutError = exports.PreconditionFailedError = exports.InvalidOperationError = exports.InvalidEtagError = exports.DeviceMessageLockLostError = exports.DeviceAlreadyExistsError = exports.ThrottlingError = exports.TooManyDevicesError = exports.JobNotFoundError = exports.IoTHubSuspendedError = exports.IotHubNotFoundError = exports.ServiceUnavailableError = exports.InternalServerError = exports.MessageTooLargeError = exports.IotHubQuotaExceededError = exports.NotConnectedError = exports.NotImplementedError = exports.UnauthorizedError = exports.FormatError = exports.DeviceNotFoundError = exports.DeviceMaximumQueueDepthExceededError = exports.ArgumentOutOfRangeError = exports.ArgumentError = void 0;
/**
 * Error thrown when an argument is invalid.
 *
 * @augments {Error}
 */
class ArgumentError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'ArgumentError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        // Set the prototype explicitly.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, ArgumentError.prototype);
    }
}
exports.ArgumentError = ArgumentError;
/**
 * Error thrown when an argument has a value that is out of the admissible range.
 *
 * @augments {Error}
 */
class ArgumentOutOfRangeError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'ArgumentOutOfRangeError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, ArgumentOutOfRangeError.prototype);
    }
}
exports.ArgumentOutOfRangeError = ArgumentOutOfRangeError;
/**
 * Error thrown when the message queue for a device is full.
 *
 * @augments {Error}
 */
class DeviceMaximumQueueDepthExceededError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceMaximumQueueDepthExceededError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceMaximumQueueDepthExceededError.prototype);
    }
}
exports.DeviceMaximumQueueDepthExceededError = DeviceMaximumQueueDepthExceededError;
/**
 * Error thrown when a device cannot be found in the IoT Hub instance registry.
 *
 * @augments {Error}
 */
class DeviceNotFoundError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceNotFoundError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceNotFoundError.prototype);
    }
}
exports.DeviceNotFoundError = DeviceNotFoundError;
/**
 * Error thrown when a string that is supposed to have a specific formatting is not formatted properly.
 *
 * @augments {Error}
 */
class FormatError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'FormatError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, FormatError.prototype);
    }
}
exports.FormatError = FormatError;
/**
 * Error thrown when the connection parameters are wrong and the server refused the connection.
 *
 * @augments {Error}
 */
class UnauthorizedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'UnauthorizedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, UnauthorizedError.prototype);
    }
}
exports.UnauthorizedError = UnauthorizedError;
/**
 * Error thrown when a feature is not implemented yet but the placeholder is present.
 *
 * @augments {Error}
 */
class NotImplementedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'NotImplementedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, NotImplementedError.prototype);
    }
}
exports.NotImplementedError = NotImplementedError;
/**
 * Error thrown when the device is disconnected and the operation cannot be completed.
 *
 * @augments {Error}
 */
class NotConnectedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'NotConnectedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, NotConnectedError.prototype);
    }
}
exports.NotConnectedError = NotConnectedError;
/**
 * Error thrown the the Azure IoT hub quota has been exceeded. Quotas are reset periodically, this operation will have to wait until then.
 * To learn more about quotas, see {@link https://docs.microsoft.com/en-us/azure/iot-hub/iot-hub-devguide-quotas-throttling|Azure IoT Hub quotas and throttling}
 *
 * @augments {Error}
 */
class IotHubQuotaExceededError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'IotHubQuotaExceededError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, IotHubQuotaExceededError.prototype);
    }
}
exports.IotHubQuotaExceededError = IotHubQuotaExceededError;
/**
 * Error thrown when the message sent is too large: the maximum size is 256Kb.
 *
 * @augments {Error}
 */
class MessageTooLargeError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'MessageTooLargeError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, MessageTooLargeError.prototype);
    }
}
exports.MessageTooLargeError = MessageTooLargeError;
/**
 * Error thrown when an internal server error occurred. You may have found a bug?
 *
 * @augments {Error}
 */
class InternalServerError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'InternalServerError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, InternalServerError.prototype);
    }
}
exports.InternalServerError = InternalServerError;
/**
 * Error thrown when the service is unavailable. The operation should be retried.
 *
 * @augments {Error}
 */
class ServiceUnavailableError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'ServiceUnavailableError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, ServiceUnavailableError.prototype);
    }
}
exports.ServiceUnavailableError = ServiceUnavailableError;
/**
 * Error thrown when the Azure IoT hub was not found.
 *
 * @augments {Error}
 */
class IotHubNotFoundError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'IotHubNotFoundError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, IotHubNotFoundError.prototype);
    }
}
exports.IotHubNotFoundError = IotHubNotFoundError;
/**
 * Error thrown when IoT Hub has been suspended.
 *
 * @augments {Error}
 */
class IoTHubSuspendedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'IoTHubSuspendedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, IoTHubSuspendedError.prototype);
    }
}
exports.IoTHubSuspendedError = IoTHubSuspendedError;
/**
 * Error thrown when the job with the specified identifier was not found.
 *
 * @augments {Error}
 */
class JobNotFoundError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'JobNotFoundError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, JobNotFoundError.prototype);
    }
}
exports.JobNotFoundError = JobNotFoundError;
/**
 * Error thrown when the maximum number of devices on a specific hub has been reached.
 *
 * @augments {Error}
 */
class TooManyDevicesError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'TooManyDevicesError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, TooManyDevicesError.prototype);
    }
}
exports.TooManyDevicesError = TooManyDevicesError;
/**
 * Error thrown when IoT Hub is throttled due to excessive activity.
 * To learn more about quotas, see {@link https://docs.microsoft.com/en-us/azure/iot-hub/iot-hub-devguide-quotas-throttling|Azure IoT Hub quotas and throttling}
 *
 * @augments {Error}
 */
class ThrottlingError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'ThrottlingError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, ThrottlingError.prototype);
    }
}
exports.ThrottlingError = ThrottlingError;
/**
 * Error thrown when the device id used for device creation already exists in the Device Identity Registry.
 *
 * @augments {Error}
 */
class DeviceAlreadyExistsError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceAlreadyExistsError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceAlreadyExistsError.prototype);
    }
}
exports.DeviceAlreadyExistsError = DeviceAlreadyExistsError;
/**
 * Error thrown when settling a message fails because the lock token associated with the message is lost.
 *
 * @augments {Error}
 */
class DeviceMessageLockLostError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceMessageLockLostError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceMessageLockLostError.prototype);
    }
}
exports.DeviceMessageLockLostError = DeviceMessageLockLostError;
/**
 * Error thrown when the eTag specified is incorrectly formatted or out of date.
 *
 * @augments {Error}
 */
class InvalidEtagError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'InvalidEtagError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, InvalidEtagError.prototype);
    }
}
exports.InvalidEtagError = InvalidEtagError;
/**
 * Error thrown when an operation is attempted but is not allowed.
 *
 * @augments {Error}
 */
class InvalidOperationError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'InvalidOperationError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, InvalidOperationError.prototype);
    }
}
exports.InvalidOperationError = InvalidOperationError;
/**
 * Error thrown when a condition that should have been met in order to execute an operation was not.
 *
 * @augments {Error}
 */
class PreconditionFailedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'PreconditionFailedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, PreconditionFailedError.prototype);
    }
}
exports.PreconditionFailedError = PreconditionFailedError;
/**
 * Error thrown when a timeout occurs
 *
 * @augments {Error}
 */
class TimeoutError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'TimeoutError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, TimeoutError.prototype);
    }
}
exports.TimeoutError = TimeoutError;
/**
 * Error thrown when a device sends a bad response to a device method call.
 *
 * @augments {Error}
 */
class BadDeviceResponseError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'BadDeviceResponseError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, BadDeviceResponseError.prototype);
    }
}
exports.BadDeviceResponseError = BadDeviceResponseError;
/**
 * Error thrown when the IoT Hub instance doesn't process the device method call in time.
 *
 * @augments {Error}
 */
class GatewayTimeoutError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'GatewayTimeoutError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, GatewayTimeoutError.prototype);
    }
}
exports.GatewayTimeoutError = GatewayTimeoutError;
/**
 * Error thrown when the device doesn't process the method call in time.
 *
 * @augments {Error}
 */
class DeviceTimeoutError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceTimeoutError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceTimeoutError.prototype);
    }
}
exports.DeviceTimeoutError = DeviceTimeoutError;
/**
 * Error thrown when the c2d feature stopped working at the transport level, requiring the client to retry starting it.
 *
 * @augments {Error}
 */
class CloudToDeviceDetachedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'CloudToDeviceDetachedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, CloudToDeviceDetachedError.prototype);
    }
}
exports.CloudToDeviceDetachedError = CloudToDeviceDetachedError;
/**
 * Error thrown when the device methods feature stopped working at the transport level, requiring the client to retry starting it.
 *
 * @augments {Error}
 */
class DeviceMethodsDetachedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'DeviceMethodsDetachedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceMethodsDetachedError.prototype);
    }
}
exports.DeviceMethodsDetachedError = DeviceMethodsDetachedError;
/**
 * Error thrown when the twin feature stopped working at the transport level, requiring the client to retry starting it.
 *
 * @augments {Error}
 */
class TwinDetachedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'TwinDetachedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, TwinDetachedError.prototype);
    }
}
exports.TwinDetachedError = TwinDetachedError;
/**
 * Generic error thrown when a twin request fails with an unknown error code.
 *
 * @augments {Error}
 */
class TwinRequestError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'TwinRequestError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, TwinRequestError.prototype);
    }
}
exports.TwinRequestError = TwinRequestError;
/**
 * Error thrown when any operation (local or remote) is cancelled
 *
 * @augments {Error}
 */
class OperationCancelledError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'OperationCancelledError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, OperationCancelledError.prototype);
    }
}
exports.OperationCancelledError = OperationCancelledError;
/**
 * Error thrown when a DPS registration operation fails
 *
 * @augments {Error}
 */
class DeviceRegistrationFailedError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'ProvisioningRegistrationFailedError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, DeviceRegistrationFailedError.prototype);
    }
}
exports.DeviceRegistrationFailedError = DeviceRegistrationFailedError;
/**
 * Error thrown when a low level security device/driver fails.
 *
 * @augments {Error}
 */
class SecurityDeviceError extends Error {
    constructor(message) {
        /* istanbul ignore next */
        super(message);
        this.name = 'SecurityDeviceError';
        this.message = message;
        Error.captureStackTrace(this, this.constructor);
        Object.setPrototypeOf(this, SecurityDeviceError.prototype);
    }
}
exports.SecurityDeviceError = SecurityDeviceError;
//# sourceMappingURL=errors.js.map