// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.Registry = void 0;
const errors = require("./common-core/errors");
const shared_access_signature_1 = require("./common-core/shared_access_signature");
const endpoint = require("./common-core/endpoint");
const version_1 = require("./version");
const https_1 = require("https");
const rest_api_client_1 = require("./common-http/rest_api_client");
const ConnectionString = require("./connection_string");
const twin_1 = require("./twin");
const query_1 = require("./query");
const device_1 = require("./device");
const promise_utils_1 = require("./common-core/promise_utils");
// eslint-disable-next-line @typescript-eslint/no-var-requires
const packageJson = require('../package.json');
/**
 * The Registry class provides access to the IoT Hub device identity service.
 * Users of the SDK should instantiate this class with one of the factory methods:
 * {@link azure-iothub.Registry.fromConnectionString|fromConnectionString} or {@link azure-iothub.Registry.fromSharedAccessSignature|fromSharedAccessSignature}.
 *
 * The protocol used for device identity registry operations is HTTPS.
 */
/*Codes_SRS_NODE_IOTHUB_REGISTRY_05_001: [The Registry constructor shall accept a transport object]*/
class Registry {
    /**
     * @private
     * @constructor
     * @param {Object}  config      An object containing the necessary information to connect to the IoT Hub instance:
     *                              - host: the hostname for the IoT Hub instance
     *                              - sharedAccessSignature: A shared access signature with valid access rights and expiry.
     */
    constructor(config, restApiClient) {
        if (!config) {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_023: [The `Registry` constructor shall throw a `ReferenceError` if the config object is falsy.]*/
            throw new ReferenceError('The \'config\' parameter cannot be \'' + config + '\'');
        }
        else if (!config.host) {
            /*SRS_NODE_IOTHUB_REGISTRY_05_001: [** The `Registry` constructor shall throw an `ArgumentException` if the config object is missing one or more of the following properties:
            - `host`: the IoT Hub hostname
            - `sharedAccessSignature`: shared access signature with the permissions for the desired operations.]*/
            throw new errors.ArgumentError('The \'config\' argument is missing either the host property');
        }
        else if ((!config.sharedAccessSignature) && (!config.tokenCredential)) {
            throw new errors.ArgumentError('The \'config\' argument is missing either the sharedAccessSignature or the tokenCredential property');
        }
        else if ((config.sharedAccessSignature) && (config.tokenCredential)) {
            throw new errors.ArgumentError('The \'config\' argument has both the sharedAccessSignature and the tokenCredential property defined');
        }
        /*SRS_NODE_IOTHUB_REGISTRY_16_024: [The `Registry` constructor shall use the `restApiClient` provided as a second argument if it is provided.]*/
        /*SRS_NODE_IOTHUB_REGISTRY_16_025: [The `Registry` constructor shall use `azure-iothub.RestApiClient` if no `restApiClient` argument is provided.]*/
        // This httpRequestBuilder parameter is used only for unit-testing purposes and should not be used in other situations.
        this._restApiClient = restApiClient || new rest_api_client_1.RestApiClient(config, packageJson.name + '/' + packageJson.version);
        if (this._restApiClient.setOptions) {
            this._restApiClient.setOptions({ http: { agent: new https_1.Agent({ keepAlive: true }) } });
        }
    }
    create(deviceInfo, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            if (!deviceInfo) {
                /*Codes_SRS_NODE_IOTHUB_REGISTRY_07_001: [The `create` method shall throw `ReferenceError` if the `deviceInfo` argument is falsy. **]*/
                throw new ReferenceError('deviceInfo cannot be \'' + deviceInfo + '\'');
            }
            else if (!deviceInfo.deviceId) {
                /*Codes_SRS_NODE_IOTHUB_REGISTRY_07_001: [The create method shall throw ArgumentError if the first argument does not contain a deviceId property.]*/
                throw new errors.ArgumentError('The object \'deviceInfo\' is missing the property: deviceId');
            }
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_026: [The `create` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PUT /devices/<encodeURIComponent(deviceInfo.deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            If-Match: *
            Request-Id: <guid>
      
            <deviceInfo>
            ```]*/
            const path = endpoint.devicePath(encodeURIComponent(deviceInfo.deviceId)) + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            const normalizedDeviceInfo = JSON.parse(JSON.stringify(deviceInfo));
            this._normalizeAuthentication(normalizedDeviceInfo);
            this._restApiClient.executeApiCall('PUT', path, httpHeaders, normalizedDeviceInfo, (err, device, httpResponse) => {
                if (err) {
                    _callback(err);
                }
                else {
                    _callback(null, new device_1.Device(device), httpResponse);
                }
            });
        }, done);
    }
    update(deviceInfo, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            if (!deviceInfo) {
                /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_043: [The `update` method shall throw `ReferenceError` if the `deviceInfo` argument is falsy.]*/
                throw new ReferenceError('deviceInfo cannot be \'' + deviceInfo + '\'');
            }
            else if (!deviceInfo.deviceId) {
                /* Codes_SRS_NODE_IOTHUB_REGISTRY_07_003: [The update method shall throw ArgumentError if the first argument does not contain a deviceId property.]*/
                throw new errors.ArgumentError('The object \'deviceInfo\' is missing the property: deviceId');
            }
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_027: [The `update` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PUT /devices/<encodeURIComponent(deviceInfo.deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            <deviceInfo>
            ```]*/
            const path = endpoint.devicePath(encodeURIComponent(deviceInfo.deviceId)) + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8',
                'If-Match': this._ensureQuoted('*')
            };
            const normalizedDeviceInfo = JSON.parse(JSON.stringify(deviceInfo));
            this._normalizeAuthentication(normalizedDeviceInfo);
            this._restApiClient.executeApiCall('PUT', path, httpHeaders, normalizedDeviceInfo, (err, device, httpResponse) => {
                if (err) {
                    _callback(err);
                }
                else {
                    _callback(null, new device_1.Device(device), httpResponse);
                }
            });
        }, done);
    }
    get(deviceId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_006: [The get method shall throw ReferenceError if the supplied deviceId is falsy.]*/
            if (!deviceId) {
                throw new ReferenceError('deviceId is \'' + deviceId + '\'');
            }
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_028: [The `get` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /devices/<encodeURIComponent(deviceInfo.deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Request-Id: <guid>
            ```]*/
            const path = endpoint.devicePath(encodeURIComponent(deviceId)) + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, (err, device, httpResponse) => {
                if (err) {
                    _callback(err);
                }
                else {
                    _callback(null, new device_1.Device(device), httpResponse);
                }
            });
        }, done);
    }
    list(done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_029: [The `list` method shall construct an HTTP request using information supplied by the caller, as follows:
               ```
               GET /devices?api-version=<version> HTTP/1.1
               Authorization: <config.sharedAccessSignature>
               Request-Id: <guid>
               ```]*/
            const path = endpoint.devicePath('') + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, (err, devices, httpResponse) => {
                if (err) {
                    _callback(err);
                }
                else {
                    _callback(null, devices ? devices.map((device) => new device_1.Device(device)) : [], httpResponse);
                }
            });
        }, done);
    }
    delete(deviceId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_07_007: [The delete method shall throw ReferenceError if the supplied deviceId is falsy.]*/
            if (!deviceId) {
                throw new ReferenceError('deviceId is \'' + deviceId + '\'');
            }
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_030: [The `delete` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            DELETE /devices/<encodeURIComponent(deviceInfo.deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            If-Match: *
            Request-Id: <guid>
            ```]*/
            const path = endpoint.devicePath(encodeURIComponent(deviceId)) + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'If-Match': this._ensureQuoted('*')
            };
            this._restApiClient.executeApiCall('DELETE', path, httpHeaders, null, _callback);
        }, done);
    }
    addDevices(devices, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            this._processBulkDevices(devices, 'create', null, null, null, _callback);
        }, done);
    }
    updateDevices(devices, forceUpdate, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            this._processBulkDevices(devices, null, forceUpdate, 'Update', 'UpdateIfMatchETag', _callback);
        }, done);
    }
    removeDevices(devices, forceRemove, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            this._processBulkDevices(devices, null, forceRemove, 'Delete', 'DeleteIfMatchETag', _callback);
        }, done);
    }
    importDevicesFromBlob(inputBlobContainerUri, outputBlobContainerUri, done) {
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_16_001: [A ReferenceError shall be thrown if importBlobContainerUri is falsy] */
            if (!inputBlobContainerUri)
                throw new ReferenceError('inputBlobContainerUri cannot be falsy');
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_16_002: [A ReferenceError shall be thrown if exportBlobContainerUri is falsy] */
            if (!outputBlobContainerUri)
                throw new ReferenceError('outputBlobContainerUri cannot be falsy');
            /*SRS_NODE_IOTHUB_REGISTRY_16_031: [The `importDeviceFromBlob` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            POST /jobs/create?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            {
              'type': 'import',
              'inputBlobContainerUri': '<input container Uri given as parameter>',
              'outputBlobContainerUri': '<output container Uri given as parameter>'
            }
            ```]*/
            const path = '/jobs/create' + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            const importRequest = {
                'type': 'import',
                'inputBlobContainerUri': inputBlobContainerUri,
                'outputBlobContainerUri': outputBlobContainerUri
            };
            this._restApiClient.executeApiCall('POST', path, httpHeaders, importRequest, _callback);
        }, done);
    }
    importDevicesFromBlobByIdentity(inputBlobContainerUri, outputBlobContainerUri, doneOrIdentity, done) {
        if (typeof doneOrIdentity === 'function') {
            done = doneOrIdentity;
        }
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_07_001: [A ReferenceError shall be thrown if importBlobContainerUri is falsy] */
            if (!inputBlobContainerUri)
                throw new ReferenceError('inputBlobContainerUri cannot be falsy');
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_07_002: [A ReferenceError shall be thrown if exportBlobContainerUri is falsy] */
            if (!outputBlobContainerUri)
                throw new ReferenceError('outputBlobContainerUri cannot be falsy');
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_07_003: [The `importDevicesFromBlobByIdentity` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            POST /jobs/create?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            {
              'type': 'import',
              'inputBlobContainerUri': '<input container Uri given as parameter>',
              'outputBlobContainerUri': '<output container Uri given as parameter>',
              'storageAuthenticationType': 'IdentityBased'
            }
            ```
      
            If a `userAssignedIdentity` is provided, the following additional property shall be in the request body:
            ```Node
            "identity": {
              "userAssignedIdentity": <resource ID for user assigned managed identity given as a parameter>
            }
            ```]*/
            const path = '/jobs/create' + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            const importRequest = {
                'type': 'import',
                'inputBlobContainerUri': inputBlobContainerUri,
                'outputBlobContainerUri': outputBlobContainerUri,
                'storageAuthenticationType': 'IdentityBased',
                ...(typeof doneOrIdentity === 'string' && {
                    'identity': {
                        'userAssignedIdentity': doneOrIdentity
                    }
                })
            };
            this._restApiClient.executeApiCall('POST', path, httpHeaders, importRequest, _callback);
        }, done);
    }
    exportDevicesToBlob(outputBlobContainerUri, excludeKeys, done) {
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_16_004: [A ReferenceError shall be thrown if outputBlobContainerUri is falsy] */
            if (!outputBlobContainerUri)
                throw new ReferenceError('outputBlobContainerUri cannot be falsy');
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_16_032: [** The `exportDeviceToBlob` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            POST /jobs/create?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            {
              'type': 'export',
              'outputBlobContainerUri': '<output container Uri given as parameter>',
              'excludeKeysInExport': '<excludeKeys Boolean given as parameter>'
            }
            ```]*/
            const path = '/jobs/create' + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            const exportRequest = {
                'type': 'export',
                'outputBlobContainerUri': outputBlobContainerUri,
                'excludeKeysInExport': excludeKeys
            };
            this._restApiClient.executeApiCall('POST', path, httpHeaders, exportRequest, _callback);
        }, done);
    }
    exportDevicesToBlobByIdentity(outputBlobContainerUri, excludeKeys, doneOrIdentity, done) {
        if (typeof doneOrIdentity === 'function') {
            done = doneOrIdentity;
        }
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_07_004: [A ReferenceError shall be thrown if outputBlobContainerUri is falsy] */
            if (!outputBlobContainerUri)
                throw new ReferenceError('outputBlobContainerUri cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_07_005: [** The `exportDeviceToBlob` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            POST /jobs/create?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            {
              'type': 'export',
              'outputBlobContainerUri': '<output container Uri given as parameter>',
              'excludeKeysInExport': '<excludeKeys Boolean given as parameter>',
              'storageAuthenticationType': 'IdentityBased'
            }
            ```
      
            If a `userAssignedIdentity` is provided, the following additional property shall be in the request body:
            ```Node
            "identity": {
              "userAssignedIdentity": <resource ID for user assigned managed identity given as a parameter>
            }
            ```]*/
            const path = '/jobs/create' + (0, version_1.versionQueryString)();
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            const exportRequest = {
                'type': 'export',
                'outputBlobContainerUri': outputBlobContainerUri,
                'excludeKeysInExport': excludeKeys,
                'storageAuthenticationType': 'IdentityBased',
                ...(typeof doneOrIdentity === 'string' && {
                    'identity': {
                        'userAssignedIdentity': doneOrIdentity
                    }
                })
            };
            this._restApiClient.executeApiCall('POST', path, httpHeaders, exportRequest, _callback);
        }, done);
    }
    listJobs(done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_037: [The `listJobs` method shall construct an HTTP request using information supplied by the caller, as follows:
                ```
                GET /jobs?api-version=<version> HTTP/1.1
                Authorization: <config.sharedAccessSignature>
                Request-Id: <guid>
                ```]*/
            const path = '/jobs' + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    getJob(jobId, done) {
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_006: [A ReferenceError shall be thrown if jobId is falsy] */
            if (!jobId)
                throw new ReferenceError('jobId cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_038: [The `getJob` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /jobs/<jobId>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Request-Id: <guid>
            ```]*/
            const path = '/jobs/' + jobId + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    cancelJob(jobId, done) {
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_012: [A ReferenceError shall be thrown if the jobId is falsy] */
            if (!jobId)
                throw new ReferenceError('jobId cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_039: [The `cancelJob` method shall construct an HTTP request using information supplied by the caller as follows:
            ```
            DELETE /jobs/<jobId>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Request-Id: <guid>
            ```]*/
            const path = '/jobs/' + jobId + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('DELETE', path, null, null, _callback);
        }, done);
    }
    getTwin(deviceId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_019: [The `getTwin` method shall throw a `ReferenceError` if the `deviceId` parameter is falsy.]*/
            if (!deviceId)
                throw new ReferenceError('the \'deviceId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_049: [The `getTwin` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /twins/<encodeURIComponent(deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Request-Id: <guid>
            ```]*/
            const path = '/twins/' + encodeURIComponent(deviceId) + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, null, null, (err, newTwin, response) => {
                if (err) {
                    _callback(err);
                }
                else {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_036: [The `getTwin` method shall call the `done` callback with a `Twin` object updated with the latest property values stored in the IoT Hub service.]*/
                    _callback(null, new twin_1.Twin(newTwin, this), response);
                }
            });
        }, done);
    }
    getModuleTwin(deviceId, moduleId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_001: [The `getModuleTwin` method shall throw a `ReferenceError` exception if `deviceId`, `moduleId`, or `done` is falsy. ]*/
            if (!deviceId)
                throw new ReferenceError('Argument \'deviceId\' cannot be falsy');
            if (!moduleId)
                throw new ReferenceError('Argument \'moduleId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_002: [The `getModuleTwin` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
              GET /twins/<encodeURIComponent(deviceId)>/modules/<encodeURIComponent(moduleId)>?api-version=<version> HTTP/1.1
              Authorization: <config.sharedAccessSignature>
              Request-Id: <guid>
            ```
            ]*/
            const path = `/twins/${encodeURIComponent(deviceId)}/modules/${encodeURIComponent(moduleId)}${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('GET', path, null, null, (err, newTwin, response) => {
                if (err) {
                    _callback(err);
                }
                else {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_003: [The `getModuleTwin` method shall call the `done` callback with a `Twin` object updated with the latest property values stored in the IoT Hub service. ]*/
                    _callback(null, new twin_1.Twin(newTwin, this), response);
                }
            });
        }, done);
    }
    updateTwin(deviceId, patch, etag, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_044: [The `updateTwin` method shall throw a `ReferenceError` if the `deviceId` argument is `undefined`, `null` or an empty string.]*/
            if (deviceId === null || deviceId === undefined || deviceId === '')
                throw new ReferenceError('deviceId cannot be \'' + deviceId + '\'');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_045: [The `updateTwin` method shall throw a `ReferenceError` if the `patch` argument is falsy.]*/
            if (!patch)
                throw new ReferenceError('patch cannot be \'' + patch + '\'');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_046: [The `updateTwin` method shall throw a `ReferenceError` if the `etag` argument is falsy.]*/
            if (!etag)
                throw new ReferenceError('etag cannot be \'' + etag + '\'');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_048: [The `updateTwin` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PATCH /twins/<encodeURIComponent(deviceId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            If-Match: <etag>
      
            <patch>
            ```]*/
            const path = '/twins/' + encodeURIComponent(deviceId) + (0, version_1.versionQueryString)();
            const headers = {
                'Content-Type': 'application/json; charset=utf-8',
                'If-Match': this._ensureQuoted(etag)
            };
            this._restApiClient.executeApiCall('PATCH', path, headers, patch, (err, newTwin, response) => {
                if (err) {
                    _callback(err);
                }
                else {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_050: [The `updateTwin` method shall call the `done` callback with a `Twin` object updated with the latest property values stored in the IoT Hub service.]*/
                    _callback(null, new twin_1.Twin(newTwin, this), response);
                }
            });
        }, done);
    }
    updateModuleTwin(deviceId, moduleId, patch, etag, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_004: [The `updateModuleTwin` method shall throw a `ReferenceError` exception if `deviceId`, `moduleId`, `patch`, `etag`,or `done` is falsy. ]*/
            if (!deviceId)
                throw new ReferenceError('Argument \'deviceId\' cannot be falsy');
            if (!moduleId)
                throw new ReferenceError('Argument \'moduleId\' cannot be falsy');
            if (!patch)
                throw new ReferenceError('Argument \'patch\' cannot be falsy');
            if (!etag)
                throw new ReferenceError('Argument \'etag\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_005: [The `updateModuleTwin` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PATCH /twins/<encodeURIComponent(deviceId)>/modules/<encodeURIComponent(moduleId)>?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
            If-Match: <etag>
            <patch>
            ```
            ]*/
            const path = `/twins/${encodeURIComponent(deviceId)}/modules/${encodeURIComponent(moduleId)}${(0, version_1.versionQueryString)()}`;
            const headers = {
                'Content-Type': 'application/json; charset=utf-8',
                'If-Match': this._ensureQuoted(etag)
            };
            this._restApiClient.executeApiCall('PATCH', path, headers, patch, (err, newTwin, response) => {
                if (err) {
                    _callback(err);
                }
                else {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_006: [The `updateModuleTwin` method shall call the `done` callback with a `Twin` object updated with the latest property values stored in the IoT Hub service. ]*/
                    _callback(null, new twin_1.Twin(newTwin, this), response);
                }
            });
        }, done);
    }
    /**
     * @method              module:azure-iothub.Registry#createQuery
     * @description         Creates a query that can be run on the IoT Hub instance to find information about devices or jobs.
     * @param {String}      sqlQuery   The query written as an SQL string.
     * @param {Number}      pageSize   The desired number of results per page (optional. default: 1000, max: 10000).
     *
     * @throws {ReferenceError}        If the sqlQuery argument is falsy.
     * @throws {TypeError}             If the sqlQuery argument is not a string or the pageSize argument not a number, null or undefined.
     */
    createQuery(sqlQuery, pageSize) {
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_051: [The `createQuery` method shall throw a `ReferenceError` if the `sqlQuery` argument is falsy.]*/
        if (!sqlQuery)
            throw new ReferenceError('sqlQuery cannot be \'' + sqlQuery + '\'');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_052: [The `createQuery` method shall throw a `TypeError` if the `sqlQuery` argument is not a string.]*/
        if (typeof sqlQuery !== 'string')
            throw new TypeError('sqlQuery must be a string');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_053: [The `createQuery` method shall throw a `TypeError` if the `pageSize` argument is not `null`, `undefined` or a number.]*/
        if (pageSize !== null && pageSize !== undefined && typeof pageSize !== 'number')
            throw new TypeError('pageSize must be a number or be null or undefined');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_054: [The `createQuery` method shall return a new `Query` instance initialized with the `sqlQuery` and the `pageSize` argument if specified.]*/
        return new query_1.Query(this._executeQueryFunc(sqlQuery, pageSize), this);
    }
    getRegistryStatistics(done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            const path = '/statistics/devices' + (0, version_1.versionQueryString)();
            this._restApiClient.executeApiCall('GET', path, {}, null, _callback);
        }, done);
    }
    addConfiguration(configuration, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_007: [The `addConfiguration` method shall throw a `ReferenceError` exception if `configuration` or `done` is falsy. ]*/
            if (!configuration)
                throw new ReferenceError('configuration cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_008: [The `addConfiguration` method shall throw an `ArgumentError` exception if `configuration.id` is falsy. ]*/
            if (!configuration.id)
                throw new errors.ArgumentError('configuration object is missing id property');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_009: [The `addConfiguration` method shall set `configuration.schemaVersion` to '1.0' if it is not already set. ]*/
            if (!configuration.schemaVersion) {
                configuration.schemaVersion = '1.0';
            }
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_010: [The `addConfiguration` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PUT /configurations/<encodeURIComponent(configuration.id)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            <configuration>
            ```
            ]*/
            const path = `/configurations/${encodeURIComponent(configuration.id)}${(0, version_1.versionQueryString)()}`;
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            this._restApiClient.executeApiCall('PUT', path, httpHeaders, configuration, _callback);
        }, done);
    }
    getConfiguration(configurationId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_011: [The `getConfiguration` method shall throw a `ReferenceError` exception if `configurationId` is falsy. ]*/
            if (!configurationId)
                throw new ReferenceError('Argument \'configurationId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_012: [The `getConfiguration` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /configurations/<encodeURIComponent(configurationId)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Request-Id: <guid>
            ```
            ]*/
            const path = `/configurations/${encodeURIComponent(configurationId)}${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    getConfigurations(done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_014: [The `getConfigurations` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /configurations?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Request-Id: <guid>
            ```
            ]*/
            const path = `/configurations${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    _updateConfiguration(configuration, forceUpdateOrDone, done) {
        let forceUpdate;
        if (typeof (forceUpdateOrDone) === 'function') {
            forceUpdate = false;
            done = forceUpdateOrDone;
        }
        else {
            forceUpdate = forceUpdateOrDone;
        }
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_015: [The `updateConfiguration` method shall throw a `ReferenceError` exception if `configuration` or `done` is falsy. ]*/
        if (!configuration)
            throw new ReferenceError('Argument \'configuration\' cannot be falsy');
        if (!done)
            throw new ReferenceError('Argument \'done\' cannot be falsy');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_016: [The `updateConfiguration` method shall throw an `ArgumentError` exception if `forceUpdate` is falsy and `configuration.etag` is also falsy. ]*/
        if (!forceUpdate && !configuration.etag) {
            throw new errors.ArgumentError('The ETag should be set while updating the Configuration.');
        }
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_017: [The `updateConfiguration` method shall throw an `ArgumentError` exception if `configuration.id` is falsy. ]*/
        if (!configuration.id)
            throw new errors.ArgumentError('configuration object is missing id property');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_018: [The `updateConfiguration` method shall set ``configuration.schemaVersion` to '1.0' if it is not already set. ]*/
        if (!configuration.schemaVersion) {
            configuration.schemaVersion = '1.0';
        }
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_019: [The `updateConfiguration` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        PUT </configurations/<encodeURIComponent(configurationId)>?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Content-Type: application/json; charset=utf-8
        If-Match: <etag | *>
        Request-Id: <guid>
    
        <configuration>
        ```
        ]*/
        const path = `/configurations/${encodeURIComponent(configuration.id)}${(0, version_1.versionQueryString)()}`;
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_020: [If `forceUpdate` is not truthy, the `updateConfigurationMethod` shall put the `etag` parameter into the `If-Match` header value. ]*/
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_021: [If `forceUpdate` is truthy, the `updateConfiguration` method shall put `*` into the `If-Match` header value. ]*/
        const httpHeaders = {
            'Content-Type': 'application/json; charset=utf-8',
            'If-Match': this._ensureQuoted(forceUpdate ? '*' : configuration.etag)
        };
        this._restApiClient.executeApiCall('PUT', path, httpHeaders, configuration, done);
    }
    updateConfiguration(configuration, forceUpdateOrDone, done) {
        const callback = done || ((typeof forceUpdateOrDone === 'function') ? forceUpdateOrDone : undefined);
        if (callback) {
            return this._updateConfiguration(configuration, forceUpdateOrDone, done);
        }
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => this._updateConfiguration(configuration, forceUpdateOrDone, _callback));
    }
    removeConfiguration(configurationId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_022: [The `removeConfiguration` method shall throw a `ReferenceError` exception if `configurationId` or `done` is falsy. ]*/
            if (!configurationId)
                throw new ReferenceError('Argument \'configurationId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_023: [The `removeConfiguration` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            DELETE /configurations/<encodeURIComponent(configurationId)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Request-Id: <guid>
            ```
            ]*/
            const path = `/configurations/${encodeURIComponent(configurationId)}${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('DELETE', path, null, null, _callback);
        }, done);
    }
    applyConfigurationContentOnDevice(deviceId, content, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_024: [The `applyConfigurationContentOnDevice` method shall throw a `ReferenceError` exception if `deviceId`, `content`, or `done` is falsy. ]*/
            if (!deviceId)
                throw new ReferenceError('Argument \'deviceId\' cannot be falsy');
            if (!content)
                throw new ReferenceError('Argument \'content\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_025: [The `applyConfigurationContentOnDevice` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            POST /devices/<encodeURIComponent(deviceId)>/applyConfigurationContent?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            <content>
            ```
            ]*/
            const path = `${endpoint.devicePath(encodeURIComponent(deviceId))}/applyConfigurationContent${(0, version_1.versionQueryString)()}`;
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            this._restApiClient.executeApiCall('POST', path, httpHeaders, content, _callback);
        }, done);
    }
    addModule(module, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_026: [The `addModule` method shall throw a `ReferenceError` exception if `module` or `done` is falsy. ]*/
            if (!module)
                throw new ReferenceError('Argument \'module\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_027: [The `addModule` method shall throw an `ArgumentError` exception if `module.deviceId` or `module.moduleId` is falsy. ]*/
            if (!module.deviceId)
                throw new errors.ArgumentError('deviceId property is missing from module object');
            if (!module.moduleId)
                throw new errors.ArgumentError('moduleId property is missing from module object');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_028: [The `addModule` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            PUT /devices/<encodeURIComponent(module.deviceId)>/modules/<encodeURIComponent(module.moduleId)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            Request-Id: <guid>
      
            <module>
            ```
            ]*/
            const preparedModule = JSON.parse(JSON.stringify(module));
            this._normalizeAuthentication(preparedModule);
            const path = `${endpoint.modulePath(encodeURIComponent(preparedModule.deviceId), encodeURIComponent(preparedModule.moduleId))}${(0, version_1.versionQueryString)()}`;
            const httpHeaders = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            this._restApiClient.executeApiCall('PUT', path, httpHeaders, preparedModule, _callback);
        }, done);
    }
    getModulesOnDevice(deviceId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_029: [The `getModulesOnDevice` method shall throw a `ReferenceError` exception if `deviceId` or `done` is falsy. ]*/
            if (!deviceId)
                throw new ReferenceError('Argument \'deviceId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_030: [The `getModulesOnDevice` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            GET /devices/<encodeURIComponent(deviceId)>/modules?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Request-Id: <guid>
            ```
            ]*/
            const path = `${endpoint.devicePath(encodeURIComponent(deviceId))}/modules${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    getModule(deviceId, moduleId, done) {
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_031: [The `getModule` method shall throw a `ReferenceError` exception if `deviceId`, `moduleId`, or `done` is falsy. ]*/
            if (!deviceId)
                throw new ReferenceError('Argument \'deviceId\' cannot be falsy');
            if (!moduleId)
                throw new ReferenceError('Argument \'moduleId\' cannot be falsy');
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_032: [The `getModule` method shall construct an HTTP request using information supplied by the caller, as follows:
            ```
            get /devices/<encodeURIComponent(deviceId)>/modules/<encodeURIComponent(moduleId)>?api-version=<version> HTTP/1.1
            Authorization: <sharedAccessSignature>
            Request-Id: <guid>
            ```
            ]*/
            const path = `${endpoint.modulePath(encodeURIComponent(deviceId), encodeURIComponent(moduleId))}${(0, version_1.versionQueryString)()}`;
            this._restApiClient.executeApiCall('GET', path, null, null, _callback);
        }, done);
    }
    _updateModule(module, forceUpdateOrDone, done) {
        let forceUpdate;
        if (typeof (forceUpdateOrDone) === 'function') {
            forceUpdate = false;
            done = forceUpdateOrDone;
        }
        else {
            forceUpdate = forceUpdateOrDone;
        }
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_033: [The `updateModule` method shall throw a `ReferenceError` exception if `module` or `done` is falsy. ]*/
        if (!module)
            throw new ReferenceError('Argument \'module\' cannot be falsy');
        if (!done)
            throw new ReferenceError('Argument \'done\' cannot be falsy');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_034: [The `updateModule` method shall throw an `ArgumentError` exception if `module.deviceId` or `module.moduleId` is falsy. ]*/
        if (!module.moduleId)
            throw new errors.ArgumentError('moduleId property is missing from module object');
        if (!module.deviceId)
            throw new errors.ArgumentError('deviceId property is missing from module object');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_035: [The `updateModule` method shall throw an `ArgumentError` exception if `forceUpdate` is falsy and `module.etag` is falsy. ]*/
        if (!forceUpdate && !module.etag) {
            throw new errors.ArgumentError('The ETag should be set while updating the Module.');
        }
        const preparedModule = JSON.parse(JSON.stringify(module));
        this._normalizeAuthentication(preparedModule);
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_036: [If `forceUpdate` is not truthy, the `updateModule` shall put the `etag` parameter into the `If-Match` header value. ]*/
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_037: [If `forceUpdate` is truthy, the `updateModule` method shall put `*` into the `If-Match` header value. ]*/
        const httpHeaders = {
            'Content-Type': 'application/json; charset=utf-8',
            'If-Match': this._ensureQuoted(forceUpdate ? '*' : preparedModule.etag)
        };
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_038: [The `updateModule` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        PUT /devices/<encodeURIComponent(module.deviceId)>/modules/<encodeURIComponent(module.moduleId)>?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Content-Type: application/json; charset=utf-8
        If-Match: <etag | *
        Request-Id: <guid>
    
        <module>
        ```
        ]*/
        const path = `${endpoint.modulePath(encodeURIComponent(preparedModule.deviceId), encodeURIComponent(preparedModule.moduleId))}${(0, version_1.versionQueryString)()}`;
        this._restApiClient.executeApiCall('PUT', path, httpHeaders, preparedModule, done);
    }
    updateModule(module, forceUpdateOrDone, done) {
        const callback = done || ((typeof forceUpdateOrDone === 'function') ? forceUpdateOrDone : undefined);
        if (callback) {
            return this._updateModule(module, forceUpdateOrDone, done);
        }
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            this._updateModule(module, forceUpdateOrDone, _callback);
        });
    }
    _removeModule(moduleOrDeviceId, doneOrModuleId, done) {
        let moduleId;
        let deviceId;
        let etag;
        if (moduleOrDeviceId && (moduleOrDeviceId.moduleId)) { // can't do "instanceof Module" at runtime because Module is an interface
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_041: [if a `Module` object is passed in, `removeModule` shall use the `deviceId`, `moduleId`, and `etag` from the `Module` object.]*/
            done = doneOrModuleId;
            const module = moduleOrDeviceId;
            deviceId = module.deviceId;
            moduleId = module.moduleId;
            etag = module.etag;
        }
        else {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_042: [if a `deviceId` and `moduleId` are passed in, `removeModule` shall use those values and the `etag` shall be `*`.]*/
            deviceId = moduleOrDeviceId;
            moduleId = doneOrModuleId;
            etag = '*';
        }
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_039: [The `removeModule` method shall throw a `ReferenceError` exception if `deviceId`, `moduleId`, or `done` is falsy. ]*/
        if (!deviceId)
            throw new ReferenceError('\'deviceId\' cannot be falsy');
        if (!moduleId)
            throw new ReferenceError('\'moduleId\' cannot be falsy');
        if (!done)
            throw new ReferenceError('Argument \'done\' cannot be falsy');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_043: [The `removeModule` method shall throw an `ArgumentError` if `deviceId` or `moduleId` parameters are not strings.]*/
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_044: [The `removeModule` method shall throw an `ArgumentError` if the `done` parameter is not a function.]*/
        if (typeof deviceId !== 'string')
            throw new errors.ArgumentError('\'deviceId\' must be a string');
        if (typeof moduleId !== 'string')
            throw new errors.ArgumentError('\'moduleId\' must be a string');
        if (typeof (done) !== 'function')
            throw new errors.ArgumentError('\'done\' must be a function');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_18_040: [The `removeModule` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        DELETE /devices/<encodeURIComponent(deviceId)>/modules/<encodeURIComponent(moduleId)>?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Request-Id: <guid>
        If-Match: "<etag>"
        ```
        ]*/
        const httpHeaders = {
            'If-Match': this._ensureQuoted(etag)
        };
        const path = `${endpoint.modulePath(encodeURIComponent(deviceId), encodeURIComponent(moduleId))}${(0, version_1.versionQueryString)()}`;
        this._restApiClient.executeApiCall('DELETE', path, httpHeaders, null, done);
    }
    removeModule(moduleOrDeviceId, doneOrModuleId, done) {
        const callback = done || ((typeof doneOrModuleId === 'function') ? doneOrModuleId : undefined);
        if (callback) {
            return this._removeModule(moduleOrDeviceId, doneOrModuleId, callback);
        }
        return (0, promise_utils_1.httpCallbackToPromise)((_callback) => {
            this._removeModule(moduleOrDeviceId, doneOrModuleId, _callback);
        });
    }
    _bulkOperation(devices, done) {
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_011: [The `addDevices` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        POST /devices?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Content-Type: application/json; charset=utf-8
        Request-Id: <guid>
    
        <stringified array supplied by the argument devices annotated with importMode property and deviceId property replaced by id>
        ```
        ]*/
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_013: [The `updateDevices` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        POST /devices?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Content-Type: application/json; charset=utf-8
        Request-Id: <guid>
    
        <list supplied by the argument devices annotated with importMode property and deviceId property replaced by id>
        ```
        ]*/
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_018: [The `removeDevices` method shall construct an HTTP request using information supplied by the caller, as follows:
        ```
        POST /devices?api-version=<version> HTTP/1.1
        Authorization: <sharedAccessSignature>
        Content-Type: application/json; charset=utf-8
        Request-Id: <guid>
    
        <stringified array supplied by the argument devices annotated with importMode property and deviceId property replaced by id>
        ```
        ]*/
        const path = '/devices' + (0, version_1.versionQueryString)();
        const httpHeaders = {
            'Content-Type': 'application/json; charset=utf-8'
        };
        this._restApiClient.executeApiCall('POST', path, httpHeaders, devices, done);
    }
    _processBulkDevices(devices, operation, force, forceTrueAlternative, forceFalseAlternative, done) {
        if (!devices) {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_004: [The `addDevices` method shall throw `ReferenceError` if the `devices` argument is falsy.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_025: [The `updateDevices` method shall throw `ReferenceError` if the `devices` argument is falsy.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_006: [The `removeDevices` method shall throw `ReferenceError` if the deviceInfo is falsy.]*/
            throw new ReferenceError('devices cannot be \'' + devices + '\'');
        }
        else if (!Array.isArray(devices)) {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_021: [The `addDevices` method shall throw `ArgumentError` if devices is NOT an array.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_020: [The `updateDevices` method shall throw `ArgumentError` if devices is NOT an array.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_019: [The `removeDevices` method shall throw `ArgumentError` if devices is NOT an array.]*/
            throw new errors.ArgumentError('devices must be an array');
        }
        else if ((devices.length === 0) || (devices.length > 100)) {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_014: [The `addDevices` method shall throw `ArgumentError` if devices.length == 0  or is greater than 100.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_015: [The `updateDevices` method shall throw `ArgumentError` if devices.length == 0  or is greater than 100.]*/
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_016: [The `removeDevices` method shall throw `ArgumentError` if devices.length == 0  or is greater than 100.]*/
            throw new errors.ArgumentError('The device array has an invalid size of ' + devices.length);
        }
        else {
            let importMode;
            if (operation === null) {
                //
                // The api utilizes a force parameter.  Check to insure it's present and a boolean.
                //
                if ((typeof force) !== 'boolean') {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_026: [The `updateDevices` method shall throw `ReferenceError` if the `forceUpdate` parameter is null or undefined.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_027: [The `removeDevices` method shall throw `ReferenceError` if the `forceRemove` parameter is null or undefined.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_024: [The `updateDevices` method shall throw `ReferenceError` if the `forceUpdate` parameter is NOT typeof boolean.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_023: [The `removeDevices` method shall throw `ReferenceError` if the `forceRemove` parameter is NOT typeof boolean.]*/
                    throw new ReferenceError('force parameter must be present and a boolean');
                }
                else {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_008: [If the `forceUpdate` parameter is true importMode will be set to `Update` otherwise it will be set to `UpdateIfMatchETag`.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_007: [If the `forceRemove` parameter is true then importMode will be set to `Delete` otherwise it will be set to `DeleteIfMatchETag`.]*/
                    importMode = force ? forceTrueAlternative : forceFalseAlternative;
                }
            }
            else {
                /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_009: [The `addDevices` method shall utilize an importMode = `create`.]*/
                importMode = operation;
            }
            const bulkArray = [];
            devices.forEach((currentDevice) => {
                if (!currentDevice.deviceId) {
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_010: [The `addDevices` method shall throw `ArgumentError` if any elements of devices do NOT contain a `deviceId` property.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_012: [The `updateDevices` method shall throw `ArgumentError` if any elements of devices do NOT contain a `deviceId` property.]*/
                    /*Codes_SRS_NODE_IOTHUB_REGISTRY_06_017: [The `removeDevices` method shall throw `ArgumentError` if any elements of devices do NOT contain a `deviceId` property.]*/
                    throw new errors.ArgumentError('The object is missing the property: deviceId');
                }
                else {
                    //
                    // And now remove the device id and put it back as id.
                    //
                    const actualDeviceId = currentDevice.deviceId;
                    const preparedDevice = JSON.parse(JSON.stringify(currentDevice));
                    delete preparedDevice.deviceId;
                    preparedDevice.id = actualDeviceId;
                    preparedDevice.importMode = importMode;
                    this._normalizeAuthentication(preparedDevice);
                    bulkArray.push(preparedDevice);
                }
            });
            this._bulkOperation(bulkArray, done);
        }
    }
    _executeQueryFunc(sqlQuery, pageSize) {
        return (continuationToken, done) => {
            /*Codes_SRS_NODE_IOTHUB_REGISTRY_16_057: [The `_executeQueryFunc` method shall construct an HTTP request as follows:
            ```
            POST /devices/query?api-version=<version> HTTP/1.1
            Authorization: <config.sharedAccessSignature>
            Content-Type: application/json; charset=utf-8
            x-ms-continuation: continuationToken
            x-ms-max-item-count: pageSize
            Request-Id: <guid>
      
            {
              query: <sqlQuery>
            }
            ```]*/
            const path = '/devices/query' + (0, version_1.versionQueryString)();
            const headers = {
                'Content-Type': 'application/json; charset=utf-8'
            };
            if (continuationToken) {
                headers['x-ms-continuation'] = continuationToken;
            }
            if (pageSize) {
                headers['x-ms-max-item-count'] = pageSize;
            }
            const query = {
                query: sqlQuery
            };
            this._restApiClient.executeApiCall('POST', path, headers, query, done);
        };
    }
    _normalizeAuthentication(deviceInfo) {
        if (!Object.prototype.hasOwnProperty.call(deviceInfo, 'authentication')) {
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_06_028: [A device information with no authentication will be normalized with the following authentication:
            authentication : {
              type: 'sas',
              symmetricKey: {
                primaryKey: '',
                secondaryKey: ''
              }
            }
            ] */
            deviceInfo.authentication = {
                type: 'sas',
                symmetricKey: {
                    primaryKey: '',
                    secondaryKey: ''
                }
            };
            /* Codes_SRS_NODE_IOTHUB_REGISTRY_06_029: [** A device information with an authentication object that contains a `type` property is considered normalized.] */
        }
        else if (!Object.prototype.hasOwnProperty.call(deviceInfo.authentication, 'type')) {
            if (deviceInfo.authentication.x509Thumbprint && (deviceInfo.authentication.x509Thumbprint.primaryThumbprint || deviceInfo.authentication.x509Thumbprint.secondaryThumbprint)) {
                /* Codes_SRS_NODE_IOTHUB_REGISTRY_06_030: [A device information with an authentication object that contains the x509Thumbprint property with at least one of `primaryThumbprint` or `secondaryThumbprint` sub-properties will be normalized with a `type` property with value "selfSigned".] */
                deviceInfo.authentication.type = 'selfSigned';
            }
            else {
                /* Codes_SRS_NODE_IOTHUB_REGISTRY_06_031: [A device information with an authentication object that doesn't contain the x509Thumbprint property will be normalized with a `type` property with value "sas".] */
                deviceInfo.authentication.type = 'sas';
            }
        }
    }
    _ensureQuoted(eTag) {
        const tagLength = eTag.length;
        if (tagLength === 0) {
            return '""';
        }
        else if ((eTag.slice(0, 1) === '"') && (eTag.slice(tagLength - 1, tagLength) === '"')) {
            return eTag;
        }
        return '"' + eTag + '"';
    }
    /**
     * @method          module:azure-iothub.Registry.fromConnectionString
     * @description     Constructs a Registry object from the given connection string.
     * @static
     * @param {String}  value       A connection string which encapsulates the
     *                              appropriate (read and/or write) Registry
     *                              permissions.
     * @returns {module:azure-iothub.Registry}
     */
    static fromConnectionString(value) {
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_008: [The `fromConnectionString` method shall throw `ReferenceError` if the value argument is falsy.]*/
        if (!value)
            throw new ReferenceError('value is \'' + value + '\'');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_009: [The `fromConnectionString` method shall derive and transform the needed parts from the connection string in order to create a `config` object for the constructor (see `SRS_NODE_IOTHUB_REGISTRY_05_001`).]*/
        const cn = ConnectionString.parse(value);
        const config = {
            host: cn.HostName,
            sharedAccessSignature: shared_access_signature_1.SharedAccessSignature.create(cn.HostName, cn.SharedAccessKeyName, cn.SharedAccessKey, Date.now()),
            tokenCredential: undefined
        };
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_010: [The `fromConnectionString` method shall return a new instance of the `Registry` object.]*/
        return new Registry(config);
    }
    /**
     * @method            module:azure-iothub.Registry.fromSharedAccessSignature
     * @description       Constructs a Registry object from the given shared access signature.
     * @static
     *
     * @param {String}    value     A shared access signature which encapsulates
     *                              the appropriate (read and/or write) Registry
     *                              permissions.
     * @returns {module:azure-iothub.Registry}
     */
    static fromSharedAccessSignature(value) {
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_011: [The `fromSharedAccessSignature` method shall throw ReferenceError if the value argument is falsy.]*/
        if (!value)
            throw new ReferenceError('value is \'' + value + '\'');
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_012: [The `fromSharedAccessSignature` method shall derive and transform the needed parts from the shared access signature in order to create a `config` object for the constructor (see `SRS_NODE_IOTHUB_REGISTRY_05_001`).]*/
        const sas = shared_access_signature_1.SharedAccessSignature.parse(value);
        const config = {
            host: sas.sr,
            sharedAccessSignature: sas.toString(),
            tokenCredential: undefined
        };
        /*Codes_SRS_NODE_IOTHUB_REGISTRY_05_013: [The fromSharedAccessSignature method shall return a new instance of the `Registry` object.]*/
        return new Registry(config);
    }
    /**
     * @method            module:azure-iothub.Registry.fromTokenCredential
     * @description       Constructs a Registry object from the given Azure TokenCredential.
     * @static
     *
     * @param {String}    hostName                  Host name of the Azure service.
     * @param {String}    tokenCredential           An Azure TokenCredential used to authenticate
     *                                              with the Azure  service
     *
     * @throws  {ReferenceError}  If the tokenCredential argument is falsy.
     *
     * @returns {module:azure-iothub.Registry}
     */
    static fromTokenCredential(hostName, tokenCredential) {
        const config = {
            host: hostName,
            tokenCredential,
            tokenScope: 'https://iothubs.azure.net/.default'
        };
        return new Registry(config);
    }
}
exports.Registry = Registry;
//# sourceMappingURL=registry.js.map