// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.
'use strict';
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceReceiver = void 0;
const promise_utils_1 = require("./common-core/promise_utils");
const amqp_message_1 = require("./common-amqp/amqp_message");
const events_1 = require("events");
class ServiceReceiver extends events_1.EventEmitter {
    constructor(receiver) {
        super();
        this._receiver = receiver;
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_001: [The constructor shall subscribe to the `message` event of the `ReceiverLink` object passed as argument.]*/
        this._receiver.on('message', (amqpMessage) => {
            /*Codes_SRS_NODE_SERVICE_RECEIVER_16_006: [The `ServiceReceiver` class shall convert any `AmqpMessage` received with the `message` event from the `ReceiverLink` object into `Message` objects and emit a `message` event with that newly created `Message` object for argument.]*/
            this.emit('message', amqp_message_1.AmqpMessage.toMessage(amqpMessage));
        });
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_002: [The constructor shall subscribe to the `error` event of the `ReceiverLink` object passed as argument.]*/
        this._receiver.on('error', (err) => {
            /*Codes_SRS_NODE_SERVICE_RECEIVER_16_007: [Any error event received from the `ReceiverLink` object shall be forwarded as is.]*/
            this.emit('error', err);
        });
    }
    complete(message, done) {
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            /*Codes_SRS_NODE_SERVICE_RECEIVER_16_003: [The `complete` method shall call the `complete` method on the `ReceiverLink` object and pass it the `AmqpMessage` stored within the `transportObj` property of the `Message` object as well as the `done` callback passed as argument.]*/
            this._receiver.complete(message.transportObj, _callback);
        }, done);
    }
    abandon(message, done) {
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_004: [The `abandon` method shall call the `abandon` method on the `ReceiverLink` object and pass it the `AmqpMessage` stored within the `transportObj` property of the `Message` object as well as the `done` callback passed as argument.]*/
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            this._receiver.abandon(message.transportObj, _callback);
        }, done);
    }
    reject(message, done) {
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_005: [The `reject` method shall call the `reject` method on the `ReceiverLink` object and pass it the `AmqpMessage` stored within the `transportObj` property of the `Message` object as well as the `done` callback passed as argument.]*/
        return (0, promise_utils_1.callbackToPromise)((_callback) => {
            this._receiver.reject(message.transportObj, _callback);
        }, done);
    }
    detach(callback) {
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_008: [The `detach` method shall call the `detach` method on the `ReceiverLink` object and pass it its `callback` argument.]*/
        return (0, promise_utils_1.errorCallbackToPromise)((_callback) => {
            this._receiver.detach(_callback);
        }, callback);
    }
    forceDetach(err) {
        /*Codes_SRS_NODE_SERVICE_RECEIVER_16_009: [The `forceDetach` method shall call the `forceDetach` method on the `ReceiverLink` object and pass it its `err` argument.]*/
        this._receiver.forceDetach(err);
    }
}
exports.ServiceReceiver = ServiceReceiver;
//# sourceMappingURL=service_receiver.js.map