"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const float16_1 = require("@petamoriken/float16");
class OctetstreamCodec {
    getMediaType() {
        return "application/octet-stream";
    }
    bytesToValue(bytes, schema, parameters) {
        var _a;
        const bigendian = parameters.byteorder ? parameters.byteorder === "bigendian" : true;
        let signed = parameters.signed ? parameters.signed === "true" : false;
        if (parameters.length && parseInt(parameters.length) !== bytes.length) {
            throw new Error("Lengths do not match, required: " + parameters.length + " provided: " + bytes.length);
        }
        let dataLength = bytes.length;
        let dataType = schema.type;
        if (/(short|(u)?int(8|16|32)?$|float(16|32|64)?|byte)/.test(dataType.toLowerCase())) {
            const typeSem = /(u)?(short|int|float|byte)(8|16|32|64)?/.exec(dataType.toLowerCase());
            if (typeSem) {
                signed = typeSem[1] === undefined;
                dataType = typeSem[2];
                dataLength = (_a = +typeSem[3] / 8) !== null && _a !== void 0 ? _a : bytes.length;
            }
        }
        switch (dataType) {
            case "boolean":
                return !bytes.every((val) => val === 0);
            case "byte":
            case "short":
            case "int":
            case "integer":
                switch (dataLength) {
                    case 1:
                        return signed ? bytes.readInt8(0) : bytes.readUInt8(0);
                    case 2:
                        return bigendian
                            ? signed
                                ? bytes.readInt16BE(0)
                                : bytes.readUInt16BE(0)
                            : signed
                                ? bytes.readInt16LE(0)
                                : bytes.readUInt16LE(0);
                    case 4:
                        return bigendian
                            ? signed
                                ? bytes.readInt32BE(0)
                                : bytes.readUInt32BE(0)
                            : signed
                                ? bytes.readInt32LE(0)
                                : bytes.readUInt32LE(0);
                    default: {
                        let result = 0;
                        let negative;
                        if (bigendian) {
                            result = bytes.reduce((prev, curr) => prev << (8 + curr));
                            negative = bytes.readInt8(0) < 0;
                        }
                        else {
                            result = bytes.reduceRight((prev, curr) => prev << (8 + curr));
                            negative = bytes.readInt8(dataLength - 1) < 0;
                        }
                        if (signed && negative) {
                            result -= 1 << (8 * dataLength);
                        }
                        if (!Number.isSafeInteger(result)) {
                            console.warn("[core/octetstream-codec]", "Result is not a safe integer");
                        }
                        return result;
                    }
                }
            case "float":
            case "double":
            case "number":
                switch (dataLength) {
                    case 2:
                        return (0, float16_1.getFloat16)(new DataView(bytes.buffer), bytes.byteOffset, !bigendian);
                    case 4:
                        return bigendian ? bytes.readFloatBE(0) : bytes.readFloatLE(0);
                    case 8:
                        return bigendian ? bytes.readDoubleBE(0) : bytes.readDoubleLE(0);
                    default:
                        throw new Error("Wrong buffer length for type 'number', must be 2, 4, 8, or is " + dataLength);
                }
            case "string":
                return bytes.toString(parameters.charset);
            case "array":
            case "object":
                throw new Error("Unable to handle object type " + dataType);
            case "null":
                return null;
            default:
                throw new Error("Unable to handle object type " + dataType);
        }
    }
    valueToBytes(value, schema, parameters) {
        var _a;
        if (!parameters.length) {
            console.warn("[core/octetstream-codec]", "Missing 'length' parameter necessary for write. I'll do my best");
        }
        const bigendian = parameters.byteorder ? parameters.byteorder === "bigendian" : true;
        let signed = parameters.signed ? parameters.signed === "true" : true;
        let length = parameters.length ? parseInt(parameters.length) : undefined;
        let buf;
        if (value === undefined) {
            throw new Error("Undefined value");
        }
        let dataType = schema.type;
        if (/(short|(u)?int(8|16|32)?$|float(16|32|64)?|byte)/.test(dataType.toLowerCase())) {
            const typeSem = /(u)?(short|int|float|byte)(8|16|32|64)?/.exec(dataType.toLowerCase());
            if (typeSem) {
                signed = typeSem[1] === undefined;
                dataType = typeSem[2];
                length = (_a = +typeSem[3] / 8) !== null && _a !== void 0 ? _a : length;
            }
        }
        switch (dataType) {
            case "boolean":
                return Buffer.alloc(length, value ? 255 : 0);
            case "byte":
            case "short":
            case "int":
            case "integer": {
                length = length !== null && length !== void 0 ? length : 4;
                if (typeof value !== "number") {
                    throw new Error("Value is not a number");
                }
                if (!Number.isSafeInteger(value)) {
                    console.warn("[core/octetstream-codec]", "Value is not a safe integer");
                }
                const limit = Math.pow(2, 8 * length) - 1;
                if (signed) {
                    if (value < -limit || value >= limit) {
                        throw new Error("Integer overflow when representing signed " + value + " in " + length + " byte(s)");
                    }
                }
                else {
                    if (value < 0 || value >= limit) {
                        throw new Error("Integer overflow when representing unsigned " + value + " in " + length + " byte(s)");
                    }
                }
                buf = Buffer.alloc(length);
                switch (length) {
                    case 1:
                        signed ? buf.writeInt8(value, 0) : buf.writeUInt8(value, 0);
                        break;
                    case 2:
                        bigendian
                            ? signed
                                ? buf.writeInt16BE(value, 0)
                                : buf.writeUInt16BE(value, 0)
                            : signed
                                ? buf.writeInt16LE(value, 0)
                                : buf.writeUInt16LE(value, 0);
                        break;
                    case 4:
                        bigendian
                            ? signed
                                ? buf.writeInt32BE(value, 0)
                                : buf.writeUInt32BE(value, 0)
                            : signed
                                ? buf.writeInt32LE(value, 0)
                                : buf.writeUInt32LE(value, 0);
                        break;
                    default:
                        if (signed && value < 0) {
                            value += 1 << (8 * length);
                        }
                        for (let i = 0; i < length; ++i) {
                            const byte = value % 0x100;
                            value /= 0x100;
                            buf.writeInt8(byte, bigendian ? length - i - 1 : i);
                        }
                }
                return buf;
            }
            case "float":
            case "number":
                if (typeof value !== "number") {
                    throw new Error("Value is not a number");
                }
                length = length !== null && length !== void 0 ? length : 8;
                buf = Buffer.alloc(length);
                switch (length) {
                    case 2:
                        (0, float16_1.setFloat16)(new DataView(buf.buffer), 0, value, !bigendian);
                        break;
                    case 4:
                        bigendian ? buf.writeFloatBE(value, 0) : buf.writeFloatLE(value, 0);
                        break;
                    case 8:
                        bigendian ? buf.writeDoubleBE(value, 0) : buf.writeDoubleLE(value, 0);
                        break;
                    default:
                        throw new Error("Wrong buffer length for type 'number', must be 4 or 8, is " + length);
                }
                return buf;
            case "string": {
                const str = String(value);
                return Buffer.from(str);
            }
            case "array":
            case "object":
                throw new Error("Unable to handle object type " + dataType);
            case "null":
                return null;
            default:
                throw new Error("Unable to handle object type " + dataType);
        }
    }
}
exports.default = OctetstreamCodec;
//# sourceMappingURL=octetstream-codec.js.map