"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContentSerdes = void 0;
const json_codec_1 = __importDefault(require("./codecs/json-codec"));
const text_codec_1 = __importDefault(require("./codecs/text-codec"));
const base64_codec_1 = __importDefault(require("./codecs/base64-codec"));
const octetstream_codec_1 = __importDefault(require("./codecs/octetstream-codec"));
const stream_1 = require("stream");
const core_1 = require("./core");
const es2018_1 = require("web-streams-polyfill/ponyfill/es2018");
class ContentSerdes {
    constructor() {
        this.codecs = new Map();
        this.offered = new Set();
    }
    static get() {
        if (!this.instance) {
            this.instance = new ContentSerdes();
            this.instance.addCodec(new json_codec_1.default(), true);
            this.instance.addCodec(new json_codec_1.default("application/senml+json"));
            this.instance.addCodec(new text_codec_1.default());
            this.instance.addCodec(new text_codec_1.default("text/html"));
            this.instance.addCodec(new text_codec_1.default("text/css"));
            this.instance.addCodec(new text_codec_1.default("application/xml"));
            this.instance.addCodec(new text_codec_1.default("application/xhtml+xml"));
            this.instance.addCodec(new text_codec_1.default("image/svg+xml"));
            this.instance.addCodec(new base64_codec_1.default("image/png"));
            this.instance.addCodec(new base64_codec_1.default("image/gif"));
            this.instance.addCodec(new base64_codec_1.default("image/jpeg"));
            this.instance.addCodec(new octetstream_codec_1.default());
        }
        return this.instance;
    }
    static getMediaType(contentType) {
        const parts = contentType.split(";");
        return parts[0].trim();
    }
    static getMediaTypeParameters(contentType) {
        const parts = contentType.split(";").slice(1);
        const params = {};
        parts.forEach((p) => {
            const eq = p.indexOf("=");
            if (eq >= 0) {
                params[p.substr(0, eq).trim()] = p.substr(eq + 1).trim();
            }
            else {
                params[p.trim()] = null;
            }
        });
        return params;
    }
    addCodec(codec, offered = false) {
        ContentSerdes.get().codecs.set(codec.getMediaType(), codec);
        if (offered)
            ContentSerdes.get().offered.add(codec.getMediaType());
    }
    getSupportedMediaTypes() {
        return Array.from(ContentSerdes.get().codecs.keys());
    }
    getOfferedMediaTypes() {
        return Array.from(ContentSerdes.get().offered);
    }
    isSupported(contentType) {
        const mt = ContentSerdes.getMediaType(contentType);
        return this.codecs.has(mt);
    }
    contentToValue(content, schema) {
        if (content.type === undefined) {
            if (content.body.byteLength > 0) {
                content.type = ContentSerdes.DEFAULT;
            }
            else {
                return undefined;
            }
        }
        const mt = ContentSerdes.getMediaType(content.type);
        const par = ContentSerdes.getMediaTypeParameters(content.type);
        if (this.codecs.has(mt)) {
            console.debug("[core/content-serdes]", `ContentSerdes deserializing from ${content.type}`);
            const codec = this.codecs.get(mt);
            const res = codec.bytesToValue(content.body, schema, par);
            return res;
        }
        else {
            console.warn("[core/content-serdes]", `ContentSerdes passthrough due to unsupported media type '${mt}'`);
            return content.body.toString();
        }
    }
    valueToContent(value, schema, contentType = ContentSerdes.DEFAULT) {
        if (value === undefined)
            console.warn("[core/content-serdes]", "ContentSerdes valueToContent got no value");
        if (value instanceof es2018_1.ReadableStream) {
            return { type: contentType, body: core_1.ProtocolHelpers.toNodeStream(value) };
        }
        let bytes = null;
        const mt = ContentSerdes.getMediaType(contentType);
        const par = ContentSerdes.getMediaTypeParameters(contentType);
        if (this.codecs.has(mt)) {
            console.debug("[core/content-serdes]", `ContentSerdes serializing to ${contentType}`);
            const codec = this.codecs.get(mt);
            bytes = codec.valueToBytes(value, schema, par);
        }
        else {
            console.warn("[core/content-serdes]", `ContentSerdes passthrough due to unsupported serialization format '${contentType}'`);
            bytes = Buffer.from(value.toString());
        }
        return { type: contentType, body: stream_1.Readable.from([bytes]) };
    }
}
exports.ContentSerdes = ContentSerdes;
ContentSerdes.DEFAULT = "application/json";
ContentSerdes.TD = "application/td+json";
ContentSerdes.JSON_LD = "application/ld+json";
exports.default = ContentSerdes.get();
//# sourceMappingURL=content-serdes.js.map