"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stream_1 = require("stream");
const es2018_1 = require("web-streams-polyfill/ponyfill/es2018");
function ManagedStream(Base) {
    return class extends Base {
        set nodeStream(nodeStream) {
            this._nodeStream = nodeStream;
        }
        get nodeStream() {
            return this._nodeStream;
        }
        set wotStream(wotStream) {
            this._wotStream = wotStream;
        }
        get wotStream() {
            return this._wotStream;
        }
    };
}
const ManagedReadable = ManagedStream(stream_1.Readable);
const ManagedReadableStream = ManagedStream(es2018_1.ReadableStream);
function isManaged(obj) {
    return obj instanceof ManagedReadableStream || obj instanceof ManagedReadable;
}
class ProtocolHelpers {
    static updatePropertyFormWithTemplate(form, tdTemplate, propertyName) {
        if (form &&
            tdTemplate &&
            tdTemplate.properties &&
            tdTemplate.properties[propertyName] &&
            tdTemplate.properties[propertyName].forms) {
            for (const formTemplate of tdTemplate.properties[propertyName].forms) {
                if (formTemplate.href) {
                }
                if (formTemplate.contentType) {
                    form.contentType = formTemplate.contentType;
                    return;
                }
            }
        }
    }
    static updateActionFormWithTemplate(form, tdTemplate, actionName) {
        if (form &&
            tdTemplate &&
            tdTemplate.actions &&
            tdTemplate.actions[actionName] &&
            tdTemplate.actions[actionName].forms) {
            for (const formTemplate of tdTemplate.actions[actionName].forms) {
                if (formTemplate.href) {
                }
                if (formTemplate.contentType) {
                    form.contentType = formTemplate.contentType;
                    return;
                }
            }
        }
    }
    static updateEventFormWithTemplate(form, tdTemplate, eventName) {
        if (form &&
            tdTemplate &&
            tdTemplate.events &&
            tdTemplate.events[eventName] &&
            tdTemplate.events[eventName].forms) {
            for (const formTemplate of tdTemplate.events[eventName].forms) {
                if (formTemplate.href) {
                }
                if (formTemplate.contentType) {
                    form.contentType = formTemplate.contentType;
                    return;
                }
            }
        }
    }
    static getPropertyContentType(td, propertyName, uriScheme) {
        if (td &&
            propertyName &&
            uriScheme &&
            td.properties &&
            td.properties[propertyName] &&
            td.properties[propertyName].forms &&
            Array.isArray(td.properties[propertyName].forms)) {
            for (const form of td.properties[propertyName].forms) {
                if (form.href && form.href.startsWith(uriScheme) && form.contentType) {
                    return form.contentType;
                }
            }
        }
        return undefined;
    }
    static getActionContentType(td, actionName, uriScheme) {
        if (td &&
            actionName &&
            uriScheme &&
            td.actions &&
            td.actions[actionName] &&
            td.actions[actionName].forms &&
            Array.isArray(td.actions[actionName].forms)) {
            for (const form of td.actions[actionName].forms) {
                if (form.href && form.href.startsWith(uriScheme) && form.contentType) {
                    return form.contentType;
                }
            }
        }
        return undefined;
    }
    static getEventContentType(td, eventName, uriScheme) {
        if (td &&
            eventName &&
            uriScheme &&
            td.events &&
            td.events[eventName] &&
            td.events[eventName].forms &&
            Array.isArray(td.events[eventName].forms)) {
            for (const form of td.events[eventName].forms) {
                if (form.href && form.href.startsWith(uriScheme) && form.contentType) {
                    return form.contentType;
                }
            }
        }
        return undefined;
    }
    static toWoTStream(stream) {
        if (isManaged(stream)) {
            return stream.wotStream;
        }
        const result = new ManagedReadableStream({
            start: (controller) => {
                stream.on("data", (data) => controller.enqueue(data));
                stream.on("error", (e) => controller.error(e));
                stream.on("end", () => controller.close());
            },
            cancel: (reason) => {
                if (stream instanceof stream_1.Readable) {
                    stream.destroy(reason);
                }
            },
        });
        if (stream instanceof stream_1.Readable) {
            result.nodeStream = stream;
        }
        else {
            result.nodeStream = new stream_1.Readable(stream);
        }
        return result;
    }
    static toNodeStream(stream) {
        if (isManaged(stream)) {
            return stream.nodeStream;
        }
        if (stream instanceof stream_1.Readable) {
            return stream;
        }
        const reader = stream.getReader();
        const result = new ManagedReadable({
            read: (size) => {
                reader.read().then((data) => {
                    result.push(data.value);
                    if (data.done) {
                        result.push(null);
                    }
                });
            },
            destroy: (error, callback) => {
                stream.cancel(error);
            },
        });
        result.wotStream = stream;
        return result;
    }
    static readStreamFully(stream) {
        return new Promise((resolve, reject) => {
            if (stream) {
                const chunks = [];
                stream.on("data", (data) => chunks.push(data));
                stream.on("error", reject);
                stream.on("end", () => {
                    if (chunks[0] &&
                        (chunks[0] instanceof Array || chunks[0] instanceof Buffer || chunks[0] instanceof Uint8Array)) {
                        resolve(Buffer.concat(chunks));
                    }
                    else if (chunks[0] && typeof chunks[0] === "string") {
                        resolve(Buffer.from(chunks.join()));
                    }
                    else {
                        resolve(Buffer.from(chunks));
                    }
                });
            }
            else {
                console.debug("[core/helpers]", `Protocol-Helper returns empty buffer for readStreamFully due to undefined stream`);
                resolve(Buffer.alloc(0));
            }
        });
    }
    static findRequestMatchingFormIndex(forms, uriScheme, requestUrl, contentType) {
        if (forms === undefined)
            return 0;
        let matchingForms = forms.filter((form) => {
            const formUrl = new URL(form.href.replace(/(\{[\S]*\})/, ""));
            const reqUrl = requestUrl !== undefined
                ? requestUrl.indexOf("?") !== -1
                    ? requestUrl.split("?")[0]
                    : requestUrl
                : undefined;
            return formUrl.protocol === uriScheme + ":" && (reqUrl === undefined || formUrl.pathname === reqUrl);
        });
        if (contentType) {
            const contentTypeMatchingForms = matchingForms.filter((form) => {
                return form.contentType === contentType;
            });
            if (contentTypeMatchingForms.length > 0)
                matchingForms = contentTypeMatchingForms;
        }
        return matchingForms.length > 0 ? forms.indexOf(matchingForms[0]) : 0;
    }
    static getFormIndexForOperation(interaction, type, operationName, formIndex) {
        var _a;
        let finalFormIndex = -1;
        let defaultOps = [];
        switch (type) {
            case "property":
                if ((interaction.readOnly && operationName === "writeproperty") ||
                    (interaction.writeOnly && operationName === "readproperty"))
                    return finalFormIndex;
                if (interaction.readOnly === undefined || !interaction.readOnly)
                    defaultOps.push("writeproperty");
                if (interaction.writeOnly === undefined || !interaction.writeOnly)
                    defaultOps.push("readproperty");
                break;
            case "action":
                defaultOps = ["invokeaction"];
                break;
            case "event":
                defaultOps = ["subscribeevent", "unsubscribeevent"];
                break;
        }
        if (defaultOps.indexOf(operationName) !== -1) {
            operationName = undefined;
        }
        if (interaction.forms !== undefined && formIndex !== undefined && interaction.forms.length > formIndex) {
            const form = interaction.forms[formIndex];
            if (form && (operationName === undefined || ((_a = form.op) === null || _a === void 0 ? void 0 : _a.includes(operationName)))) {
                finalFormIndex = formIndex;
            }
        }
        if (interaction.forms !== undefined && finalFormIndex === -1) {
            if (operationName !== undefined) {
                interaction.forms.every((form) => {
                    var _a;
                    if ((_a = form.op) === null || _a === void 0 ? void 0 : _a.includes(operationName)) {
                        finalFormIndex = interaction.forms.indexOf(form);
                    }
                    return finalFormIndex === -1;
                });
            }
            else {
                interaction.forms.every((form) => {
                    finalFormIndex = interaction.forms.indexOf(form);
                    return false;
                });
            }
        }
        return finalFormIndex;
    }
}
exports.default = ProtocolHelpers;
//# sourceMappingURL=protocol-helpers.js.map