"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const vm2_1 = require("vm2");
const wot_impl_1 = __importDefault(require("./wot-impl"));
const helpers_1 = __importDefault(require("./helpers"));
const content_serdes_1 = __importDefault(require("./content-serdes"));
const uuid_1 = require("uuid");
const td_tools_1 = require("@node-wot/td-tools");
class Servient {
    constructor() {
        this.servers = [];
        this.clientFactories = new Map();
        this.things = new Map();
        this.credentialStore = new Map();
        this.uncaughtListeners = [];
    }
    runScript(code, filename = "script") {
        const helpers = new helpers_1.default(this);
        const context = {
            WoT: new wot_impl_1.default(this),
            WoTHelpers: helpers,
            ModelHelpers: new td_tools_1.ThingModelHelpers(helpers),
        };
        const vm = new vm2_1.NodeVM({
            sandbox: context,
        });
        const listener = (err) => {
            this.logScriptError(`Asynchronous script error '${filename}'`, err);
            process.exit(1);
        };
        process.prependListener("uncaughtException", listener);
        this.uncaughtListeners.push(listener);
        try {
            return vm.run(code, filename);
        }
        catch (err) {
            this.logScriptError(`Servient found error in privileged script '${filename}'`, err);
            return undefined;
        }
    }
    runPrivilegedScript(code, filename = "script", options = {}) {
        const helpers = new helpers_1.default(this);
        const context = {
            WoT: new wot_impl_1.default(this),
            WoTHelpers: helpers,
            ModelHelpers: new td_tools_1.ThingModelHelpers(helpers),
        };
        const vm = new vm2_1.NodeVM({
            sandbox: context,
            require: {
                external: true,
                builtin: ["*"],
            },
            argv: options.argv,
            compiler: options.compiler,
            env: options.env,
        });
        const listener = (err) => {
            this.logScriptError(`Asynchronous script error '${filename}'`, err);
            process.exit(1);
        };
        process.prependListener("uncaughtException", listener);
        this.uncaughtListeners.push(listener);
        try {
            return vm.run(code, filename);
        }
        catch (err) {
            this.logScriptError(`Servient found error in privileged script '${filename}'`, err);
            return undefined;
        }
    }
    logScriptError(description, error) {
        let message;
        if (typeof error === "object" && error.stack) {
            const match = error.stack.match(/evalmachine\.<anonymous>:([0-9]+:[0-9]+)/);
            if (Array.isArray(match)) {
                message = `and halted at line ${match[1]}\n    ${error}`;
            }
            else {
                message = `and halted with ${error.stack}`;
            }
        }
        else {
            message = `that threw ${typeof error} instead of Error\n    ${error}`;
        }
        console.error("[core/servient]", `Servient caught ${description} ${message}`);
    }
    addMediaType(codec, offered = false) {
        content_serdes_1.default.addCodec(codec, offered);
    }
    expose(thing) {
        if (this.servers.length === 0) {
            console.warn("[core/servient]", `Servient has no servers to expose Things`);
            return new Promise((resolve) => {
                resolve();
            });
        }
        console.debug("[core/servient]", `Servient exposing '${thing.title}'`);
        const tdTemplate = JSON.parse(JSON.stringify(thing));
        thing.forms = [];
        for (const name in thing.properties) {
            thing.properties[name].forms = [];
        }
        for (const name in thing.actions) {
            thing.actions[name].forms = [];
        }
        for (const name in thing.events) {
            thing.events[name].forms = [];
        }
        const serverPromises = [];
        this.servers.forEach((server) => {
            serverPromises.push(server.expose(thing, tdTemplate));
        });
        return new Promise((resolve, reject) => {
            Promise.all(serverPromises)
                .then(() => resolve())
                .catch((err) => reject(err));
        });
    }
    addThing(thing) {
        if (thing.id === undefined) {
            thing.id = "urn:uuid:" + (0, uuid_1.v4)();
            console.warn("[core/servient]", `Servient generating ID for '${thing.title}': '${thing.id}'`);
        }
        if (!this.things.has(thing.id)) {
            this.things.set(thing.id, thing);
            console.debug("[core/servient]", `Servient reset ID '${thing.id}' with '${thing.title}'`);
            return true;
        }
        else {
            return false;
        }
    }
    destroyThing(thingId) {
        return new Promise((resolve, reject) => {
            if (this.things.has(thingId)) {
                console.debug("[core/servient]", `Servient destroying thing with id '${thingId}'`);
                this.things.delete(thingId);
                const serverPromises = [];
                this.servers.forEach((server) => {
                    serverPromises.push(server.destroy(thingId));
                });
                Promise.all(serverPromises)
                    .then(() => resolve(true))
                    .catch((err) => reject(err));
            }
            else {
                console.warn("[core/servient]", `Servient was asked to destroy thing but failed to find thing with id '${thingId}'`);
                resolve(false);
            }
        });
    }
    getThing(id) {
        if (this.things.has(id)) {
            return this.things.get(id);
        }
        else
            return null;
    }
    getThings() {
        console.debug("[core/servient]", `Servient getThings size == '${this.things.size}'`);
        const ts = {};
        this.things.forEach((thing, id) => {
            ts[id] = thing.getThingDescription();
        });
        return ts;
    }
    addServer(server) {
        this.things.forEach((thing, id) => server.expose(thing));
        this.servers.push(server);
        return true;
    }
    getServers() {
        return this.servers.slice(0);
    }
    addClientFactory(clientFactory) {
        this.clientFactories.set(clientFactory.scheme, clientFactory);
    }
    hasClientFor(scheme) {
        console.debug("[core/servient]", `Servient checking for '${scheme}' scheme in ${this.clientFactories.size} ClientFactories`);
        return this.clientFactories.has(scheme);
    }
    getClientFor(scheme) {
        if (this.clientFactories.has(scheme)) {
            console.debug("[core/servient]", `Servient creating client for scheme '${scheme}'`);
            return this.clientFactories.get(scheme).getClient();
        }
        else {
            throw new Error(`Servient has no ClientFactory for scheme '${scheme}'`);
        }
    }
    getClientSchemes() {
        return Array.from(this.clientFactories.keys());
    }
    addCredentials(credentials) {
        if (typeof credentials === "object") {
            for (const i in credentials) {
                console.debug("[core/servient]", `Servient storing credentials for '${i}'`);
                let currentCredentials = this.credentialStore.get(i);
                if (!currentCredentials) {
                    currentCredentials = [];
                    this.credentialStore.set(i, currentCredentials);
                }
                currentCredentials.push(credentials[i]);
            }
        }
    }
    getCredentials(identifier) {
        console.debug("[core/servient]", `Servient looking up credentials for '${identifier}' (@deprecated)`);
        const currentCredentials = this.credentialStore.get(identifier);
        if (currentCredentials && currentCredentials.length > 0) {
            return currentCredentials[0];
        }
        else {
            return undefined;
        }
    }
    retrieveCredentials(identifier) {
        console.debug("[core/servient]", `Servient looking up credentials for '${identifier}'`);
        return this.credentialStore.get(identifier);
    }
    start() {
        return __awaiter(this, void 0, void 0, function* () {
            const serverStatus = [];
            this.servers.forEach((server) => serverStatus.push(server.start(this)));
            this.clientFactories.forEach((clientFactory) => clientFactory.init());
            yield Promise.all(serverStatus);
            return new wot_impl_1.default(this);
        });
    }
    shutdown() {
        return __awaiter(this, void 0, void 0, function* () {
            this.clientFactories.forEach((clientFactory) => clientFactory.destroy());
            const promises = this.servers.map((server) => server.stop());
            yield Promise.all(promises);
            this.uncaughtListeners.forEach((listener) => {
                process.removeListener("uncaughtException", listener);
            });
        });
    }
}
exports.default = Servient;
//# sourceMappingURL=servient.js.map