/*
 *  Database class for DMAP sharing
 *
 *  Copyright (C) 2008 W. Michael Petullo <mike@flyn.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib/gstdio.h>
#include <glib.h>

#include "util.h"
#include "dmapd-dmap-db-ghashtable.h"

/* Media ID's start at max and go down. Container ID's start at 1 and go up. */
static guint _nextid = G_MAXINT; /* NOTE: this should be G_MAXUINT, but iPhoto can't handle it. */

struct DmapdDmapDbGHashTablePrivate {
	GHashTable *db;
	gchar *db_dir;
	DmapRecordFactory *record_factory;
	GSList *acceptable_formats;
};

enum {
        PROP_0,
	PROP_DB_DIR,
	PROP_RECORD_FACTORY,
	PROP_ACCEPTABLE_FORMATS
};

struct loc_id {
	const gchar *key;
	const gchar *location;
	guint id;
};

static gboolean
_path_match (gpointer key, gpointer val, gpointer user_data)
{
	gboolean fnval = FALSE;
	gchar *location = NULL;
	struct loc_id *loc_id = user_data;

	loc_id->id = GPOINTER_TO_UINT (key);

	g_object_get (DMAP_RECORD (val), loc_id->key, &location, NULL);

	fnval = location && ! strcmp (location, loc_id->location);

	g_free (location);

	return fnval;
}

static guint
_lookup_id_by_location (const DmapDb *db, const gchar *location)
{
	guint fnval = DMAP_DB_ID_BAD;
	struct loc_id user_data;

	/* Try (possibly transcoded) location ... */
	user_data.key = "location";
	user_data.location = location;
	user_data.id = 0;

	if (g_hash_table_find (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db,
		(GHRFunc) _path_match,
		(gpointer) &user_data)) {
		fnval = user_data.id;
		goto done;
	}

	/*
	 * ... and then original location. For example, playlists will contain
	 * original location in the case of transcoded files.
	 */
	user_data.key = "original-location";

	if (g_hash_table_find (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db,
		(GHRFunc) _path_match,
		(gpointer) &user_data)) {
		fnval = user_data.id;
		goto done;
	}

done:
	return fnval;
}

static DmapRecord *
_lookup_by_id (const DmapDb *db, guint id)
{
	DmapRecord *record = NULL;

	if (DMAP_DB_ID_BAD == id) {
		goto done;
	}

	record = g_hash_table_lookup (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db, GUINT_TO_POINTER (id));
	g_object_ref (record);

done:
	return record;
}

static void
_foreach(const DmapDb *db, DmapIdRecordFunc func, gpointer data)
{
	g_hash_table_foreach (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db, (GHFunc) func, data);
}

static gint64
_count (const DmapDb *db)
{
	return g_hash_table_size (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db);
}

static GArray *
_cache_read (const gchar *path)
{
        gchar *data;
        size_t size;
        GArray *blob = NULL;
        GError *error = NULL;

        g_file_get_contents (path, &data, &size, &error);
        if (error != NULL) {
                g_debug ("No record cached at %s", path);
        } else {
                blob = g_array_new (FALSE, FALSE, 1);
                g_array_append_vals (blob, (guint8 *) data, size);
                g_free (data);
        }

        return blob;
}

static guint
_add_with_id (DmapDb *db, DmapRecord *record, guint id, GError **error)
{
	if (DMAP_DB_ID_BAD == id) {
		g_set_error(error,
		            DMAP_ERROR,
	                    DMAP_STATUS_DB_BAD_ID,
	                   "Bad record identifier provided");
		goto done;
	}

	g_hash_table_insert (DMAPD_DMAP_DB_GHASHTABLE (db)->priv->db,
	                     GUINT_TO_POINTER (id),
	                     record);

done:
	return id;
}

static guint
_add_no_cache (DmapDb *db, DmapRecord *record, GError **error)
{
	guint id;

	id = _add_with_id (db, record, _nextid--, error);
	if (DMAP_DB_ID_BAD != id) {
		g_object_ref (record);
	}

	return id;
}

static void
_load_cached_record (DmapDb *db, const gchar *db_dir,
                     DmapRecordFactory *factory, const gchar *entry)
{
	gboolean is_regular;
	gboolean is_record;
	GError *error      = NULL;
	GArray *blob       = NULL;
	DmapRecord *record = NULL;

	gchar *path = g_strdup_printf ("%s/%s", db_dir, entry);
	is_regular = g_file_test (path, G_FILE_TEST_IS_REGULAR);
	is_record  = is_regular ? g_str_has_suffix (path, ".record") : FALSE;

	if (!is_regular || !is_record) {
		goto done;
	}

	blob = _cache_read (path);
	if (NULL == blob) {
		goto done;
	}

	g_debug ("Loading cached %s", path);

	record = dmap_record_factory_create (factory, NULL, &error);
	if (NULL == record) {
		g_assert(NULL != error);
		g_warning("Unable to add record for %s: %s", path, error->message);
		goto done;
	}

	if (dmap_record_set_from_blob (record, blob)) {
		_add_no_cache (DMAP_DB (db), record, &error);
		if (NULL != error) {
			g_warning("Unable to add record for %s: %s", path, error->message);
		}
	} else {
		g_warning ("Removing stale cache entry %s", path);
		g_unlink (path);
	}


done:
	g_free(path);

	if (NULL != record) {
		g_object_unref (record);
	}

	if (NULL != blob) {
		g_array_free(blob, TRUE);
	}
}

static void
_load_cached_records (DmapDb *db, const gchar *db_dir, DmapRecordFactory *factory)
{
	GDir *d;
	GError *error = NULL;
	const gchar *entry;

	if (NULL == db_dir) {
		goto done;
	}

	d = g_dir_open (db_dir, 0, &error);
	if (NULL == d) {
		g_assert(NULL != error);

		g_warning ("%s", error->message);
		goto done;
	}

	while ((entry = g_dir_read_name (d))) {
		_load_cached_record(db, db_dir, factory, entry);
	}

	g_dir_close (d);

done:
	return;
}

static guint
_add (DmapDb *db, DmapRecord *record, GError **error)
{
	gchar *db_dir   = NULL;
	GArray *blob    = NULL;
	gchar *location = NULL;;

	g_object_get (record, "location", &location, NULL);
	g_object_get (db, "db-dir", &db_dir, NULL);

	if (db_dir != NULL) {
		blob = dmap_record_to_blob (record);
		util_cache_store (db_dir, location, blob);
	}

	if (NULL != location) {
		g_free (location);
	}

	if (NULL != db_dir) {
		g_free (db_dir);
	}

	if (NULL != blob) {
		g_array_unref (blob);
	}

	return _add_no_cache (db, record, error);
}

static guint
_add_path (DmapDb *db, const gchar *path, GError **error)
{
	guint id = 0;
	DmapRecord *record;
	DmapRecordFactory *factory = NULL;
	char *format = NULL;
	const gchar * const *acceptable_formats = NULL;

	g_object_get (db, "record-factory", &factory, NULL);
	g_assert (factory);

	record = dmap_record_factory_create (factory, (gpointer) path, error);
	if (NULL == record) {
		goto done;
	}

	g_object_get (record, "format", &format, NULL);
	g_object_get (db, "acceptable-formats", &acceptable_formats, NULL);

	if (! acceptable_formats || g_strv_contains(acceptable_formats, format)) {
		id = _add (db, record, error);
	}

	g_object_unref (record);

done:
	if (NULL != format) {
		g_free (format);
	}

	return id;
}

static void
_dmap_db_interface_init (gpointer iface, G_GNUC_UNUSED gpointer data)
{
        DmapDbInterface *dmap_db = iface;

	g_assert (G_TYPE_FROM_INTERFACE (dmap_db) == DMAP_TYPE_DB);

	dmap_db->add = _add;
	dmap_db->add_with_id = _add_with_id;
	dmap_db->add_path = _add_path;
	dmap_db->lookup_by_id = _lookup_by_id;
	dmap_db->lookup_id_by_location = _lookup_id_by_location;
	dmap_db->foreach = _foreach;
	dmap_db->count = _count;
}

G_DEFINE_TYPE_WITH_CODE (DmapdDmapDbGHashTable, dmapd_dmap_db_ghashtable, G_TYPE_OBJECT,
                         G_IMPLEMENT_INTERFACE (DMAP_TYPE_DB, _dmap_db_interface_init)
                         G_ADD_PRIVATE (DmapdDmapDbGHashTable))

static void
dmapd_dmap_db_ghashtable_init (DmapdDmapDbGHashTable *db)
{
	db->priv = dmapd_dmap_db_ghashtable_get_instance_private (db);
	db->priv->db = g_hash_table_new_full (g_direct_hash,
					      g_direct_equal,
					      NULL,
					      g_object_unref);
}

static GObject*
_constructor (GType type, guint n_construct_params, GObjectConstructParam *construct_params)
{
	GObject *object;
	gchar *db_dir = NULL;
	DmapRecordFactory *factory = NULL;

	object = G_OBJECT_CLASS (dmapd_dmap_db_ghashtable_parent_class)->constructor (type, n_construct_params, construct_params);

	g_object_get (object, "db-dir", &db_dir, "record-factory", &factory, NULL);
	/* NOTE: Don't load cache when used for DmapdDmapContainerRecord: */
	if (db_dir && factory) {
		_load_cached_records (DMAP_DB (object), db_dir, factory);
	}
	g_free (db_dir);

	return object;
}

static void
_finalize (GObject *object)
{
	DmapdDmapDbGHashTable *db = DMAPD_DMAP_DB_GHASHTABLE (object);

	g_debug ("Finalizing DmapdDmapDbGHashTable (%d records)",
		 g_hash_table_size (db->priv->db));

	g_hash_table_destroy (db->priv->db);
}

static void
_set_property (GObject *object,
               guint prop_id,
               const GValue *value,
               GParamSpec *pspec)
{
        DmapdDmapDbGHashTable *db = DMAPD_DMAP_DB_GHASHTABLE (object);

	switch (prop_id) {
		case PROP_DB_DIR:
			g_free (db->priv->db_dir);
			db->priv->db_dir = g_value_dup_string (value);
			break;
		case PROP_RECORD_FACTORY:
			if (db->priv->record_factory) {
				g_object_unref (db->priv->record_factory);
			}
			db->priv->record_factory = DMAP_RECORD_FACTORY (g_value_get_pointer (value));
			break;	
		case PROP_ACCEPTABLE_FORMATS:
			db->priv->acceptable_formats = g_value_get_pointer (value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}

static void
_get_property (GObject *object,
               guint prop_id,
               GValue *value,
               GParamSpec *pspec)
{
        DmapdDmapDbGHashTable *db = DMAPD_DMAP_DB_GHASHTABLE (object);

	switch (prop_id) {
		case PROP_DB_DIR:
			g_value_set_static_string (value, db->priv->db_dir);
			break;
		case PROP_RECORD_FACTORY:
			g_value_set_pointer (value, db->priv->record_factory);
			break;	
		case PROP_ACCEPTABLE_FORMATS:
			g_value_set_pointer (value, db->priv->acceptable_formats);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}


static void dmapd_dmap_db_ghashtable_class_init (DmapdDmapDbGHashTableClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize     = _finalize;
	object_class->constructor  = _constructor;
	object_class->set_property = _set_property;
	object_class->get_property = _get_property;

	g_object_class_install_property (object_class,
					 PROP_RECORD_FACTORY,
					 g_param_spec_pointer ("record-factory",
					 		       "Record factory",
							       "Record factory",
							        G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	g_object_class_install_property (object_class,
					 PROP_DB_DIR,
					 g_param_spec_string ("db-dir",
					 		      "Directory for database cache",
							      "Directory for database cache",
							      NULL,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

	g_object_class_install_property (object_class, PROP_ACCEPTABLE_FORMATS,
					 g_param_spec_pointer ("acceptable-formats",
					                       "Acceptable formats",
							       "Acceptable formats",
							        G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

#ifdef HAVE_CHECK

#include "dmapd-daap-record.h"
#include "dmapd-daap-record-factory.h"

START_TEST(test_dmapd_dmap_db_ghashtable_lookup_by_location)
{
	guint id;
	DmapDb *db;
	DmapdDmapAvRecordFactory *factory;
	DmapRecord *record1, *record2;

	factory = g_object_new (TYPE_DMAPD_DMAP_RECORD_FACTORY, NULL);

	db = DMAP_DB (util_object_from_module (TYPE_DMAPD_DMAP_DB,
	                                       DEFAULT_MODULEDIR,
	                                       "ghashtable",
	                                       "record-factory",
	                                       factory,
	                                       NULL));

	record1 = DMAP_RECORD (g_object_new(TYPE_DMAPD_DMAP_AV_RECORD, "location", "foo"));
	record2 = DMAP_RECORD (g_object_new(TYPE_DMAPD_DMAP_AV_RECORD, "original-location", "bar"));

	dmap_db_add (db, record1, NULL);
	dmap_db_add (db, record2, NULL);

	id = _lookup_id_by_location(db, "foo");
	fail_unless (id != DMAP_DB_ID_BAD);

	id = _lookup_id_by_location(db, "bar");
	fail_unless (id != DMAP_DB_ID_BAD);

	g_object_unref (db);
	g_object_unref (factory);
}
END_TEST

Suite *dmapd_test_dmap_db_ghashtable_suite(void)
{
	Suite *s = suite_create("dmapd-test-dmapd-db-ghashtable");

	TCase *tc_dmapd_dmap_db_ghashtable_lookup_by_location = tcase_create("test_dmapd_dmap_db_ghashtable_lookup_by_location");
	tcase_add_test(tc_dmapd_dmap_db_ghashtable_lookup_by_location, test_dmapd_dmap_db_ghashtable_lookup_by_location);
	suite_add_tcase(s, tc_dmapd_dmap_db_ghashtable_lookup_by_location);

	return s;
}

#endif
