/*   FILE: util.c -- utility functions
 * AUTHOR: W. Michael Petullo <mike@flyn.org>
 *   DATE: 01 January 2009 
 *
 * Copyright (c) 2009 W. Michael Petullo <new@flyn.org>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "util.h"
#include "dmapd-module.h"
#include "dmapd-dmap-db.h"
#include "dmapd-dmap-db-ghashtable.h"
#include "db-builder.h"
#include "av-meta-reader.h"
#include "av-render.h"
#include "photo-meta-reader.h"

static GHashTable *_stringleton;
static GMutex _mutex;

gchar *
util_parse_plugin_option (gchar *str, GHashTable *hash_table)
{
	// The user may specify, e.g.:
	//   "gst:sink=foo,opt1=bar,opt2=baz"
	// or
	//   "gst"

	gchar *plugin = str;
	gchar *colon = strchr (str, ':');

	if (colon) {
		gchar *eq, *key;
		*colon = 0x00;
		key = colon + 1;	
		eq = strchr (key, '=');
		if (str && eq) {
			gchar *comma;
			do {
				gchar *val;
				*eq = 0x00;
				val = eq + 1;
				comma = strchr (val, ',');
				if (comma) {
					*comma = 0x00;
				}
				g_hash_table_insert (hash_table, g_strdup (key), g_strdup(val));
				if (comma) {
					key = comma + 1;
					eq = strchr (key, '=');
					if (! str || ! eq) {
						g_error ("Badly formatted plugin string");
					}
				}
			} while (comma);
		} else {
			g_error ("Badly formatted plugin string");
		}
	}

	return plugin;
}

#ifdef HAVE_CHECK

START_TEST(_test_parse_plugin_option_simple)
{
	GHashTable *hash_table = g_hash_table_new (g_str_hash, g_str_equal);
	gchar *str = g_strdup ("gst");

	gchar *result = util_parse_plugin_option (str, hash_table);

	fail_unless (! strcmp (result, "gst"));

	g_hash_table_destroy (hash_table);
	g_free (str);
}
END_TEST

START_TEST(_test_parse_plugin_option_full)
{
	gchar *val;
	GHashTable *hash_table = g_hash_table_new (g_str_hash, g_str_equal);
	gchar *str = g_strdup ("gst:sink=apex,host=Host.local,port=5000,generation=1,protocol=1");

	gchar *result = util_parse_plugin_option (str, hash_table);

	fail_unless (! strcmp (result, "gst"));

	val = g_hash_table_lookup (hash_table, "sink");
	fail_unless (! strcmp (val, "apex"));

	val = g_hash_table_lookup (hash_table, "host");
	fail_unless (! strcmp (val, "Host.local"));

	val = g_hash_table_lookup (hash_table, "port");
	fail_unless (! strcmp (val, "5000"));

	val = g_hash_table_lookup (hash_table, "generation");
	fail_unless (! strcmp (val, "1"));

	val = g_hash_table_lookup (hash_table, "protocol");
	fail_unless (! strcmp (val, "1"));

	g_hash_table_destroy (hash_table);
	g_free (str);
}
END_TEST

#endif

GArray *
util_blob_add_atomic (GArray *blob, const guint8 *ptr, const size_t size)
{
	return g_array_append_vals (blob, ptr, size);
}

GArray *
util_blob_add_string (GArray *blob, const gchar *str)
{
	return g_array_append_vals (blob,
	                           (const guint8 *) str,
	                            strlen (str) + 1);
}

void 
util_stringleton_init (void)
{
	static gboolean initialized = FALSE;

	if (! initialized) {
		g_mutex_init(&_mutex);

		_stringleton = g_hash_table_new_full (g_str_hash,
						     g_str_equal,
						     g_free,
						     NULL);
	}

	initialized = TRUE;
}

const gchar *
util_stringleton_ref (const gchar *str)
{
	gpointer key;
	gpointer val;

	g_assert (_stringleton);
	g_assert (str);

	g_mutex_lock(&_mutex);

	/* NOTE: insert will free passed str if the key already exists,
	 * not existing key in hash table.
	 */
	if (g_hash_table_lookup_extended (_stringleton, str, &key, &val)) {
		str = (gchar *) key;
		g_hash_table_insert (_stringleton,
				    (gpointer) g_strdup (str),
		                     val + 1);
	} else {
		val = NULL;
		str = g_strdup (str);
		g_hash_table_insert (_stringleton,
				    (gpointer) str,
		                     val + 1);
	}

	g_debug ("        Increment stringleton %s reference count to %u.", str, GPOINTER_TO_UINT (val));

	g_mutex_unlock(&_mutex);

	return str;
}

void
util_stringleton_unref (const gchar *str)
{
	guint count;

	g_assert (_stringleton);

	g_mutex_lock(&_mutex);

	if (str != NULL) {
		count = GPOINTER_TO_UINT (g_hash_table_lookup (_stringleton,
							      (gpointer) str));

		g_debug ("        Decrement stringleton %s reference count to %u.", str, count - 1);

		/* NOTE: insert will free passed str if the key already exists,
		 * not existing key in hash table.
		 */
		if (count > 1) {
			g_hash_table_insert (_stringleton,
					    (gpointer) g_strdup (str),
					     GUINT_TO_POINTER (count - 1));
		} else if (count == 1) {
			g_hash_table_remove (_stringleton, (gpointer) str);
		}
	}

	g_mutex_unlock(&_mutex);
}

void
util_stringleton_deinit (void)
{
	g_hash_table_destroy (_stringleton);
}

static char *
_find_plugin_template (GType type)
{
	char *template = NULL;

	if (type == TYPE_DMAPD_DMAP_DB) {
		template = "dmapd-dmap-db-%s";
	} else if (type == TYPE_DB_BUILDER) {
		template = "db-builder-%s";
	} else if (type == TYPE_AV_META_READER) {
		template = "av-meta-reader-%s";
	} else if (type == TYPE_AV_RENDER) {
		template = "av-render-%s";
	} else if (type == TYPE_PHOTO_META_READER) {
		template = "photo-meta-reader-%s";
	}

	return template;
}


GObject *
util_object_from_module (GType type,
                         const gchar *module_dir,
                         const gchar *module_name,
                         const gchar *first_property_name,
                         ...)
{
	va_list ap;
	GType *filters = NULL;
	GType child_type = G_TYPE_INVALID;
	guint n_filters;
	const gchar *fmt;
	gchar *module_filename;
        gchar *module_path;
	GObject *fnval = NULL;
	DmapdModule *module;

	va_start (ap, first_property_name);

	if (! (fmt = _find_plugin_template (type))) {
		g_error ("Could not find plugin template");
	}

	/* dmapd-dmap-db-ghashtable is built in because it is used by DmapdDmapContainerRecord: */
	if (! strcmp (module_name, "ghashtable")) {
		g_debug ("Not loading built in %s.", g_type_name (TYPE_DMAPD_DMAP_DB_GHASHTABLE));
		child_type = TYPE_DMAPD_DMAP_DB_GHASHTABLE;
		fnval = g_object_new_valist (child_type, first_property_name, ap);
	} else {
		module_filename = g_strdup_printf (fmt, module_name);
		module_path = g_module_build_path (module_dir, module_filename);

		module = dmapd_module_new (module_path);
		if (module == NULL || ! g_type_module_use (G_TYPE_MODULE (module))) {
			g_warning ("Error opening %s", module_path);
		} else {
			/* FIXME: free filters */
			filters = g_type_children (type, &n_filters);
			g_assert (n_filters == 1);
			g_assert (g_type_is_a (filters[0], type));

			child_type = filters[0];
			fnval = g_object_new_valist (child_type, first_property_name, ap);
		}

		g_free (filters);
		g_free (module_filename);
		g_free (module_path);
	}

	va_end (ap);

	return fnval;
}

gboolean
util_hash_file (const gchar *uri, unsigned char hash[DMAP_HASH_SIZE])
{
	g_assert (NULL != uri);
	g_assert (NULL != hash);

	gsize bytes_read = 0;
	gboolean fnval = FALSE;
	GError *error = NULL;
	GFile *file = NULL;
	GFileInputStream *stream = NULL;
	unsigned char buffer[BUFSIZ];
	DmapHashContext context;

	file = g_file_new_for_uri (uri);
	if (NULL == file) {
		g_warning ("Could not open %s", uri);
		goto _done;
	}

	stream = g_file_read (file, NULL, &error);
	if (error != NULL) {
		g_warning ("Could not read %s: %s", uri, error->message);
		goto _done;
	}

	dmap_md5_progressive_init (&context);

	while (0 < (bytes_read = g_input_stream_read (G_INPUT_STREAM (stream), buffer, BUFSIZ, NULL, &error))) {
		dmap_md5_progressive_update (&context, buffer, bytes_read);
	}	

	if (NULL != error) {
		g_warning ("Could not read %s: %s", uri, error->message);
		goto _done;
	}

	dmap_md5_progressive_final (&context, hash);

	fnval = TRUE;

_done:
	if (NULL != file) {
		g_object_unref (file);
	}

	if (NULL != stream) {
		g_object_unref (stream);
	}

	return fnval;
}

gchar *
util_cache_path (cache_type_t type, const gchar *db_dir, const gchar *uri)
{
        gchar *cachepath = NULL;
	guchar raw_hash[DMAP_HASH_SIZE] = { 0 };
        gchar hash[DMAP_HASH_SIZE * 2 + 1] = { 0 };
	
	if (! util_hash_file (uri, raw_hash)) {
		goto _done;
	}

	dmap_md5_progressive_to_string (raw_hash, hash);

	switch (type) {
	case CACHE_TYPE_RECORD:
		cachepath = g_strdup_printf ("%s/%s.%s", db_dir, hash, "record");
		break;
	case CACHE_TYPE_TRANSCODED_DATA:
		/* FIXME: set extension properly? */
		cachepath = g_strdup_printf ("%s/%s.%s", db_dir, hash, "data");
		break;
	case CACHE_TYPE_THUMBNAIL_DATA:
		cachepath = g_strdup_printf ("%s/%s.%s", db_dir, hash, "thumb");
		break;
	default:
		g_error ("Bad cache path type");
	}

_done:
        return cachepath;
}

void
util_cache_store (const gchar *db_dir, const gchar *uri, GArray *blob)
{
        struct stat st;
        gchar *cachepath = NULL;
        GError *error = NULL;
        /* NOTE: g_stat seemed broken; would corrupt GError *error. */
        if (stat (db_dir, &st) != 0) {
                g_warning ("cache directory %s does not exist, will not cache", db_dir);
		goto _done;
        }
        if (! (st.st_mode & S_IFDIR)) {
                g_warning ("%s is not a directory, will not cache", db_dir);
		goto _done;
        }
        cachepath = util_cache_path (CACHE_TYPE_RECORD, db_dir, uri);
	if (NULL == cachepath) {
		goto _done;
	}

        g_file_set_contents (cachepath,
			    (gchar *) blob->data,
			     blob->len,
			     &error);
        if (error != NULL) {
                g_warning ("Error writing %s: %s", cachepath, error->message);
		goto _done;
        }

_done:
	if (NULL != cachepath) {
		g_free (cachepath);
	}

	return;
}

#ifdef HAVE_CHECK

Suite *dmapd_test_parse_plugin_option_suite(void)
{
	TCase *tc;
        Suite *s = suite_create("dmapd-test-parse-plugin-option-suite");

	tc = tcase_create("test_plugin_option_simple");
	tcase_add_test(tc, _test_parse_plugin_option_simple);
	suite_add_tcase(s, tc);

	tc = tcase_create("test_parse_plugin_option_full");
	tcase_add_test(tc, _test_parse_plugin_option_full);
	suite_add_tcase(s, tc);

	return s;
}

#endif
