// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { Constants } from "@azure/core-amqp";
import { isDefined, objectHasProperty } from "./util/typeGuards";
const messagePropertiesMap = {
    message_id: "messageId",
    user_id: "userId",
    to: "to",
    subject: "subject",
    reply_to: "replyTo",
    correlation_id: "correlationId",
    content_type: "contentType",
    content_encoding: "contentEncoding",
    absolute_expiry_time: "absoluteExpiryTime",
    creation_time: "creationTime",
    group_id: "groupId",
    group_sequence: "groupSequence",
    reply_to_group_id: "replyToGroupId"
};
/**
 * Converts the AMQP message to an EventData.
 * @param msg - The AMQP message that needs to be converted to EventData.
 * @hidden
 */
export function fromRheaMessage(msg) {
    const data = {
        body: msg.body
    };
    if (msg.message_annotations) {
        for (const annotationKey of Object.keys(msg.message_annotations)) {
            switch (annotationKey) {
                case Constants.partitionKey:
                    data.partitionKey = msg.message_annotations[annotationKey];
                    break;
                case Constants.sequenceNumber:
                    data.sequenceNumber = msg.message_annotations[annotationKey];
                    break;
                case Constants.enqueuedTime:
                    data.enqueuedTimeUtc = new Date(msg.message_annotations[annotationKey]);
                    break;
                case Constants.offset:
                    data.offset = msg.message_annotations[annotationKey];
                    break;
                default:
                    if (!data.systemProperties) {
                        data.systemProperties = {};
                    }
                    data.systemProperties[annotationKey] = convertDatesToNumbers(msg.message_annotations[annotationKey]);
                    break;
            }
        }
    }
    if (msg.application_properties) {
        data.properties = convertDatesToNumbers(msg.application_properties);
    }
    if (msg.delivery_annotations) {
        data.lastEnqueuedOffset = msg.delivery_annotations.last_enqueued_offset;
        data.lastSequenceNumber = msg.delivery_annotations.last_enqueued_sequence_number;
        data.lastEnqueuedTime = new Date(msg.delivery_annotations.last_enqueued_time_utc);
        data.retrievalTime = new Date(msg.delivery_annotations.runtime_info_retrieval_time_utc);
    }
    const messageProperties = Object.keys(messagePropertiesMap);
    for (const messageProperty of messageProperties) {
        if (!data.systemProperties) {
            data.systemProperties = {};
        }
        if (msg[messageProperty] != null) {
            data.systemProperties[messagePropertiesMap[messageProperty]] = convertDatesToNumbers(msg[messageProperty]);
        }
    }
    return data;
}
/**
 * Converts an EventData object to an AMQP message.
 * @param data - The EventData object that needs to be converted to an AMQP message.
 * @param partitionKey - An optional key to determine the partition that this event should land in.
 * @hidden
 */
export function toRheaMessage(data, partitionKey) {
    const msg = {
        body: data.body
    };
    // As per the AMQP 1.0 spec If the message-annotations or delivery-annotations section is omitted,
    // it is equivalent to a message-annotations section containing anempty map of annotations.
    msg.message_annotations = {};
    if (data.properties) {
        msg.application_properties = data.properties;
    }
    if (isDefined(partitionKey)) {
        msg.message_annotations[Constants.partitionKey] = partitionKey;
        // Event Hub service cannot route messages to a specific partition based on the partition key
        // if AMQP message header is an empty object. Hence we make sure that header is always present
        // with atleast one property. Setting durable to true, helps us achieve that.
        msg.durable = true;
    }
    return msg;
}
/**
 * Converts any Date objects into a number representing date.getTime().
 * Recursively checks for any Date objects in arrays and objects.
 * @internal
 */
function convertDatesToNumbers(thing) {
    // fast exit
    if (!isDefined(thing))
        return thing;
    // When 'thing' is a Date, return the number representation
    if (typeof thing === "object" &&
        objectHasProperty(thing, "getTime") &&
        typeof thing.getTime === "function") {
        return thing.getTime();
    }
    /*
      Examples:
      [0, 'foo', new Date(), { nested: new Date()}]
    */
    if (Array.isArray(thing)) {
        return thing.map(convertDatesToNumbers);
    }
    /*
      Examples:
      { foo: new Date(), children: { nested: new Date() }}
    */
    if (typeof thing === "object" && isDefined(thing)) {
        thing = Object.assign({}, thing);
        for (const key of Object.keys(thing)) {
            thing[key] = convertDatesToNumbers(thing[key]);
        }
    }
    return thing;
}
//# sourceMappingURL=eventData.js.map