// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { v4 as uuid } from "uuid";
import { logErrorStackTrace, logger } from "./log";
import { message } from "rhea-promise";
import { delay, ErrorNameConditionMapper, RetryOperationType, defaultCancellableLock, retry, translate } from "@azure/core-amqp";
import { toRheaMessage } from "./eventData";
import { LinkEntity } from "./linkEntity";
import { getRetryAttemptTimeoutInMs } from "./util/retries";
import { isEventDataBatch } from "./eventDataBatch";
/**
 * Describes the EventHubSender that will send event data to EventHub.
 * @internal
 */
export class EventHubSender extends LinkEntity {
    /**
     * Creates a new EventHubSender instance.
     * @hidden
     * @param context - The connection context.
     * @param partitionId - The EventHub partition id to which the sender
     * wants to send the event data.
     */
    constructor(context, partitionId) {
        super(context, {
            name: context.config.getSenderAddress(partitionId),
            partitionId: partitionId
        });
        /**
         * The unique lock name per connection that is used to acquire the
         * lock for establishing a sender link by an entity on that connection.
         * @readonly
         */
        this.senderLock = `sender-${uuid()}`;
        this.address = context.config.getSenderAddress(partitionId);
        this.audience = context.config.getSenderAudience(partitionId);
        this._onAmqpError = (eventContext) => {
            const senderError = eventContext.sender && eventContext.sender.error;
            logger.verbose("[%s] 'sender_error' event occurred on the sender '%s' with address '%s'. " +
                "The associated error is: %O", this._context.connectionId, this.name, this.address, senderError);
            // TODO: Consider rejecting promise in trySendBatch() or createBatch()
        };
        this._onSessionError = (eventContext) => {
            const sessionError = eventContext.session && eventContext.session.error;
            logger.verbose("[%s] 'session_error' event occurred on the session of sender '%s' with address '%s'. " +
                "The associated error is: %O", this._context.connectionId, this.name, this.address, sessionError);
            // TODO: Consider rejecting promise in trySendBatch() or createBatch()
        };
        this._onAmqpClose = async (eventContext) => {
            const sender = this._sender || eventContext.sender;
            logger.verbose("[%s] 'sender_close' event occurred on the sender '%s' with address '%s'. " +
                "Value for isItselfClosed on the receiver is: '%s' " +
                "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, sender ? sender.isItselfClosed().toString() : undefined, this.isConnecting);
            if (sender && !this.isConnecting) {
                // Call close to clean up timers & other resources
                await sender.close().catch((err) => {
                    logger.verbose("[%s] Error when closing sender [%s] after 'sender_close' event: %O", this._context.connectionId, this.name, err);
                });
            }
        };
        this._onSessionClose = async (eventContext) => {
            const sender = this._sender || eventContext.sender;
            logger.verbose("[%s] 'session_close' event occurred on the session of sender '%s' with address '%s'. " +
                "Value for isSessionItselfClosed on the session is: '%s' " +
                "Value for isConnecting on the session is: '%s'.", this._context.connectionId, this.name, this.address, sender ? sender.isSessionItselfClosed().toString() : undefined, this.isConnecting);
            if (sender && !this.isConnecting) {
                // Call close to clean up timers & other resources
                await sender.close().catch((err) => {
                    logger.verbose("[%s] Error when closing sender [%s] after 'session_close' event: %O", this._context.connectionId, this.name, err);
                });
            }
        };
    }
    /**
     * Deletes the sender from the context. Clears the token renewal timer. Closes the sender link.
     * @hidden
     * @returns Promise<void>
     */
    async close() {
        try {
            if (this._sender) {
                logger.info("[%s] Closing the Sender for the entity '%s'.", this._context.connectionId, this._context.config.entityPath);
                const senderLink = this._sender;
                this._deleteFromCache();
                await this._closeLink(senderLink);
            }
        }
        catch (err) {
            const msg = `[${this._context.connectionId}] An error occurred while closing sender ${this.name}: ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`;
            logger.warning(msg);
            logErrorStackTrace(err);
            throw err;
        }
    }
    /**
     * Determines whether the AMQP sender link is open. If open then returns true else returns false.
     * @hidden
     * @returns boolean
     */
    isOpen() {
        const result = Boolean(this._sender && this._sender.isOpen());
        logger.verbose("[%s] Sender '%s' with address '%s' is open? -> %s", this._context.connectionId, this.name, this.address, result);
        return result;
    }
    /**
     * Returns maximum message size on the AMQP sender link.
     * @param abortSignal - An implementation of the `AbortSignalLike` interface to signal the request to cancel the operation.
     * For example, use the &commat;azure/abort-controller to create an `AbortSignal`.
     * @returns Promise<number>
     * @throws AbortError if the operation is cancelled via the abortSignal.
     */
    async getMaxMessageSize(options = {}) {
        const sender = await this._getLink(options);
        return sender.maxMessageSize;
    }
    /**
     * Send a batch of EventData to the EventHub. The "message_annotations",
     * "application_properties" and "properties" of the first message will be set as that
     * of the envelope (batch message).
     * @hidden
     * @param events -  An array of EventData objects to be sent in a Batch message.
     * @param options - Options to control the way the events are batched along with request options
     */
    async send(events, options) {
        try {
            logger.info("[%s] Sender '%s', trying to send EventData[].", this._context.connectionId, this.name);
            let encodedBatchMessage;
            if (isEventDataBatch(events)) {
                if (events.count === 0) {
                    logger.info(`[${this._context.connectionId}] Empty batch was passsed. No events to send.`);
                    return;
                }
                encodedBatchMessage = events._generateMessage();
            }
            else {
                if (events.length === 0) {
                    logger.info(`[${this._context.connectionId}] Empty array was passed. No events to send.`);
                    return;
                }
                const partitionKey = (options && options.partitionKey) || undefined;
                const messages = [];
                // Convert EventData to RheaMessage.
                for (let i = 0; i < events.length; i++) {
                    const rheaMessage = toRheaMessage(events[i], partitionKey);
                    messages[i] = rheaMessage;
                }
                // Encode every amqp message and then convert every encoded message to amqp data section
                const batchMessage = {
                    body: message.data_sections(messages.map(message.encode))
                };
                // Set message_annotations of the first message as
                // that of the envelope (batch message).
                if (messages[0].message_annotations) {
                    batchMessage.message_annotations = messages[0].message_annotations;
                }
                // Finally encode the envelope (batch message).
                encodedBatchMessage = message.encode(batchMessage);
            }
            logger.info("[%s] Sender '%s', sending encoded batch message.", this._context.connectionId, this.name, encodedBatchMessage);
            return await this._trySendBatch(encodedBatchMessage, options);
        }
        catch (err) {
            logger.warning(`An error occurred while sending the batch message ${err === null || err === void 0 ? void 0 : err.name}: ${err === null || err === void 0 ? void 0 : err.message}`);
            logErrorStackTrace(err);
            throw err;
        }
    }
    _deleteFromCache() {
        this._sender = undefined;
        delete this._context.senders[this.name];
        logger.verbose("[%s] Deleted the sender '%s' with address '%s' from the client cache.", this._context.connectionId, this.name, this.address);
    }
    _createSenderOptions(newName) {
        if (newName)
            this.name = `${uuid()}`;
        const srOptions = {
            name: this.name,
            target: {
                address: this.address
            },
            onError: this._onAmqpError,
            onClose: this._onAmqpClose,
            onSessionError: this._onSessionError,
            onSessionClose: this._onSessionClose
        };
        logger.verbose("Creating sender with options: %O", srOptions);
        return srOptions;
    }
    /**
     * Tries to send the message to EventHub if there is enough credit to send them
     * and the circular buffer has available space to settle the message after sending them.
     *
     * We have implemented a synchronous send over here in the sense that we shall be waiting
     * for the message to be accepted or rejected and accordingly resolve or reject the promise.
     * @hidden
     * @param rheaMessage - The message to be sent to EventHub.
     * @returns Promise<void>
     */
    async _trySendBatch(rheaMessage, options = {}) {
        const abortSignal = options.abortSignal;
        const retryOptions = options.retryOptions || {};
        const timeoutInMs = getRetryAttemptTimeoutInMs(retryOptions);
        retryOptions.timeoutInMs = timeoutInMs;
        const sendEventPromise = async () => {
            var _a, _b;
            const initStartTime = Date.now();
            const sender = await this._getLink(options);
            const timeTakenByInit = Date.now() - initStartTime;
            logger.verbose("[%s] Sender '%s', credit: %d available: %d", this._context.connectionId, this.name, sender.credit, sender.session.outgoing.available());
            let waitTimeForSendable = 1000;
            if (!sender.sendable() && timeoutInMs - timeTakenByInit > waitTimeForSendable) {
                logger.verbose("%s Sender '%s', waiting for 1 second for sender to become sendable", this._context.connectionId, this.name);
                await delay(waitTimeForSendable);
                logger.verbose("%s Sender '%s' after waiting for a second, credit: %d available: %d", this._context.connectionId, this.name, sender.credit, (_b = (_a = sender.session) === null || _a === void 0 ? void 0 : _a.outgoing) === null || _b === void 0 ? void 0 : _b.available());
            }
            else {
                waitTimeForSendable = 0;
            }
            if (!sender.sendable()) {
                // let us retry to send the message after some time.
                const msg = `[${this._context.connectionId}] Sender "${this.name}", ` +
                    `cannot send the message right now. Please try later.`;
                logger.warning(msg);
                const amqpError = {
                    condition: ErrorNameConditionMapper.SenderBusyError,
                    description: msg
                };
                throw translate(amqpError);
            }
            logger.verbose("[%s] Sender '%s', sending message with id '%s'.", this._context.connectionId, this.name);
            if (timeoutInMs <= timeTakenByInit + waitTimeForSendable) {
                const desc = `${this._context.connectionId} Sender "${this.name}" ` +
                    `with address "${this.address}", was not able to send the message right now, due ` +
                    `to operation timeout.`;
                logger.warning(desc);
                const e = {
                    condition: ErrorNameConditionMapper.ServiceUnavailableError,
                    description: desc
                };
                throw translate(e);
            }
            try {
                const delivery = await sender.send(rheaMessage, {
                    format: 0x80013700,
                    timeoutInSeconds: (timeoutInMs - timeTakenByInit - waitTimeForSendable) / 1000,
                    abortSignal
                });
                logger.info("[%s] Sender '%s', sent message with delivery id: %d", this._context.connectionId, this.name, delivery.id);
            }
            catch (err) {
                throw err.innerError || err;
            }
        };
        const config = {
            operation: sendEventPromise,
            connectionId: this._context.connectionId,
            operationType: RetryOperationType.sendMessage,
            abortSignal: abortSignal,
            retryOptions: retryOptions
        };
        try {
            await retry(config);
        }
        catch (err) {
            const translatedError = translate(err);
            logger.warning("[%s] Sender '%s', An error occurred while sending the message %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    async _getLink(options = {}) {
        if (this.isOpen() && this._sender) {
            return this._sender;
        }
        const retryOptions = options.retryOptions || {};
        const timeoutInMs = getRetryAttemptTimeoutInMs(retryOptions);
        retryOptions.timeoutInMs = timeoutInMs;
        const senderOptions = this._createSenderOptions();
        const startTime = Date.now();
        const createLinkPromise = async () => {
            return defaultCancellableLock.acquire(this.senderLock, () => {
                const taskStartTime = Date.now();
                const taskTimeoutInMs = timeoutInMs - (taskStartTime - startTime);
                return this._init(Object.assign(Object.assign({}, senderOptions), { abortSignal: options.abortSignal, timeoutInMs: taskTimeoutInMs }));
            }, { abortSignal: options.abortSignal, timeoutInMs: timeoutInMs });
        };
        const config = {
            operation: createLinkPromise,
            connectionId: this._context.connectionId,
            operationType: RetryOperationType.senderLink,
            abortSignal: options.abortSignal,
            retryOptions: retryOptions
        };
        try {
            return await retry(config);
        }
        catch (err) {
            const translatedError = translate(err);
            logger.warning("[%s] An error occurred while creating the sender %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Initializes the sender session on the connection.
     * Should only be called from _createLinkIfNotOpen
     * @hidden
     */
    async _init(options) {
        try {
            if (!this.isOpen() || !this._sender) {
                // Wait for the connectionContext to be ready to open the link.
                await this._context.readyToOpenLink();
                await this._negotiateClaim({
                    setTokenRenewal: false,
                    abortSignal: options.abortSignal,
                    timeoutInMs: options.timeoutInMs
                });
                logger.verbose("[%s] Trying to create sender '%s'...", this._context.connectionId, this.name);
                const sender = await this._context.connection.createAwaitableSender(options);
                this._sender = sender;
                logger.verbose("[%s] Sender '%s' created with sender options: %O", this._context.connectionId, this.name, options);
                sender.setMaxListeners(1000);
                // It is possible for someone to close the sender and then start it again.
                // Thus make sure that the sender is present in the client cache.
                if (!this._context.senders[this.name])
                    this._context.senders[this.name] = this;
                this._ensureTokenRenewal();
                return sender;
            }
            else {
                logger.verbose("[%s] The sender '%s' with address '%s' is open -> %s. Hence not reconnecting.", this._context.connectionId, this.name, this.address, this.isOpen());
                return this._sender;
            }
        }
        catch (err) {
            const translatedError = translate(err);
            logger.warning("[%s] An error occurred while creating the sender %s: %s", this._context.connectionId, this.name, `${translatedError === null || translatedError === void 0 ? void 0 : translatedError.name}: ${translatedError === null || translatedError === void 0 ? void 0 : translatedError.message}`);
            logErrorStackTrace(translatedError);
            throw translatedError;
        }
    }
    /**
     * Creates a new sender to the given event hub, and optionally to a given partition if it is
     * not present in the context or returns the one present in the context.
     * @hidden
     * @param partitionId - Partition ID to which it will send event data.
     */
    static create(context, partitionId) {
        const ehSender = new EventHubSender(context, partitionId);
        if (!context.senders[ehSender.name]) {
            context.senders[ehSender.name] = ehSender;
        }
        return context.senders[ehSender.name];
    }
}
//# sourceMappingURL=eventHubSender.js.map