// This file is a collection of functions taken from the original iaxclient_lib.c and maybe somewhere else into the iaxclient project.
// There could be some modified function somewhere so please have a look at the original at http://iaxclient.sourceforge.net

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include "iaxclient_lib.h"
#include <stdarg.h>


#define DEFAULT_CALLERID_NAME    "eja.it"
#define DEFAULT_CALLERID_NUMBER  "700352048"
#define IAXC_ERROR  IAXC_TEXT_TYPE_ERROR
#define IAXC_STATUS IAXC_TEXT_TYPE_STATUS
#define IAXC_NOTICE IAXC_TEXT_TYPE_NOTICE


struct iaxc_registration {
    struct iax_session *session;
    int firstpass;
    struct timeval last;
    char host[256];
    char user[256];
    char pass[256];
    long   refresh;
    struct iaxc_registration *next;
};

struct iaxc_registration *registrations = NULL;

static int iEncodeType;

MUTEX iaxc_lock;

int netfd;
int port;
int c, i;

static int selected_call;
static struct iaxc_call* calls;
static int nCalls;	

struct timeval lastouttm;

static void do_iax_event();

static THREAD procThread;

static int procThreadQuitFlag = -1;

static void iaxc_do_pings(void);

iaxc_event_callback_t iaxc_event_callback = NULL;

 
long iaxc_usecdiff( struct timeval *timeA, struct timeval *timeB ){
      long secs = timeA->tv_sec - timeB->tv_sec;
      long usecs = secs * 1000000;
      usecs += (timeA->tv_usec - timeB->tv_usec);
      return usecs;
}



void iaxc_post_event(iaxc_event e) {
    if(iaxc_event_callback)
    {
	int rv;
	rv = iaxc_event_callback(e);
	if(rv < 0) 
	  fprintf(stderr,"Miax: BIG PROBLEM, event callback returned failure!");
	if(rv > 0) return;
    }

    switch(e.type)
    {
	case IAXC_EVENT_TEXT:
	    fprintf(stderr,"Miax: %s\n", e.ev.text.message);
	return;
    }
}


void iaxc_usermsg(int type, const char *fmt, ...)
{
    va_list args;
    iaxc_event e;

    e.type=IAXC_EVENT_TEXT;
    e.ev.text.type=type;

    va_start(args, fmt);
    vsnprintf(e.ev.text.message, IAXC_EVENT_BUFSIZ, fmt, args);
    va_end(args);

    iaxc_post_event(e);
}


void iaxc_do_state_callback(int callNo)
{  
      iaxc_event e;   
      if(callNo < 0 || callNo >= nCalls) return;
      e.type = IAXC_EVENT_STATE;
      e.ev.call.callNo = callNo;
      e.ev.call.state = calls[callNo].state;
      strncpy(e.ev.call.remote,        calls[callNo].remote,        IAXC_EVENT_BUFSIZ);
      strncpy(e.ev.call.remote_name,   calls[callNo].remote_name,   IAXC_EVENT_BUFSIZ);
      strncpy(e.ev.call.local,         calls[callNo].local,         IAXC_EVENT_BUFSIZ);
      strncpy(e.ev.call.local_context, calls[callNo].local_context, IAXC_EVENT_BUFSIZ);
      iaxc_post_event(e);
}


int iaxc_first_free_call()  {
	int i;
	for(i=0;i<nCalls;i++) 
	    if(calls[i].state == IAXC_CALL_STATE_FREE) 
		return i;
	
	return -1;
}


static void iaxc_clear_call(int toDump)
{
      calls[toDump].state = IAXC_CALL_STATE_FREE;
      calls[toDump].session = NULL;
      iaxc_do_state_callback(toDump);
}


int iaxc_select_call(int callNo) {

	if(callNo >= nCalls) {
		iaxc_usermsg(IAXC_ERROR, "Error: tried to select out_of_range call %d", callNo);
		return -1;
	}
  
	if(callNo < 0) {
	    selected_call = callNo;
	    return 0;
	}
  
	if(callNo != selected_call) {
	    calls[selected_call].state &= ~IAXC_CALL_STATE_SELECTED;
	    selected_call = callNo;
	    iaxc_do_state_callback(selected_call);

	    calls[callNo].state |= IAXC_CALL_STATE_SELECTED;
	}


	if( !(calls[selected_call].state & IAXC_CALL_STATE_OUTGOING) && 
	     (calls[selected_call].state & IAXC_CALL_STATE_RINGING)) {
	    iaxc_answer_call(selected_call);
	} else {
	  iaxc_do_state_callback(selected_call);
	}

	return 0;
}
	  

int iaxc_initialize(int inCalls) {
	int i;

	MUTEXINIT(&iaxc_lock);

	if ( (port = iax_init(0) < 0)) {
		iaxc_usermsg(IAXC_ERROR, "Fatal error: failed to initialize iax with port %d", port);
		return -1;
	}
	netfd = iax_get_fd();

	nCalls = inCalls;
	if(nCalls == 0) nCalls = 1; /* 0 == Default? */

	calls = calloc(sizeof(struct iaxc_call), nCalls);
	if(!calls)
	{
		iaxc_usermsg(IAXC_ERROR, "Fatal error: can't allocate memory");
		return -1;
	}
	selected_call = 0;

	for(i=0; i<nCalls; i++) {
	    strncpy(calls[i].callerid_name,   DEFAULT_CALLERID_NAME,   IAXC_EVENT_BUFSIZ);
	    strncpy(calls[i].callerid_number, DEFAULT_CALLERID_NUMBER, IAXC_EVENT_BUFSIZ);
	}

	gettimeofday(&lastouttm,NULL);
	return 0;
}


void iaxc_shutdown() {

	iaxc_dump_all_calls();
	MUTEXLOCK(&iaxc_lock);
	MUTEXUNLOCK(&iaxc_lock);
	MUTEXDESTROY(&iaxc_lock);
	}


void iaxc_set_encode_format(int fmt)
{
	iEncodeType = fmt;
	iax_set_formats(fmt);
}


void iaxc_set_callerid(char *name, char *number) {
    int i;
    
    for(i=0; i<nCalls; i++) {
        strncpy(calls[i].callerid_name,   name,   IAXC_EVENT_BUFSIZ);
        strncpy(calls[i].callerid_number, number, IAXC_EVENT_BUFSIZ);
    }
}


static void iaxc_note_activity(int callNo) {
  if(callNo < 0)
      return;
  gettimeofday(&calls[callNo].last_activity, NULL);   
}


static void iaxc_do_pings(void) {
  int i;
  struct timeval now;

  gettimeofday(&now, NULL);
  for(i = 0; i < nCalls; i++)
  {
      long act_since;
      long ping_since;

      if(!(calls[i].state & IAXC_CALL_STATE_ACTIVE))
	  break;

      act_since = iaxc_usecdiff(&now, &calls[i].last_activity)/1000;

      if(act_since < IAXC_CALL_TIMEOUT/3)
	  break;  

      ping_since = iaxc_usecdiff(&now, &calls[i].last_ping)/1000;

      if(ping_since > IAXC_CALL_TIMEOUT/3) { 
	  calls[i].last_ping = now;
	  iax_send_ping(calls[i].session);
	  break; 
      }

      if(act_since > IAXC_CALL_TIMEOUT) {
	  iax_hangup(calls[i].session,"Timed out waiting for ping or activity");
	  iaxc_usermsg(IAXC_STATUS, "call %d timed out (ping/act = %ld/%ld)", i, ping_since/1000, act_since/1000);
	  iaxc_clear_call(i);
      }

  }

}


void iaxc_refresh_registrations() {
    struct iaxc_registration *cur;
    struct timeval now;

    gettimeofday(&now,NULL);

    for(cur = registrations; cur != NULL; cur=cur->next) {
	if(iaxc_usecdiff(&now, &cur->last) > cur->refresh ) {
	    fprintf(stderr, "refreshing registration %s:%s@%s\n", 
		cur->user, cur->pass, cur->host);

	    cur->session = iax_session_new();
	    if(!cur->session) {
		    iaxc_usermsg(IAXC_ERROR, "Can't make new registration session");
		    return;
	    }

	    iax_register(cur->session, cur->host, cur->user, cur->pass, 60);
	    cur->last = now;
	}
    }
}


void iaxc_process_calls(void) {

    MUTEXLOCK(&iaxc_lock);
    iaxc_service_network(netfd);
    iaxc_do_pings();
    audio_in(&calls[selected_call]);
    iaxc_refresh_registrations();
    MUTEXUNLOCK(&iaxc_lock);
}


THREADFUNCDECL(iaxc_processor)
{
    THREADFUNCRET(ret);
    while(1) { 
	iaxc_process_calls();
	iaxc_millisleep(5);	
	if(procThreadQuitFlag)
	  break;
    }
    return ret;
}


int iaxc_start_processing_thread()
{
      procThreadQuitFlag = 0;
      if( THREADCREATE(iaxc_processor, NULL, procThread, procThreadID) 
	    == THREADCREATE_ERROR)	
	  return -1;

      return 0;
}


int iaxc_stop_processing_thread()
{
    if(procThreadQuitFlag >= 0)
    {
	procThreadQuitFlag = 1;
    }
    procThreadQuitFlag = -1;
    return 0;
}


void handle_text_event(struct iax_event *e, int callNo) {
    iaxc_event ev;

   if(callNo < 0)
       return;
    ev.type=IAXC_EVENT_TEXT;
    ev.ev.text.type=IAXC_TEXT_TYPE_IAX;
    ev.ev.text.callNo = callNo;

    strncpy(ev.ev.text.message, e->data, IAXC_EVENT_BUFSIZ);
    iaxc_post_event(ev);
}


void iaxc_handle_network_event(struct iax_event *e, int callNo) {

        if(callNo < 0)
            return;

	iaxc_note_activity(callNo);

	switch(e->etype) {
		case IAX_EVENT_HANGUP:
			iaxc_usermsg(IAXC_STATUS, "Call disconnected by remote");
			iaxc_clear_call(callNo);
			break;
		case IAX_EVENT_REJECT:
			iaxc_usermsg(IAXC_STATUS, "Call rejected by remote");
			iaxc_clear_call(callNo);
			break;
		case IAX_EVENT_ACCEPT:
			calls[callNo].state |= IAXC_CALL_STATE_RINGING;	
			iaxc_do_state_callback(callNo);
			iaxc_usermsg(IAXC_STATUS,"Call %d ringing", callNo);
			break;
		case IAX_EVENT_ANSWER:
			calls[callNo].state &= ~IAXC_CALL_STATE_RINGING;	
			calls[callNo].state |= IAXC_CALL_STATE_COMPLETE;	
			iaxc_do_state_callback(callNo);
			iaxc_usermsg(IAXC_STATUS,"Call %d answered", callNo);
 			break;
		case IAX_EVENT_VOICE:
			audio_out(&calls[selected_call],e->data);
			break;
		case IAX_EVENT_TEXT:
			handle_text_event(e, callNo);
			break;
		case IAX_EVENT_RINGA:
			break;
		case IAX_EVENT_PONG:  
			break;
		default:
			iaxc_usermsg(IAXC_STATUS, "Unknown event: %d for call %d", e->etype, callNo);
			break;
		}
	miax_callback(e, callNo);
	}


void iaxc_register(char *user, char *pass, char *host)
{
	struct iaxc_registration *newreg;

	newreg = malloc(sizeof (struct iaxc_registration));
	if(!newreg) {
		iaxc_usermsg(IAXC_ERROR, "Can't make new registration");
		return;
	}

	newreg->session = iax_session_new();
	if(!newreg->session) {
		iaxc_usermsg(IAXC_ERROR, "Can't make new registration session");
		return;
	}

	gettimeofday(&newreg->last,NULL);
	newreg->refresh = 60*1000*1000;  // 60 seconds, in usecs

	strncpy(newreg->host, host, 256);
	strncpy(newreg->user, user, 256);
	strncpy(newreg->pass, pass, 256);

	newreg->firstpass = 1;

	iax_register(newreg->session, host, user, pass, 300);

	newreg->next = registrations;
	registrations = newreg;
}


void iaxc_call(char *num)
{
	int callNo;
	struct iax_session *newsession;
	char *ext = strstr(num, "/");

	MUTEXLOCK(&iaxc_lock);

        if(selected_call < 0) {
            callNo = iaxc_first_free_call();
        } else {
            if(calls[selected_call].state  & IAXC_CALL_STATE_ACTIVE) {
                callNo = iaxc_first_free_call();
            } else {
                 callNo = selected_call;
            }
        }

	if(callNo < 0) {
		iaxc_usermsg(IAXC_STATUS, "No free call appearances");
		goto iaxc_call_bail;
	}

	newsession = iax_session_new();
	if(!newsession) {
		iaxc_usermsg(IAXC_ERROR, "Can't make new session");
		goto iaxc_call_bail;
	}

	calls[callNo].session = newsession;

	calls[callNo].gsmin = 0;
	calls[callNo].gsmout = 0;

	if(ext) {
	    strncpy(calls[callNo].remote_name, num, IAXC_EVENT_BUFSIZ); 
    	    strncpy(calls[callNo].remote,    ++ext, IAXC_EVENT_BUFSIZ);
    	} else {
	    strncpy(calls[callNo].remote_name, num, IAXC_EVENT_BUFSIZ);
    	    strncpy(calls[callNo].remote,      "" , IAXC_EVENT_BUFSIZ);
     	}

 	strncpy(calls[callNo].local        , calls[callNo].callerid_name, IAXC_EVENT_BUFSIZ);
	strncpy(calls[callNo].local_context, "default", IAXC_EVENT_BUFSIZ);

	calls[callNo].state = IAXC_CALL_STATE_ACTIVE | IAXC_CALL_STATE_OUTGOING;

	iaxc_note_activity(callNo);
	calls[callNo].last_ping = calls[callNo].last_activity;

	iax_call(calls[callNo].session, calls[callNo].callerid_number,
	                                calls[callNo].callerid_name, num, NULL, 0);

	iaxc_select_call(callNo);


iaxc_call_bail:
	MUTEXUNLOCK(&iaxc_lock);
}


void iaxc_answer_call(int callNo) 
{

	if(callNo < 0)
	    return;
	    
	calls[callNo].state |= IAXC_CALL_STATE_COMPLETE;
	calls[callNo].state &= ~IAXC_CALL_STATE_RINGING;
	iax_answer(calls[callNo].session);
	iaxc_do_state_callback(callNo);
}


void iaxc_blind_transfer_call(int callNo, char *DestExtn)
{
	iax_transfer(calls[callNo].session, DestExtn);
}


static void iaxc_dump_one_call(int callNo)
{
      if(callNo < 0)
          return;
      if(calls[callNo].state == IAXC_CALL_STATE_FREE) return;
      
      iax_hangup(calls[callNo].session,"Dumped Call");
      iaxc_usermsg(IAXC_STATUS, "Hanging up call %d", callNo);
      iaxc_clear_call(callNo);
}


void iaxc_dump_all_calls(void)
{
      int callNo;
      MUTEXLOCK(&iaxc_lock);
	for(callNo=0; callNo<nCalls; callNo++)
	    iaxc_dump_one_call(callNo);
      MUTEXUNLOCK(&iaxc_lock);
}


void iaxc_dump_call(void)
{
    if(selected_call >= 0) {
	MUTEXLOCK(&iaxc_lock);
	iaxc_dump_one_call(selected_call);
	MUTEXUNLOCK(&iaxc_lock);
    }
}


void iaxc_reject_call(void)
{
    if(selected_call >= 0) {
	MUTEXLOCK(&iaxc_lock);
	iax_reject(calls[selected_call].session, "Call rejected manually.");
	iaxc_clear_call(selected_call);
	MUTEXUNLOCK(&iaxc_lock);
    }
}


void iaxc_send_dtmf(char digit)
{
    if(selected_call >= 0) {
	MUTEXLOCK(&iaxc_lock);
	if(calls[selected_call].state & IAXC_CALL_STATE_ACTIVE)
		iax_send_dtmf(calls[selected_call].session,digit);
	MUTEXUNLOCK(&iaxc_lock);
    }
}


static int iaxc_find_call_by_session(struct iax_session *session)
{
	int i;
	for(i=0;i<nCalls;i++)
		if (calls[i].session == session)
			return i;
	return -1;
}


void iaxc_service_network(int netfd)
{
	fd_set readfd;
	struct timeval dumbtimer;

	dumbtimer.tv_sec = 0;
	dumbtimer.tv_usec = 0;


		for(;;) 
		{
			FD_ZERO(&readfd);
			FD_SET(netfd, &readfd);
			if (select(netfd + 1, &readfd, 0, 0, &dumbtimer) > 0)
			{
				if (FD_ISSET(netfd,&readfd))
				{
					do_iax_event();
				} else break;
			} else break;
		}
		do_iax_event(); 
}


static void iaxc_handle_regreply(struct iax_event *e) {
  struct iaxc_registration *cur;

    for(cur = registrations; cur != NULL; cur=cur->next) 
	if(cur->session == e->session) break;

    if(!cur) {
	iaxc_usermsg(IAXC_ERROR, "Unexpected registration reply");
	return;
    }

    if(cur->firstpass) {
	cur->firstpass = 0;
      
	if(e->etype == IAX_EVENT_REGACK ) {
	    iaxc_usermsg(IAXC_STATUS, "Registration accepted");
	} else if(e->etype == IAX_EVENT_REGREJ ) {
	    iaxc_usermsg(IAXC_STATUS, "Registration rejected");
	}
    }

    iax_destroy(cur->session);
    cur->session = NULL;
}


static void do_iax_event() {
	struct iax_event *e = 0;
	int callNo;

	while ( (e = iax_get_event(0))) {
		callNo = iaxc_find_call_by_session(e->session);
		if(callNo >= 0) {
			iaxc_handle_network_event(e, callNo);
		} else if 
		((e->etype == IAX_EVENT_REGACK ) || (e->etype == IAX_EVENT_REGREJ ))
		{ 
		    iaxc_handle_regreply(e);
		} else if(e->etype == IAX_EVENT_REGREQ ) {
			iaxc_usermsg(IAXC_ERROR, "Registration requested by someone, but we don't understand!");
		} else  if(e->etype == IAX_EVENT_CONNECT) {
			
			callNo = iaxc_first_free_call();

			if(callNo < 0) {
				iaxc_usermsg(IAXC_STATUS, "Incoming Call, but no appearances");
				iax_reject(e->session, "Too many calls, we're busy!");
				goto bail;
			}

			if(e->ies.called_number)
			    strncpy(calls[callNo].local,e->ies.called_number,
				IAXC_EVENT_BUFSIZ);
			else
			    strncpy(calls[callNo].local,"unknown",
				IAXC_EVENT_BUFSIZ);

			if(e->ies.called_context)
			    strncpy(calls[callNo].local_context,e->ies.called_context,
				IAXC_EVENT_BUFSIZ);
			else
			    strncpy(calls[callNo].local_context,"",
    				IAXC_EVENT_BUFSIZ);

			if(e->ies.calling_number)
			    strncpy(calls[callNo].remote,
  				e->ies.calling_number, IAXC_EVENT_BUFSIZ);
			else
			    strncpy(calls[callNo].remote,
    				"unknown", IAXC_EVENT_BUFSIZ);

			if(e->ies.calling_name)
			    strncpy(calls[callNo].remote_name,
    				e->ies.calling_name, IAXC_EVENT_BUFSIZ);
			else
			    strncpy(calls[callNo].remote_name,
    				"unknown", IAXC_EVENT_BUFSIZ);

			iaxc_note_activity(callNo);
			iaxc_usermsg(IAXC_STATUS, "Call from (%s)", calls[callNo].remote);

			calls[callNo].gsmin = 0;
			calls[callNo].gsmout = 0;
			calls[callNo].session = e->session;
			calls[callNo].state = IAXC_CALL_STATE_ACTIVE|IAXC_CALL_STATE_RINGING;


			// should we even accept?  or, we accept, but
			// don't necessarily answer..
			iax_accept(calls[callNo].session);
			iax_ring_announce(calls[callNo].session);

			iaxc_do_state_callback(callNo);

			iaxc_usermsg(IAXC_STATUS, "Incoming call on line %d", callNo);
			
			miax_callback(e,callNo);
		} else {
			iaxc_usermsg(IAXC_STATUS, "Event (type %d) for a non-existant session.  Dropping", e->etype);
		}
bail:
		iax_event_free(e);
	}
}


int iaxc_quelch(int callNo, int MOH)
{
	struct iax_session *session = calls[callNo].session;
	if (!session)
		return -1;

	return iax_quelch_moh(session, MOH);
}


int iaxc_unquelch(int call)
{
	return iax_unquelch(calls[call].session);
}


void iaxc_millisleep(long ms) {
        struct timespec req;
        
        req.tv_nsec=10*1000*1000;  
        req.tv_sec=0;
                        
        nanosleep(&req,NULL);
	}
                                        

