--[[
# Copyright 2001-2014 Cisco Systems, Inc. and/or its affiliates. All rights
# reserved.
#
# This file contains proprietary Detector Content created by Cisco Systems,
# Inc. or its affiliates ("Cisco") and is distributed under the GNU General
# Public License, v2 (the "GPL").  This file may also include Detector Content
# contributed by third parties. Third party contributors are identified in the
# "authors" file.  The Detector Content created by Cisco is owned by, and
# remains the property of, Cisco.  Detector Content from third party
# contributors is owned by, and remains the property of, such third parties and
# is distributed under the GPL.  The term "Detector Content" means specifically
# formulated patterns and logic to identify applications based on network
# traffic characteristics, comprised of instructions in source code or object
# code form (including the structure, sequence, organization, and syntax
# thereof), and all documentation related thereto that have been officially
# approved by Cisco.  Modifications are considered part of the Detector
# Content.
--]]
--[[
detection_name: PcAnywhere
version: 3
description: PC remote control software.
bundle_description: $VAR1 = {
          'PcAnywhere' => 'PC remote control software.'
        };

--]]

require "DetectorCommon"



local DC = DetectorCommon
local FT = flowTrackerModule

gServiceId = 20088
gServiceName = 'PcAnywhere'
gDetector = nil

DetectorPackageInfo = {
    name =  "PcAnywhere",
    proto =  DC.ipproto.tcp,
    server = {
        init =  'DetectorInit',
        validate =  'DetectorValidator',
    }
}

gSfAppIdPcAnywhere = 781

gPatterns = {
	pattern1 = { "\000X\008\000\125\008\013\010", 0, gSfAppIdPcAnywhere},
	pattern2= { "NR", 0, gSfAppIdPcAnywhere},
	pattern3= { "ST\000\001", 0, gSfAppIdPcAnywhere},
}

gFastPatterns = {
    {DC.ipproto.tcp, gPatterns.pattern1},
	{DC.ipproto.udp, gPatterns.pattern2},
	{DC.ipproto.udp, gPatterns.pattern3},   
}
	
gPorts = {
    {DC.ipproto.udp, 5632},
    {DC.ipproto.tcp, 5631},
}


gAppRegistry = {
	--AppIdValue          Extracts Info
	---------------------------------------
	{gSfAppIdPcAnywhere,		         1}
}

function serviceInProcess(context)

    local flowFlag = context.detectorFlow:getFlowFlag(DC.flowFlags.serviceDetected)

    if ((not flowFlag) or (flowFlag == 0)) then
        gDetector:inProcessService()
    end

    DC.printf('%s: Inprocess, packetCount: %d\n', gServiceName, context.packetCount);
    return DC.serviceStatus.inProcess
end

function serviceSuccess(context)
    local flowFlag = context.detectorFlow:getFlowFlag(DC.flowFlags.serviceDetected)

    if ((not flowFlag) or (flowFlag == 0)) then
        gDetector:addService(gServiceId, "Symantec", "", gSfAppIdPcAnywhere);
    end

    DC.printf('%s: Detected, packetCount: %d\n', gServiceName, context.packetCount);
    return DC.serviceStatus.success
end

function serviceFail(context)
    local flowFlag = context.detectorFlow:getFlowFlag(DC.flowFlags.serviceDetected)

    if ((not flowFlag) or (flowFlag == 0)) then
        gDetector:failService()
    end

    context.detectorFlow:clearFlowFlag(DC.flowFlags.continue)
    DC.printf('%s: Failed, packetCount: %d\n', gServiceName, context.packetCount);
    return DC.serviceStatus.nomatch
end

function registerPortsPatterns()

    --register port based detection
    for i,v in ipairs(gPorts) do
        gDetector:addPort(v[1], v[2])
    end

    --register pattern based detection
    for i,v in ipairs(gFastPatterns) do
        if ( gDetector:registerPattern(v[1], v[2][1], #v[2][1], v[2][2], v[2][3]) ~= 0) then
            DC.printf ('%s: register pattern failed for %s\n', gServiceName,v[2][1])
        else
            DC.printf ('%s: register pattern successful for %s\n', gServiceName,v[2][1])
        end
    end

	for i,v in ipairs(gAppRegistry) do
		pcall(function () gDetector:registerAppId(v[1],v[2]) end)
	end

end

--[[ Core engine calls DetectorInit() to initialize a detector.
--]]
function DetectorInit( detectorInstance)
    gDetector = detectorInstance
    DC.printf ('%s:DetectorInit()\n', gServiceName);
    gDetector:init(gServiceName, 'DetectorValidator', 'DetectorFini')
    registerPortsPatterns()

    return gDetector
end


--[[Validator function registered in DetectorInit()
--]]
function DetectorValidator()
    local context = {}
    context.detectorFlow = gDetector:getFlow()
    context.packetDataLen = gDetector:getPacketSize()
    context.packetDir = gDetector:getPacketDir()
    context.srcIp = gDetector:getPktSrcAddr()
    context.dstIp = gDetector:getPktDstAddr()
    context.srcPort = gDetector:getPktSrcPort()
    context.dstPort = gDetector:getPktDstPort()
    context.flowKey = context.detectorFlow:getFlowKey()
    context.packetCount = gDetector:getPktCount()
    local size = context.packetDataLen
    local dir = context.packetDir
    local srcPort = context.srcPort
    local dstPort = context.dstPort
    local flowKey = context.flowKey

    DC.printf ('%s:DetectorValidator(): packetCount %d, dir %d, size %d\n', gServiceName,
               context.packetCount, dir, size);

	if (size == 0 or dir == 0) then
		return serviceInProcess(context)
	end

	--for PCAnywhere, expect either a greeting or a query response
	--check first for service greeting
    if ((size >= #gPatterns.pattern1[1]) and
		(srcPort == 5631) and
    	(gDetector:memcmp(gPatterns.pattern1[1], #gPatterns.pattern1[1], gPatterns.pattern1[2]) == 0))
    then
    	return serviceSuccess(context)   
    end

	--check for the two possible query responses, name response or status response
	--name response
	if ((size >= 34) and
		(srcPort == 5632) and
		(gDetector:memcmp(gPatterns.pattern2[1], #gPatterns.pattern2[1], gPatterns.pattern2[2]) == 0) and
		(gDetector:getPcreGroups("NR" .. string.rep("[a-zA-Z0-9_i\\-]", 32), 0)))
	then 
		return serviceSuccess(context)
	end

	--status response
	if ((size >= 5) and
		(srcPort == 5632) and
		(gDetector:memcmp(gPatterns.pattern3[1], #gPatterns.pattern3[1], gPatterns.pattern3[2]) == 0))
    then 
		matched, statusPattern = gDetector:getPcreGroups("ST(...)", 0)
		if (matched and (statusPattern == '\000\001\011' or statusPattern == '\000\001\067')) then 
        	return serviceSuccess(context)
		end
    end

	sawGreeting = false
	return serviceFail(context)
end




--[[Required DetectorFini function
--]]
function DetectorFini()
    --print (gServiceName .. ': DetectorFini()')
end

