/*
* Copyright (c) 2013 Jonathan Perkin <jonathan@perkin.org.uk>
* Copyright (c) 2015-2017 Elias Karakoulakis <elias.karakoulakis@gmail.com>
*
* Permission to use, copy, modify, and distribute this software for any
* purpose with or without fee is hereby granted, provided that the above
* copyright notice and this permission notice appear in all copies.
*
* THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
* WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
* MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
* ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
* WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
* ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
* OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/

#include "openzwave.hpp"

using namespace v8;
using namespace node;

namespace OZW {

 	// Get the time period between polls of a node's state
	// ===================================================================
	NAN_METHOD(OZW::GetPollInterval)
	// ===================================================================
	{
		Nan::HandleScope scope;
		uint32 pi = -1;
		OZWManagerAssign(pi, GetPollInterval );
		info.GetReturnValue().Set(Nan::New<Integer>(pi));
	}

	/* Set the time period between polls of a node's state. Due to patent
	 * concerns, some devices do not report state changes automatically to
	 * the controller. These devices need to have their state polled at
	 * regular intervals. The length of the interval is the same for all
	 * devices. To even out the Z-Wave network traffic generated by
	 * polling, OpenZWave divides the polling interval by the number of
	 * devices that have polling enabled, and polls each in turn.
	 * It is recommended that if possible, the interval should not be set
	 * shorter than the number of polled devices in seconds (so that the
	 * network does not have to cope with more than one poll per second).
	 */
	// ===================================================================
	NAN_METHOD(OZW::SetPollInterval)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(1, "intervalMillisecs");
		uint32 intervalMillisecs = Nan::To<Number>(info[0]).ToLocalChecked()->Value();
		OZWManager( SetPollInterval, intervalMillisecs, false);
	}

	/*
	* Enable/Disable polling on a ValueID
	*/
	// ===================================================================
	NAN_METHOD(OZW::EnablePoll)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(2, "valueId_object, intensity");
		OpenZWave::ValueID* vit = populateValueId(info);
		bool b = false;
		if (vit) {
			uint8 idxpos  =  (info[0]->IsObject()) ? 1 : 4;
			uint8 intensity = Nan::To<Number>(info[idxpos]).ToLocalChecked()->Value();

			OZWManagerAssign(b, EnablePoll, (*vit), intensity);
		}
		info.GetReturnValue().Set(Nan::New<Boolean>(b));
	}

	// ===================================================================
	NAN_METHOD(OZW::DisablePoll)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(1, "valueId");
		OpenZWave::ValueID* vit = populateValueId(info);
		bool b = false;
		if (vit) {
			OZWManagerAssign(b, DisablePoll, (*vit));
		}
		info.GetReturnValue().Set(Nan::New<Boolean>(b));
	}

	// Determine the polling of a device's state.
	// ===================================================================
	NAN_METHOD(OZW::IsPolled)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(1, "valueId");
		OpenZWave::ValueID* ozwvid = populateValueId(info);
		if (ozwvid) {
			bool b = false;
			OZWManagerAssign(b, isPolled, *ozwvid);
			info.GetReturnValue().Set(Nan::New<Boolean>(b));
		}
	}

	// ===================================================================
	NAN_METHOD(OZW::SetPollIntensity)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(1, "valueId");
		OpenZWave::ValueID* ozwvid = populateValueId(info);
		uint8 intensity;
		if (ozwvid) {
			uint8 intensity_index = ( info[0]->IsObject() ) ? 1 : 4;
			intensity = Nan::To<Number>(info[intensity_index]).ToLocalChecked()->Value();
			OZWManager( SetPollIntensity, *ozwvid, intensity);
		}
	}

 	// Get the polling intensity of a device's state.
	// ===================================================================
	NAN_METHOD(OZW::GetPollIntensity)
	// ===================================================================
	{
		Nan::HandleScope scope;
		CheckMinArgs(1, "valueId");
		OpenZWave::ValueID* ozwvid = populateValueId(info);
		if (ozwvid) {
			uint8 i = -1;
			OZWManagerAssign(i, GetPollIntensity, *ozwvid);
			info.GetReturnValue().Set(Nan::New<Integer>(i));
		}
	}
}
