package main

import (
	"fmt"
	"reflect"
	"strings"
	"time"
)

const (
	tagForQuery    = "query"
	tagForRequired = "required"
)

func (d *Descriptor) genNewRequestQueryElemString(sb *strings.Builder, f *reflect.StructField) {
	name := f.Name
	query := f.Tag.Get(tagForQuery)
	if f.Tag.Get(tagForRequired) == "true" {
		fmt.Fprintf(sb, "\tif req.%s == \"\" {\n", name)
		fmt.Fprintf(sb, "\t\treturn nil, newErrEmptyField(\"%s\")\n", name)
		fmt.Fprint(sb, "\t}\n")
		fmt.Fprintf(sb, "\tq.Add(\"%s\", req.%s)\n", query, name)
		return
	}
	fmt.Fprintf(sb, "\tif req.%s != \"\" {\n", name)
	fmt.Fprintf(sb, "\t\tq.Add(\"%s\", req.%s)\n", query, name)
	fmt.Fprint(sb, "\t}\n")
}

func (d *Descriptor) genNewRequestQueryElemBool(sb *strings.Builder, f *reflect.StructField) {
	// required does not make much sense for a boolean field
	name := f.Name
	query := f.Tag.Get(tagForQuery)
	fmt.Fprintf(sb, "\tif req.%s {\n", name)
	fmt.Fprintf(sb, "\t\tq.Add(\"%s\", \"true\")\n", query)
	fmt.Fprint(sb, "\t}\n")
}

func (d *Descriptor) genNewRequestQueryElemInt64(sb *strings.Builder, f *reflect.StructField) {
	// required does not make much sense for an integer field
	name := f.Name
	query := f.Tag.Get(tagForQuery)
	fmt.Fprintf(sb, "\tif req.%s != 0 {\n", name)
	fmt.Fprintf(sb, "\t\tq.Add(\"%s\", newQueryFieldInt64(req.%s))\n", query, name)
	fmt.Fprint(sb, "\t}\n")
}

func (d *Descriptor) genNewRequestQuery(sb *strings.Builder) {
	if d.Method != "GET" {
		return // we only generate query for GET
	}
	fields := d.StructFieldsWithTag(d.Request, tagForQuery)
	if len(fields) <= 0 {
		return
	}
	fmt.Fprint(sb, "\tq := url.Values{}\n")
	for idx, f := range fields {
		switch f.Type.Kind() {
		case reflect.String:
			d.genNewRequestQueryElemString(sb, f)
		case reflect.Bool:
			d.genNewRequestQueryElemBool(sb, f)
		case reflect.Int64:
			d.genNewRequestQueryElemInt64(sb, f)
		default:
			panic(fmt.Sprintf("unexpected query type at index %d", idx))
		}
	}
	fmt.Fprint(sb, "\tURL.RawQuery = q.Encode()\n")
}

func (d *Descriptor) genNewRequestCallNewRequest(sb *strings.Builder) {
	if d.Method == "POST" {
		fmt.Fprint(sb, "\tbody, err := api.jsonCodec().Encode(req)\n")
		fmt.Fprint(sb, "\tif err != nil {\n")
		fmt.Fprint(sb, "\t\treturn nil, err\n")
		fmt.Fprint(sb, "\t}\n")
		fmt.Fprint(sb, "\tout, err := api.requestMaker().NewRequest(")
		fmt.Fprintf(sb, "ctx, \"%s\", URL.String(), ", d.Method)
		fmt.Fprint(sb, "bytes.NewReader(body))\n")
		fmt.Fprint(sb, "\tif err != nil {\n")
		fmt.Fprint(sb, "\t\treturn nil, err\n")
		fmt.Fprint(sb, "\t}\n")
		fmt.Fprint(sb, "\tout.Header.Set(\"Content-Type\", \"application/json\")\n")
		fmt.Fprint(sb, "\treturn out, nil\n")
		return
	}
	fmt.Fprint(sb, "\treturn api.requestMaker().NewRequest(")
	fmt.Fprintf(sb, "ctx, \"%s\", URL.String(), ", d.Method)
	fmt.Fprint(sb, "nil)\n")
}

func (d *Descriptor) genNewRequest(sb *strings.Builder) {

	fmt.Fprintf(
		sb, "func (api *%s) newRequest(ctx context.Context, req %s) %s {\n",
		d.APIStructName(), d.RequestTypeName(), "(*http.Request, error)")
	fmt.Fprint(sb, "\tURL, err := url.Parse(api.baseURL())\n")
	fmt.Fprint(sb, "\tif err != nil {\n")
	fmt.Fprint(sb, "\t\treturn nil, err\n")
	fmt.Fprint(sb, "\t}\n")

	switch d.URLPath.IsTemplate {
	case false:
		fmt.Fprintf(sb, "\tURL.Path = \"%s\"\n", d.URLPath.Value)
	case true:
		fmt.Fprintf(
			sb, "\tup, err := api.templateExecutor().Execute(\"%s\", req)\n",
			d.URLPath.Value)
		fmt.Fprint(sb, "\tif err != nil {\n")
		fmt.Fprint(sb, "\t\treturn nil, err\n")
		fmt.Fprint(sb, "\t}\n")
		fmt.Fprint(sb, "\tURL.Path = up\n")
	}

	d.genNewRequestQuery(sb)
	d.genNewRequestCallNewRequest(sb)

	fmt.Fprintf(sb, "}\n\n")
}

// GenRequestsGo generates requests.go.
func GenRequestsGo(file string) {
	var sb strings.Builder
	fmt.Fprint(&sb, "// Code generated by go generate; DO NOT EDIT.\n")
	fmt.Fprintf(&sb, "// %s\n\n", time.Now())
	fmt.Fprint(&sb, "package ooapi\n\n")
	fmt.Fprintf(&sb, "//go:generate go run ./internal/generator -file %s\n\n", file)
	fmt.Fprint(&sb, "import (\n")
	fmt.Fprint(&sb, "\t\"bytes\"\n")
	fmt.Fprint(&sb, "\t\"context\"\n")
	fmt.Fprint(&sb, "\t\"net/http\"\n")
	fmt.Fprint(&sb, "\t\"net/url\"\n")
	fmt.Fprint(&sb, "\n")
	fmt.Fprint(&sb, "\t\"github.com/ooni/probe-cli/v3/internal/ooapi/apimodel\"\n")
	fmt.Fprint(&sb, ")\n\n")
	for _, desc := range Descriptors {
		desc.genNewRequest(&sb)
	}
	writefile(file, &sb)
}
