package procfs

import (
	"testing"
)

func TestNetDevParseLine(t *testing.T) {
	const rawLine = `  eth0: 1 2 3    4    5     6          7         8 9  10    11    12    13     14       15          16`

	have, err := NetDev{}.parseLine(rawLine)
	if err != nil {
		t.Fatal(err)
	}

	want := NetDevLine{"eth0", 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16}
	if want != *have {
		t.Errorf("want %v, have %v", want, have)
	}
}

func TestNewNetDev(t *testing.T) {
	fs, err := NewFS("fixtures")
	if err != nil {
		t.Fatal(err)
	}

	nd, err := fs.NewNetDev()
	if err != nil {
		t.Fatal(err)
	}

	lines := map[string]NetDevLine{
		"vethf345468": {Name: "vethf345468", RxBytes: 648, RxPackets: 8, TxBytes: 438, TxPackets: 5},
		"lo":          {Name: "lo", RxBytes: 1664039048, RxPackets: 1566805, TxBytes: 1664039048, TxPackets: 1566805},
		"docker0":     {Name: "docker0", RxBytes: 2568, RxPackets: 38, TxBytes: 438, TxPackets: 5},
		"eth0":        {Name: "eth0", RxBytes: 874354587, RxPackets: 1036395, TxBytes: 563352563, TxPackets: 732147},
	}

	if want, have := len(lines), len(nd); want != have {
		t.Errorf("want %d parsed net/dev lines, have %d", want, have)
	}
	for _, line := range nd {
		if want, have := lines[line.Name], line; want != have {
			t.Errorf("%s: want %v, have %v", line.Name, want, have)
		}
	}
}

func TestProcNewNetDev(t *testing.T) {
	p, err := FS("fixtures").NewProc(26231)
	if err != nil {
		t.Fatal(err)
	}

	nd, err := p.NewNetDev()
	if err != nil {
		t.Fatal(err)
	}

	lines := map[string]NetDevLine{
		"lo":   {Name: "lo"},
		"eth0": {Name: "eth0", RxBytes: 438, RxPackets: 5, TxBytes: 648, TxPackets: 8},
	}

	if want, have := len(lines), len(nd); want != have {
		t.Errorf("want %d parsed net/dev lines, have %d", want, have)
	}
	for _, line := range nd {
		if want, have := lines[line.Name], line; want != have {
			t.Errorf("%s: want %v, have %v", line.Name, want, have)
		}
	}
}
