/* SPDX-License-Identifier: GPL-3.0 */
/*
 * Copyright (C) 2017-2021  Stijn Tintel <stijn@linux-ipv6.be>
 */

#include <arpa/inet.h>
#include <libipset/session.h>
#include <libipset/types.h>
#include <string.h>

#ifdef WITH_LIBIPSET_V6_COMPAT
#include <libipset/ui.h>
#else
#include <libipset/ipset.h>
#endif

#include "log.h"

static int exit_error(int e, struct ipset_session *sess)
{
#ifdef WITH_LIBIPSET_V6_COMPAT
    pr_err("ipset: %s\n", ipset_session_error(sess));
#else
    ipset_session_report(sess, ipset_session_report_type(sess), "ipset: %s\n");
#endif
    ipset_session_fini(sess);

    return e;
}

static int ip_valid(char *ipaddr)
{
    unsigned int family = 0;
    unsigned int ret = 0;
    struct sockaddr_in sa;

    family = strchr(ipaddr, '.') ? AF_INET : AF_INET6;

    ret = inet_pton(family, ipaddr, &(sa.sin_addr));
    return ret != 0;
}

int ipset_do(int c, char *set, char *elem)
{
    const struct ipset_type *type = NULL;
    enum ipset_cmd cmd = c;
    int ret = 0;
    struct ipset_session *sess;

    if (!ip_valid(elem)) {
        pr_err("ipset: %s is not a valid IP address", elem);
        return 1;
    }

    ipset_load_types();

#ifdef WITH_LIBIPSET_V6_COMPAT
    sess = ipset_session_init(printf);
#else
    sess = ipset_session_init(NULL, NULL);
#endif
    if (sess == NULL) {
        pr_err("ipset: failed to initialize session\n");
        return 1;
    }

    if (cmd == IPSET_CMD_ADD) {
#ifdef WITH_LIBIPSET_V6_COMPAT
        ret = ipset_envopt_parse(sess, IPSET_ENV_EXIST, NULL);
        if (ret < 0) {
            return exit_error(1, sess);
        }
#else
        ipset_envopt_set(sess, IPSET_ENV_EXIST);
#endif
    }

    ret = ipset_parse_setname(sess, IPSET_SETNAME, set);
    if (ret < 0) {
        return exit_error(1, sess);
    }

    type = ipset_type_get(sess, cmd);
    if (type == NULL) {
        return exit_error(1, sess);
    }

    ret = ipset_parse_elem(sess, type->last_elem_optional, elem);
    if (ret < 0) {
        return exit_error(1, sess);
    }

    ret = ipset_cmd(sess, cmd, 0);
    if (ret < 0) {
        return exit_error(1, sess);
    }

    ipset_session_fini(sess);

    return 0;
}

int ipset_add(char *set, char *elem)
{
    int ret = ipset_do(IPSET_CMD_ADD, set, elem);
    if (ret == 0) {
        pr_info("ipset: added %s to %s\n", elem, set);
    }

    return ret;
}

int ipset_del(char *set, char *elem)
{
    int ret = ipset_do(IPSET_CMD_DEL, set, elem);
    if (ret == 0) {
        pr_info("ipset: deleted %s from %s\n", elem, set);
    }

    return ret;
}
