"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.discoverRemoteSerialPorts = void 0;
const typeguards_1 = require("alcalzone-shared/typeguards");
const mdns_server_1 = __importDefault(require("mdns-server"));
const domain = "_zwave._tcp.local";
function discoverRemoteSerialPorts(timeout = 1000) {
    const mdns = (0, mdns_server_1.default)({
        reuseAddr: true,
        loopback: false,
        noInit: true,
    });
    let timer;
    return new Promise((resolve, reject) => {
        mdns.on("response", (resp) => {
            const matches = resp.answers
                .filter((n) => n.type === "PTR" &&
                n.name === domain &&
                typeof n.data === "string")
                .map(({ data }) => {
                return {
                    txt: resp.answers.find((n) => n.type === "TXT" && n.name === data),
                    srv: resp.answers.find((n) => n.type === "SRV" && n.name === data),
                };
            })
                .filter(({ srv }) => !!srv &&
                (0, typeguards_1.isObject)(srv.data) &&
                typeof srv.data.target === "string" &&
                typeof srv.data.port === "number")
                .map(({ txt, srv }) => {
                const info = {};
                if (!!txt && (0, typeguards_1.isArray)(txt.data)) {
                    const strings = txt.data
                        .filter((d) => Buffer.isBuffer(d))
                        .map((d) => d.toString())
                        .filter((d) => d.includes("="));
                    for (const string of strings) {
                        const [key, value] = string.split("=", 2);
                        info[key] = value;
                    }
                }
                const addr = srv.data;
                const port = `tcp://${addr.target}:${addr.port}`;
                return {
                    port,
                    info,
                };
            });
            if (matches.length) {
                clearTimeout(timer);
                resolve(matches);
            }
        });
        mdns.on("ready", () => {
            mdns.query([
                { name: domain, type: "PTR" },
                { name: domain, type: "SRV" },
                { name: domain, type: "TXT" },
            ]);
        });
        mdns.on("error", reject);
        mdns.initServer();
        if (typeof timeout === "number" && timeout > 0) {
            timer = setTimeout(() => {
                mdns.destroy();
                resolve(undefined);
            }, timeout);
        }
    });
}
exports.discoverRemoteSerialPorts = discoverRemoteSerialPorts;
//# sourceMappingURL=mDNSDiscovery.js.map