"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cc_1 = require("@zwave-js/cc");
const safe_1 = require("@zwave-js/cc/safe");
const core_1 = require("@zwave-js/core");
const arrays_1 = require("alcalzone-shared/arrays");
const VirtualEndpoint_1 = require("./VirtualEndpoint");
function groupNodesBySecurityClass(nodes) {
    const ret = new Map();
    for (const node of nodes) {
        const secClass = node.getHighestSecurityClass();
        if (secClass === core_1.SecurityClass.Temporary || secClass == undefined) {
            continue;
        }
        if (!ret.has(secClass)) {
            ret.set(secClass, []);
        }
        ret.get(secClass).push(node);
    }
    return ret;
}
class VirtualNode extends VirtualEndpoint_1.VirtualEndpoint {
    constructor(id, driver, 
    /** The references to the physical node this virtual node abstracts */
    physicalNodes) {
        // Define this node's intrinsic endpoint as the root device (0)
        super(undefined, driver, 0);
        this.id = id;
        /** Cache for this node's endpoint instances */
        this._endpointInstances = new Map();
        // Set the reference to this and the physical nodes
        super.setNode(this);
        this.physicalNodes = [...physicalNodes].filter((n) => 
        // And avoid including the controller node in the support checks
        n.id !== driver.controller.ownNodeId &&
            // And omit nodes using Security S0 which does not support broadcast / multicast
            n.getHighestSecurityClass() !== core_1.SecurityClass.S0_Legacy);
        this.nodesBySecurityClass = groupNodesBySecurityClass(this.physicalNodes);
        // If broadcasting is attempted with mixed security classes, automatically fall back to multicast
        if (this.hasMixedSecurityClasses)
            this.id = undefined;
    }
    get hasMixedSecurityClasses() {
        return this.nodesBySecurityClass.size > 1;
    }
    /**
     * Updates a value for a given property of a given CommandClass.
     * This will communicate with the physical node(s) this virtual node represents!
     */
    async setValue(valueId, value, options) {
        // Ensure we're dealing with a valid value ID, with no extra properties
        valueId = (0, core_1.normalizeValueID)(valueId);
        // Try to retrieve the corresponding CC API
        try {
            // Access the CC API by name
            const endpointInstance = this.getEndpoint(valueId.endpoint || 0);
            if (!endpointInstance) {
                return {
                    status: safe_1.SetValueStatus.EndpointNotFound,
                    message: `Endpoint ${valueId.endpoint} does not exist on virtual node ${this.id ?? "??"}`,
                };
            }
            let api = endpointInstance.commandClasses[valueId.commandClass];
            // Check if the setValue method is implemented
            if (!api.setValue) {
                return {
                    status: safe_1.SetValueStatus.NotImplemented,
                    message: `The ${(0, core_1.getCCName)(valueId.commandClass)} CC does not support setting values`,
                };
            }
            const valueIdProps = {
                property: valueId.property,
                propertyKey: valueId.propertyKey,
            };
            const hooks = api.setValueHooks?.(valueIdProps, value, options);
            if (hooks?.supervisionDelayedUpdates) {
                api = api.withOptions({
                    requestStatusUpdates: true,
                    onUpdate: async (update) => {
                        try {
                            if (update.status === core_1.SupervisionStatus.Success) {
                                await hooks.supervisionOnSuccess();
                            }
                            else if (update.status === core_1.SupervisionStatus.Fail) {
                                await hooks.supervisionOnFailure();
                            }
                        }
                        catch {
                            // TODO: Log error?
                        }
                    },
                });
            }
            // And call it
            const result = await api.setValue.call(api, valueIdProps, value, options);
            // api.setValue could technically return a SupervisionResult
            // but supervision isn't used for multicast / broadcast
            // FIXME: It just may for S2 multicast
            if (api.isSetValueOptimistic(valueId)) {
                // If the call did not throw, assume that the call was successful and remember the new value
                // for each node that was affected by this command
                const affectedNodes = endpointInstance.node.physicalNodes.filter((node) => node
                    .getEndpoint(endpointInstance.index)
                    ?.supportsCC(valueId.commandClass));
                for (const node of affectedNodes) {
                    node.valueDB.setValue(valueId, value);
                }
            }
            // Depending on the settings of the SET_VALUE implementation, we may have to
            // optimistically update a different value and/or verify the changes
            if (hooks) {
                const supervisedAndSuccessful = (0, core_1.isSupervisionResult)(result) &&
                    result.status === core_1.SupervisionStatus.Success;
                const shouldUpdateOptimistically = api.isSetValueOptimistic(valueId) &&
                    // For successful supervised commands, we know that an optimistic update is ok
                    (supervisedAndSuccessful ||
                        // For unsupervised commands that did not fail, we let the applciation decide whether
                        // to update related value optimistically
                        (!this.driver.options.disableOptimisticValueUpdate &&
                            result == undefined));
                // The actual API implementation handles additional optimistic updates
                if (shouldUpdateOptimistically) {
                    hooks.optimisticallyUpdateRelatedValues?.(supervisedAndSuccessful);
                }
                // Verify the current value after a delay, unless...
                // ...the command was supervised and successful
                // ...and the CC API decides not to verify anyways
                if (!(0, core_1.supervisedCommandSucceeded)(result) ||
                    hooks.forceVerifyChanges?.()) {
                    // Let the CC API implementation handle the verification.
                    // It may still decide not to do it.
                    await hooks.verifyChanges?.();
                }
            }
            return (0, safe_1.supervisionResultToSetValueResult)(result);
        }
        catch (e) {
            // Define which errors during setValue are expected and won't throw an error
            if ((0, core_1.isZWaveError)(e)) {
                let result;
                switch (e.code) {
                    // This CC or API is not implemented
                    case core_1.ZWaveErrorCodes.CC_NotImplemented:
                    case core_1.ZWaveErrorCodes.CC_NoAPI:
                        result = {
                            status: safe_1.SetValueStatus.NotImplemented,
                            message: e.message,
                        };
                        break;
                    // A user tried to set an invalid value
                    case core_1.ZWaveErrorCodes.Argument_Invalid:
                        result = {
                            status: safe_1.SetValueStatus.InvalidValue,
                            message: e.message,
                        };
                        break;
                }
                if (result)
                    return result;
            }
            throw e;
        }
    }
    /**
     * Returns a list of all value IDs and their metadata that can be used to
     * control the physical node(s) this virtual node represents.
     */
    getDefinedValueIDs() {
        // In order to compare value ids, we need them to be strings
        const ret = new Map();
        for (const pNode of this.physicalNodes) {
            // // Nodes using Security S0 cannot be used for broadcast
            // if (pNode.getHighestSecurityClass() === SecurityClass.S0_Legacy) {
            // 	continue;
            // }
            // Take only the actuator values
            const valueIDs = pNode
                .getDefinedValueIDs()
                .filter((v) => core_1.actuatorCCs.includes(v.commandClass));
            // And add them to the returned array if they aren't included yet or if the version is higher
            for (const valueId of valueIDs) {
                const mapKey = (0, core_1.valueIdToString)(valueId);
                const ccVersion = pNode.getCCVersion(valueId.commandClass);
                const metadata = pNode.getValueMetadata(valueId);
                // Don't expose read-only values for virtual nodes, they won't ever have any value
                if (!metadata.writeable)
                    continue;
                const needsUpdate = !ret.has(mapKey) || ret.get(mapKey).ccVersion < ccVersion;
                if (needsUpdate) {
                    ret.set(mapKey, {
                        ...valueId,
                        ccVersion,
                        metadata: {
                            ...metadata,
                            // Metadata of virtual nodes is only writable
                            readable: false,
                        },
                    });
                }
            }
        }
        // Basic CC is not exposed, but virtual nodes need it to control multiple different devices together
        const exposedEndpoints = (0, arrays_1.distinct)([...ret.values()]
            .map((v) => v.endpoint)
            .filter((e) => e !== undefined));
        for (const endpoint of exposedEndpoints) {
            // TODO: This should be defined in the Basic CC file
            const valueId = {
                ...cc_1.BasicCCValues.targetValue.endpoint(endpoint),
                commandClassName: "Basic",
                propertyName: "Target value",
            };
            const ccVersion = 1;
            const metadata = {
                ...cc_1.BasicCCValues.targetValue.meta,
                readable: false,
            };
            ret.set((0, core_1.valueIdToString)(valueId), {
                ...valueId,
                ccVersion,
                metadata,
            });
        }
        return [...ret.values()];
    }
    getEndpoint(index) {
        if (index < 0)
            throw new core_1.ZWaveError("The endpoint index must be positive!", core_1.ZWaveErrorCodes.Argument_Invalid);
        // Zero is the root endpoint - i.e. this node
        if (index === 0)
            return this;
        // Check if the Multi Channel CC interviews for all nodes are completed,
        // because we don't have all the information before that
        if (!this.isMultiChannelInterviewComplete) {
            this.driver.driverLog.print(`Virtual node ${this.id ?? "??"}, Endpoint ${index}: Trying to access endpoint instance before the Multi Channel interview of all nodes was completed!`, "error");
            return undefined;
        }
        // Check if the requested endpoint exists on any physical node
        if (index > this.getEndpointCount())
            return undefined;
        // Create an endpoint instance if it does not exist
        if (!this._endpointInstances.has(index)) {
            this._endpointInstances.set(index, new VirtualEndpoint_1.VirtualEndpoint(this, this.driver, index));
        }
        return this._endpointInstances.get(index);
    }
    getEndpointOrThrow(index) {
        const ret = this.getEndpoint(index);
        if (!ret) {
            throw new core_1.ZWaveError(`Endpoint ${index} does not exist on virtual node ${this.id ?? "??"}`, core_1.ZWaveErrorCodes.Controller_EndpointNotFound);
        }
        return ret;
    }
    /** Returns the current endpoint count of this virtual node (the maximum in the list of physical nodes) */
    getEndpointCount() {
        let ret = 0;
        for (const node of this.physicalNodes) {
            const count = node.getEndpointCount();
            ret = Math.max(ret, count);
        }
        return ret;
    }
    get isMultiChannelInterviewComplete() {
        for (const node of this.physicalNodes) {
            if (!node["isMultiChannelInterviewComplete"])
                return false;
        }
        return true;
    }
}
exports.VirtualNode = VirtualNode;
//# sourceMappingURL=VirtualNode.js.map