/// \file ThisMfcApp.cpp
/// \brief App object of MFC version of Password Safe
//-----------------------------------------------------------------------------

#include "PasswordSafe.h"
#include "PwsPlatform.h"

#if defined(POCKET_PC)
  #include "pocketpc/PocketPC.h"
#else
  #include <io.h>
  #include <fcntl.h>
  #include <sys/stat.h>
  #include <errno.h>
#endif

#include "ThisMfcApp.h"
#include "Util.h"
#include "DboxMain.h"

#if defined(POCKET_PC)
  #include "pocketpc/resource.h"
#else
  #include "resource.h"
#endif

#include "CryptKeyEntry.h"


//-----------------------------------------------------------------------------
/*
  The one and only ThisMfcApp object.  In the MFC world, creating
  this global object is what starts the application.
*/

ThisMfcApp app;

//-----------------------------------------------------------------------------

BEGIN_MESSAGE_MAP(ThisMfcApp, CWinApp)
//   ON_COMMAND(ID_HELP, CWinApp::OnHelp)
   ON_COMMAND(ID_HELP, OnHelp)
END_MESSAGE_MAP()


ThisMfcApp::ThisMfcApp() :
#if defined(POCKET_PC)
	m_bUseAccelerator( false ),
#else
	m_bUseAccelerator( true ),
#endif
	m_pMRU( NULL )
{
   srand((unsigned)time(NULL));

// {kjp} Temporary until I'm sure that PwsPlatform.h configures the endianness properly
#if defined(POCKET_PC)
	// Double check that *_ENDIAN has been correctly set!
#if defined(LITTLE_ENDIAN)
	unsigned char	buf[4]	= { 1, 0, 0, 0 };
	unsigned int	ii		= 1;
#define ENDIANNESS	_T("little endian")
#define ENDIANNESS2	_T("big endian")
#elif defined(BIG_ENDIAN)
	unsigned char	buf[4]	= { 0, 0, 0, 1 };
	unsigned int	ii		= 1;
#define ENDIANNESS	_T("big endian")
#define ENDIANNESS2	_T("little endian")
#endif
	if (*(unsigned int*)buf != ii)
	{
		AfxMessageBox(_T("Password Safe has been compiled as ") ENDIANNESS
			_T(" but CPU is really ") ENDIANNESS2 _T("\n")
			_T("You may not be able to open files or saved files may be incompatible with other platforms."));
	}
#endif
}


ThisMfcApp::~ThisMfcApp()
{
	if ( m_pMRU )
	{
		m_pMRU->WriteList();
		delete m_pMRU;
	}

   /*
     apparently, with vc7, there's a CWinApp::HtmlHelp - I'd like
     to see the docs, someday.  In the meantime, force with :: syntax
   */

#ifndef POCKET_PC
   ::HtmlHelp(NULL, NULL, HH_CLOSE_ALL, 0);
#endif
}

#if !defined(POCKET_PC)
static void Usage()
{
  AfxMessageBox(_T("Usage: PasswordSafe [password database]\n")
		_T("or PasswordSafe [-e|-d] filename"));
}

// tests if file exists, returns empty string if so, displays error message if not
static BOOL CheckFile(const CString &fn)
{
  DWORD status = ::GetFileAttributes(fn);
  CString ErrMess;

  if (status == -1) {
    ErrMess = _T("Could not access file: ");
    ErrMess += fn;
  } else if (status & FILE_ATTRIBUTE_DIRECTORY) {
    ErrMess = fn;
    ErrMess += _T(" is a directory");
  }

  if (ErrMess.IsEmpty()) {
    return TRUE;
  } else {
    AfxMessageBox(ErrMess);
    return FALSE;
  }
}

//Complain if the file has not opened correctly

#if defined(POCKET_PC)
static void
ErrorMessages(const CString &fn, FILE *fp)
{
}
#else
static void
ErrorMessages(const CString &fn, int fp)
{
  if (fp==-1)
    {
      CString text;
      text = "A fatal error occured: ";

      if (errno==EACCES)
	text += "Given path is a directory or file is read-only";
      else if (errno==EEXIST)
	text += "The filename already exists.";
      else if (errno==EINVAL)
	text += "Invalid oflag or shflag argument.";
      else if (errno==EMFILE)
	text += "No more file handles available.";
      else if (errno==ENOENT)
	text += "File or path not found.";
      text += "\nProgram will terminate.";

      CString title = "Password Safe - " + fn;
      AfxGetMainWnd()->MessageBox(text, title, MB_ICONEXCLAMATION|MB_OK);
    }
}
#endif

static BOOL EncryptFile(const CString &fn, const CMyString &passwd)
{
  unsigned int len;
  unsigned char* buf;

#if defined(POCKET_PC)
#if defined(UNICODE)
  FILE *in = _wfopen(fn, _T("rb"));
#else
  FILE *in = fopen(fn, _T("rb"));
#endif
  if (in != NULL) {
#else
  int in = _open(fn,
		 _O_BINARY|_O_RDONLY|_O_SEQUENTIAL,
		 S_IREAD | _S_IWRITE);
  if (in != -1) {
#endif
    len = _filelength(in);
    buf = new unsigned char[len];

#if defined(POCKET_PC)
    fread(buf, 1, len, in);
    fclose(in);
#else
    _read(in, buf, len);

    _close(in);
#endif
  } else {
#ifndef POCKET_PC
    ErrorMessages(fn, in);
#endif
    return FALSE;
  }

  CString out_fn = fn;
  out_fn += CIPHERTEXT_SUFFIX;

#if defined(POCKET_PC)
#if defined(UNICODE)
  FILE *out = _wfopen(out_fn, _T("wb"));
#else
  FILE *out = fopen(out_fn, _T("wb"));
#endif
  if (out != NULL) {
#else
  int out = _open(out_fn,
		  _O_BINARY|_O_WRONLY|_O_SEQUENTIAL|_O_TRUNC|_O_CREAT,
		  _S_IREAD | _S_IWRITE);
  if (out != -1) {
#endif
#ifdef KEEP_FILE_MODE_BWD_COMPAT
#if defined(POCKET_PC)
	fwrite( &len, 1, sizeof(len), out);
#else
    _write(out, &len, sizeof(len)); // XXX portability issue!
#endif
#else
    for (int i=0; i < 8; i++)
      app.m_randstuff[i] = newrand();

    // miserable bug - have to fix this way to avoid breaking existing files
    app.m_randstuff[8] = app.m_randstuff[9] = '\0';
    GenRandhash(passwd,
		app.m_randstuff,
		app.m_randhash);
#if defined(POCKET_PC)
   fwrite(app.m_randstuff, 1,  8, out);
   fwrite(app.m_randhash,  1, 20, out);
#else
   _write(out, app.m_randstuff, 8);
   _write(out, app.m_randhash, 20);
#endif
#endif // KEEP_FILE_MODE_BWD_COMPAT
		
    unsigned char thesalt[SaltLength];
    for (int x=0;x<SaltLength;x++)
      thesalt[x] = newrand();
#if defined(POCKET_PC)
    fwrite(thesalt, 1, SaltLength, out);
#else
    _write(out, thesalt, SaltLength);
#endif
		
    unsigned char ipthing[8];
    for (x=0;x<8;x++)
      ipthing[x] = newrand();
#if defined(POCKET_PC)
    fwrite(ipthing, 1, 8, out);
#else
    _write(out, ipthing, 8);
#endif

    LPCSTR pwd = LPCSTR(passwd);
    _writecbc(out, buf, len,
	      (unsigned char *)pwd, passwd.GetLength(),
	      thesalt, SaltLength,
	      ipthing);
		
#if defined(POCKET_PC)
    fclose(out);
#else
    _close(out);
#endif

  } else {
#ifndef POCKET_PC
    ErrorMessages(out_fn, out);
#endif
    delete [] buf;
    return FALSE;
  }
  delete[] buf;
  return TRUE;
}

static BOOL DecryptFile(const CString &fn, const CMyString &passwd)
{
  unsigned int len;
  unsigned char* buf;

#if defined(POCKET_PC)
#if defined(UNICODE)
  FILE *in = _wfopen(fn, _T("rb"));
#else
  FILE *in = fopen(fn, _T("rb"));
#endif
  if (in != NULL) {
#else
  int in = _open(fn,
		 _O_BINARY|_O_RDONLY|_O_SEQUENTIAL,
		 S_IREAD | _S_IWRITE);
  if (in != -1) {
#endif
      unsigned char salt[SaltLength];
      unsigned char ipthing[8];

#ifdef KEEP_FILE_MODE_BWD_COMPAT
#if defoned(POCKET_PC)
      fread(&len, 1, sizeof(len), in); // XXX portability issue
#else
      _read(in, &len, sizeof(len)); // XXX portability issue
#endif
#else
#if defined(POCKET_PC)
      fread(app.m_randstuff, 1, 8, in);
#else
      _read(in, app.m_randstuff, 8);
#endif
      app.m_randstuff[8] = app.m_randstuff[9] = '\0'; // ugly bug workaround
#if defined(POCKET_PC)
      fread(app.m_randhash, 1, 20, in);
#else
      _read(in, app.m_randhash, 20);
#endif

      unsigned char temphash[20]; // HashSize
      GenRandhash(passwd,
		  app.m_randstuff,
		  temphash);
      if (0 != memcmp((char*)app.m_randhash,
		      (char*)temphash,
		      20)) // HashSize
	{
#if defined(POCKET_PC)
	  fclose(in);
#else
	  _close(in);
#endif
	  AfxMessageBox(_T("Incorrect password"));
	  return FALSE;
	}
#endif // KEEP_FILE_MODE_BWD_COMPAT
      buf = NULL; // allocated by _readcbc - see there for apologia

#if defined(POCKET_PC)
      fread(salt,    1, SaltLength, in);
      fread(ipthing, 1, 8,          in);
#else
      _read(in, salt, SaltLength);
      _read(in, ipthing, 8);
#endif
      LPCSTR pwd = LPCSTR(passwd);
      if (_readcbc(in, buf, len,
		   (unsigned char *)pwd, passwd.GetLength(),
		   salt, SaltLength, ipthing) == 0) {
	delete[] buf; // if not yet allocated, delete[] NULL, which is OK
	return FALSE;
      }
		
#if defined(POCKET_PC)
      fclose(in);
#else
      _close(in);
#endif
    } else {
#if !defined(POCKET_PC)
      ErrorMessages(fn, in);
#endif
      return FALSE;
    }

  int suffix_len = strlen(CIPHERTEXT_SUFFIX);
  int filepath_len = fn.GetLength();

  CString out_fn = fn;
  out_fn = out_fn.Left(filepath_len - suffix_len);

#if defined(POCKET_PC)
#if defined(UNICODE)
  FILE *out = _wfopen(out_fn, _T("wb"));
#else
  FILE *out = fopen(out_fn, _T("wb"));
#endif
  if (out != NULL) {
    fwrite(buf, 1, len, out);
    fclose(out);
#else
  int out = _open(out_fn,
		  _O_BINARY|_O_WRONLY|_O_SEQUENTIAL|_O_TRUNC|_O_CREAT,
		  _S_IREAD | _S_IWRITE);
  if (out != -1) {
    _write(out, buf, len);
    _close(out);
#endif
    } else
#ifndef POCKET_PC
      ErrorMessages(out_fn, out);
#endif

  delete[] buf; // allocated by _readcbc
  return TRUE;
}
#endif

BOOL
ThisMfcApp::InitInstance()
{


   /*
    * It's always best to start at the beginning.  [Glinda, Witch of the North]
    */

   /*
     this pulls 'Counterpane Systems' out of the string table, verifies
     that it exists (actually, only verifies that *some* IDS_COMPANY
     string exists -- it could be 'Microsoft' for all we know), and then
     instructs the app to use the registry instead of .ini files.  The
     path ends up being

     HKEY_CURRENT_USER\Software\(companyname)\(appname)\(sectionname)\(valuename)

     Assuming the open-source version of this is going to become less
     Counterpane-centric, I expect this may change, but if it does, an
     automagic migration ought to happen. -- {jpr}
   */

   CMyString companyname;
   VERIFY(companyname.LoadString(IDS_COMPANY) != 0);
   SetRegistryKey(companyname);

   int	nMRUItems = GetProfileInt(_T(PWS_REG_OPTIONS), _T("maxmruitems"), 4);
   m_pMRU = new CRecentFileList( 0, _T("MRU"), _T("Safe%d"), nMRUItems );;
   m_pMRU->ReadList();

   DboxMain dbox(NULL);

   /*
    * Command line processing:
    * Historically, it appears that if a filename was passed as a commadline argument,
    * the application would prompt the user for the password, and the encrypt or decrypt
    * the named file, based on the file's suffix. Ugh.
    *
    * What I'll do is as follows:
    * If a file is given in the command line, it is used as the database, overriding the
    * registry value. This will allow the user to have several databases, say, one for work
    * and one for personal use, and to set up a different shortcut for each.
    *
    * I think I'll keep the old functionality, but activate it with a "-e" or "-d" flag. (ronys)
	* {kjp} ... and I've removed all of it from the Pocket PC build.
    */

#if !defined(POCKET_PC)
   if (m_lpCmdLine[0] != '\0') {
     CString args = m_lpCmdLine;

     if (args[0] != _T('-')) {

       if (CheckFile(args)) {
	 dbox.SetCurFile(args);
       } else {
	 return FALSE;
       }
     } else { // here if first char of arg is '-'
       // first, let's check that there's a second arg
       CString fn = args.Right(args.GetLength()-2);
       fn.TrimLeft();
       if (fn.IsEmpty() || !CheckFile(fn)) {
	 Usage();
	 return FALSE;
       }

       CMyString passkey;
       if (args[1] == 'e' || args[1] == 'E' || args[1] == 'd' || args[1] == 'D') {
	 // get password from user if valid flag given. If note, default below will
	 // pop usage message
	 CCryptKeyEntry dlg(NULL);
	 int nResponse = dlg.DoModal();

	 if (nResponse==IDOK) {
	     passkey = dlg.m_cryptkey1;
	 } else {
	   return FALSE;
	 }
       }
       BOOL status;
       switch (args[1]) {
       case 'e': case 'E': // do encrpytion
	 status = EncryptFile(fn, passkey);
	 if (!status) {
	   AfxMessageBox(_T("Encryption failed"));
	 }
	 break;
       case 'd': case 'D': // do decryption
	 status = DecryptFile(fn, passkey);
	 if (!status) {
	   // nothing to do - DecryptFile displays its own error messages
	 }
	 break;
       default:
	 Usage();
	 return FALSE;
       } // switch
       return FALSE;
     } // else
   } // m_lpCmdLine[0] != '\0';
#endif
       
   /*
    * normal startup
    */

   /*
     Here's where PWS currently does DboxMain, which in turn will do
     the initial PasskeyEntry (the one that looks like a splash screen).
     This makes things very hard to control.
     The app object (here) should instead do the initial PasskeyEntry,
     and, if successful, move on to DboxMain.  I think. {jpr}
    */
   m_maindlg = &dbox;
   m_pMainWnd = m_maindlg;

   // Set up an Accelerator table
#if !defined(POCKET_PC)
   m_ghAccelTable = LoadAccelerators(AfxGetInstanceHandle(),
                                     MAKEINTRESOURCE(IDR_ACCS));
#endif
   //Run dialog
   (void) dbox.DoModal();

   /*
     note that we don't particularly care what the response was
   */


   // Since the dialog has been closed, return FALSE so that we exit the
   //  application, rather than start the application's message pump.

   return FALSE;
}


#if !defined(POCKET_PC)
//Copied from Knowledge Base article Q100770
//But not for WinCE {kjp}
BOOL
ThisMfcApp::ProcessMessageFilter(int code,
                                 LPMSG lpMsg)
{
   if (code < 0)
      CWinApp::ProcessMessageFilter(code, lpMsg);
	
   if (m_bUseAccelerator &&
	   m_maindlg != NULL
       && m_ghAccelTable != NULL)
   {
      if (::TranslateAccelerator(m_maindlg->m_hWnd, m_ghAccelTable, lpMsg))
         return TRUE;
   }
   return CWinApp::ProcessMessageFilter(code, lpMsg);
}
#endif


void
ThisMfcApp::OnHelp()
{
#if defined(POCKET_PC)
	CreateProcess( _T("PegHelp.exe"), _T("pws_ce_help.html#mainhelp"), NULL, NULL, FALSE, 0, NULL, NULL, NULL, NULL );
#else
	::HtmlHelp(NULL,
              "pwsafe.chm::/html/pws_intro.htm",
              HH_DISPLAY_TOPIC, 0);
#endif
}
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
