/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2023 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.LinkedList;
import java.util.Vector;
import java.util.zip.DataFormatException;
import java.util.zip.Inflater;

public class Server {

    static final int ACTION_RESPONSE  = 0x10000000;
    static final int ACTION_EXCEPTION = 0x1ffffff1;

    static Server instance = null;
    static Action local = null;

    static class Request {
        Blob request;
        Blob response;
        boolean expectingResponse;
    }

    private Vector<Container> containers = new Vector<Container>(128,128);
    private LinkedList<Request> requestsPending = new LinkedList<Request>();

    private Server() {
    }

    public static synchronized Server getInstance() {
        if (null == instance) {
            instance = new Server();
        }
        return instance;
    }

    public static void reset() {
        if (instance != null) {
            instance = null;
        }
    }

    public static synchronized Container getContainer(int id) {
        if (id==-1) return Container.local;
        Server server = getInstance();
        return server.containers.elementAt(id);
    }

    public static synchronized int createContainer() {
        Server server = getInstance();
        int id = server.containers.size();
        server.containers.add(new Container(id));
        return id;
    }

    public static void destroyContainer(int id) {
        Server server = getInstance();
        Container container = null;
        synchronized(Server.class) {
            if (id==-1) {
                container = Container.local;
                Container.local = null;
            } else {
                container = server.containers.elementAt(id);
                server.containers.set(id,null);
            }
        }
        if (null!=container) {
            container.close();
        }
    }

    public static void setLocal(Action action) {
        local = action;
        Container.local = new Container(-1);
    }

    static Blob waitRequest(long timeout) {
        Server server = getInstance();

        synchronized (server.requestsPending) {
            if (server.requestsPending.isEmpty() == false) {
                return server.removeRequest();
            }
            try {
                server.requestsPending.wait(timeout);
            } catch (InterruptedException e) {
                Thread.interrupted();
            }
            if (server.requestsPending.isEmpty() == false) {
                return server.removeRequest();
            }
        }

        return null;
    }

    private Blob removeRequest() {
        Request req = requestsPending.peekFirst();
        Blob request = req.request;
        if (req.expectingResponse == false) {
            requestsPending.removeFirst();
        }
        return request;
    }

    static Blob pollRequest() {
        Server server = getInstance();

        synchronized (server.requestsPending) {
            if (server.requestsPending.isEmpty() == false) {
                return server.removeRequest();
            }
        }

        return null;
    }

    static void submitRespone(Blob response) {
        Server server = getInstance();

        Request req = null;

        synchronized (server.requestsPending) {
            if (server.requestsPending.isEmpty() == false) {
                req = server.requestsPending.removeFirst();
            }
        }

        if (null == req)
            throw new Error("No active request");

        synchronized (req) {
            req.response = response;
            req.notify();
        }

        return;
    }

    public static Blob callRequest(Blob request) {
        if (local==null) {
            return Server.getInstance().callRequest0(request);
        } else {
            Blob r;
            try {
                synchronized(local) {
                    r = local.action(request);
                }
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            return r;
        }
    }

    private Blob callRequest0(Blob request) {
        Request req = new Request();
        req.request = request;
        req.response = null;
        req.expectingResponse = true;
        Blob response;

        synchronized (req) {
            synchronized (requestsPending) {
                requestsPending.addLast(req);
                requestsPending.notifyAll();
            }
            try {
                req.wait();
            } catch (InterruptedException e) {
                req.response = null;
            }
            response = req.response;
            req = null;
        }
        return response;
    }

    void putActionResponse(Blob request) {
        Request req = new Request();
        req.request = request;
        req.response = null;
        req.expectingResponse = false;

        synchronized (requestsPending) {
            requestsPending.addLast(req);
            requestsPending.notifyAll();
        }
    }

    static Blob exceptionResponse(Throwable e,int actionId) {
        Blob response = new Blob(Server.ACTION_EXCEPTION);
        response.args = new int[1];
        response.args[0] = actionId;
        response.strings = new String[2];
        response.strings[0] = e.getClass().getName();
        response.strings[1] = e.getMessage();
        ByteArrayOutputStream ba = new ByteArrayOutputStream(1024);
        PrintStream ps = new PrintStream(ba);
        e.printStackTrace(ps);
        ps.flush(); ps.close();
        ba.write(0);
        response.data = ba.toByteArray();
        return response;
    }

    public static void execAction(final int containerId,final String actionName,final Blob action,final Action handler) {
        if (actionName.equals("-")) {
            new Thread(new Runnable(){
                public void run() {
                    Blob response;
                    try {
                        response = serverAction(action);
                        response.id = action.id + Server.ACTION_RESPONSE;
                    } catch (Throwable e) {
                        response = exceptionResponse(e,action.id);
                    }
                    try {
                        handler.action(response);
                    } catch (Exception e) {
                    }
                }}).start();
        } else {
            getContainer(containerId).execAction(actionName,action,handler);
        }
    }

    public static Blob requestFindClass(String name) {
        Blob request = new Blob(1);
        request.strings = new String[1];
        request.strings[0] = name;
        return callRequest(request);
    }

    public static byte[] deflate(byte[] inData, int method, int size) {
        if (method != 8)
            return null;

        Inflater inflater = new Inflater(true);

        if ( (inData[0]==(byte)0x1f) &&
             (inData[1]==(byte)0x8b) &&
             (inData[2]==(byte)0x08) &&
             (inData[3]==(byte)0x00) )
        {
            size = 0;
            size |= (inData[inData.length-1]&0xff); size <<= 8;
            size |= (inData[inData.length-2]&0xff); size <<= 8;
            size |= (inData[inData.length-3]&0xff); size <<= 8;
            size |= (inData[inData.length-4]&0xff);
            inflater.setInput(inData, 10, inData.length-18);
        } else {
            inflater.setInput(inData);
        }

        byte[] outData = new byte[size];
        try {
            if (size != inflater.inflate(outData)) {
                outData = null;
            }
        } catch (DataFormatException e) {
            outData = null;
        }
        inflater.end();
        return outData;
    }

    public static synchronized void close() {

        if (null!=Container.local) {
            Container.local.close();
            Container.local = null;
        }

        if (null!=instance) {
            for (int i=0;i<instance.containers.size();i++) {
                destroyContainer(i);
            }
        }
        Container.cleanupTempDir();
    }

    private static Blob serverAction(Blob b) throws Exception {
        switch(b.id) {
        case 1:
            destroyContainer(b.args[0]);
            return new Blob();
        }
        throw new IllegalArgumentException();
    }

}
