/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2023 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
package blues.libbluray;

import org.videolan.PlaylistInfo;
import org.videolan.StreamInfo;
import org.videolan.TIClip;
import org.videolan.TIMark;
import org.videolan.TitleInfo;
import org.videolan.bdjo.AppCache;
import org.videolan.bdjo.AppEntry;
import org.videolan.bdjo.Bdjo;
import org.videolan.bdjo.PlayListTable;
import org.videolan.bdjo.TerminalInfo;
import org.videolan.Libbluray;

import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.IOException;
import java.security.AccessController;
import java.util.HashMap;
import java.util.Properties;

import org.videolan.BDJSecurityManager;
import org.videolan.BDJUtil;
import org.videolan.BDJXletContext;

import blues.Blob;
import blues.Cache;
import blues.Log;
import blues.Container;
import hdmv.REG;
import hdmv.VM.HDMV_EVENT;
import hdmv.VM.HDMV_EVENT_PLAY_PL;
import hdmv.VM.HDMV_EVENT_PLAY_STOP;
import hdmv.VM.HDMV_EVENT_TITLE;
import impl.java.io.BluesFileSystem;
import impl.org.bluray.system.RegisterAccessImpl;
import jail.org.bluray.system.RegisterAccess;
import net.java.bd.tools.bdjo.AppCacheEntry;
import net.java.bd.tools.bdjo.AppInfo;
import net.java.bd.tools.bdjo.AppName;
import net.java.bd.tools.bdjo.ApplicationDescriptor;
import net.java.bd.tools.bdjo.BDJO;
import net.java.bd.tools.bdjo.BDJOReader;
import net.java.bd.tools.bdjo.TableOfAccessiblePlayLists;
import net.java.bd.tools.id.Id;
import net.java.bd.tools.id.IdReader;
import net.java.bd.tools.index.BDJIndexObject;
import net.java.bd.tools.index.HDMVIndexObject;
import net.java.bd.tools.index.Index;
import net.java.bd.tools.index.IndexObject;
import net.java.bd.tools.index.IndexObject.IndexObjectType;
import net.java.bd.tools.index.Title;
import net.java.bd.tools.movieobject.MovieObjectFile;
import net.java.bd.tools.movieobject.MovieObjectReader;
import net.java.bd.tools.movieobject.MovieObjects;
import net.java.bd.tools.playlist.MPLSObject;
import net.java.bd.tools.playlist.Mark;
import net.java.bd.tools.playlist.PlayItem;
import net.java.bd.tools.playlist.STNTableEntry;
import net.java.bd.tools.playlist.StreamAttribute;

public class LB {

    static HashMap<Integer,PlaylistInfo> playlists = new HashMap<Integer,PlaylistInfo>();

    public static void init(Blob b) throws Exception{
        Properties props = glue.java.lang.System.properties();

        String discRoot = props.getProperty("bluray.vfs.root");
        String persistentRoot = props.getProperty("dvb.persistent.root");
        String budaRoot = props.getProperty("bluray.bindingunit.root");

        AccessController.getContext().checkPermission(new java.security.AllPermission());

        props.setProperty("awt.toolkit", "java.awt.BDToolkit");
        props.setProperty("java.awt.graphicsenv", "java.awt.BDGraphicsEnvironment");

        String id = readBdmvId();
        if (id==null) {
            id = "123456";
        }

        mobj = readMovieObject();

        Sync.setMode(Sync.MODE_ASYNC); //Sync.MODE_INSYNC
        if (b.args.length>=1) {
            Sync.setTimeout(b.args[0]);
        }

        REG.bd_registers_init();

        if (b.args.length>=3) {
        	switch(b.args[2]&0x0f) {
        	case 0: break;
        	case 1: REG.psr_init_3D(true,false); break;
        	case 2: REG.psr_init_3D(false,false); break;
        	case 3: REG.psr_init_UHD(false); break;
        	default: break;
        	}
        }

        if ((b.args.length>=2) && (b.args[1]!=0)) {
            Libbluray.writePSR(RegisterAccess.PSR_PLAYER_PROFILE, b.args[1]);
        }

        Libbluray.init(1, id, discRoot, persistentRoot, budaRoot);

        BDJSecurityManager sm = (BDJSecurityManager) glue.java.lang.System.getSecurityManager();
        sm.setCacheRoot("/mnt/vfs");

        Container.my().atExit(new Closeable(){
            @Override
            public void close() {
                LB.shutdown();
            }
        });
    }

    public static void shutdown() {
        Libbluray.shutdown();
        playlists.clear();
    }

    public static String getDMSCCMountPoint(int jarFileId) {
        StringBuilder sb = new StringBuilder(64);
        sb.append("/mnt/vfs/BDMV/JAR/");
        sb.append(BDJUtil.makeFiveDigitStr(jarFileId));
        sb.append(".jar");
        return sb.toString();
    }

    public static String getDMSCCMountPoint(String path) {
        if (path.length()<5) {
            Log.log(Log.LOG_ERROR, "getDMSCCMountPoint: path ",path," invalid");
            return null;
        }

        String strId,strPath;
        if (path.length()==5) {
            strId = path;
            strPath = null;
        } else {
            strId = path.substring(0, 5);
            if (path.charAt(5)!='/') {
                Log.log(Log.LOG_ERROR, "getDMSCCMountPoint: path ",path," invalid");
                strPath = null;
            } else {
                strPath=path.substring(5);
            }
        }

        String vfsPath = getDMSCCMountPoint(Integer.parseInt(strId));
        if (strPath!=null) {
            vfsPath += strPath;
        }

        return vfsPath;
    }


    public static String getDMSCCRoot() {
        return BDJXletContext.getCurrentXletHome();
    }

    public static byte[] getAacsDataN(long np, int type) {
        Log.unimplemented();
        return null;
    }

    private static byte[] readBdmvFile(String path) {
        String rootPath = glue.java.lang.System.properties().getProperty("bluray.vfs.root");

        byte[] data = (byte[])Cache.get(Cache.ID_DiscFile, path);

        if (null==data) {
            try {
                data = BluesFileSystem.readJailFile(rootPath + path,128*1024*1024);
            } catch (IOException e) {
                Log.log(Log.LOG_ERROR,"Exception while reading ",path," : ",e.toString());
                return null;
            }
            if (data.length<16384) {
                Cache.put(Cache.ID_DiscFile, path, data);
            }
        }

        return data;
    }

    public static TitleInfo[] getTitleInfosN(long np) {
        byte[] indexData = readBdmvFile("/BDMV/index.bdmv");
        if (indexData == null) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to read index.bdmv");
            return null;
        }

        DataInputStream din = new DataInputStream(new ByteArrayInputStream(indexData));
        Index index = new Index();

        try {
            index.readObject(din);
            din.close();
        } catch (IOException e) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to parse index.bdmv");
            return null;
        }

        Title[] titles = index.getIndexes().getTitles().getTitle();

        TitleInfo[] titleInfos = new TitleInfo[titles.length+2];
        titleInfos[0] = createTitleInfo(index.getIndexes().getTopMenu().getTopMenuObject(),0);
        for (int i=0;i<titles.length;i++) {
            titleInfos[i+1]=createTitleInfo(titles[i].getIndexObject(),i+1);
        }
        titleInfos[titles.length+1] = createTitleInfo(index.getIndexes().getFirstPlayback().getFirstPlaybackObject(),0xffff);

        return titleInfos;
    }

    private static TitleInfo createTitleInfo(IndexObject indexObject,int title) {
        int objType,playbackType=0,idRef=0;

        objType = indexObject.getObjectType().ordinal();
        if (indexObject.getObjectType()==IndexObjectType.V_01) {
            HDMVIndexObject indexHDMV = (HDMVIndexObject)indexObject;
            playbackType = indexHDMV.getPlaybackType().ordinal();
            idRef = indexHDMV.getHDMVName();
        } else if (indexObject.getObjectType()==IndexObjectType.V_10) {
            BDJIndexObject indexBDJ = (BDJIndexObject)indexObject;
            playbackType = indexBDJ.getPlaybackType().ordinal();
            idRef = Integer.parseInt(indexBDJ.getBDJOName(), 10);
        }
        return new TitleInfo(title,objType,playbackType,idRef);
    }

    public static Bdjo getBdjoN(long np, String name) {
        byte[] bdjoData = readBdmvFile("/BDMV/BDJO/" + name);
        if (bdjoData == null) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to read " + name);
            return null;
        }

        BDJO bdjo;

        try {
            bdjo = BDJOReader.readBDJO(new ByteArrayInputStream(bdjoData));
        } catch (IOException e) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to parse "+name);
            return null;
        }

        net.java.bd.tools.bdjo.TerminalInfo ti = bdjo.getTerminalInfo();
        TerminalInfo terminalInfo = new TerminalInfo(
                ti.getDefaultFontFile(),
                ti.getInitialHaviConfig().getId(),
                ti.isMenuCallMask(),
                ti.isTitleSearchMask());

        AppCacheEntry[] appCacheEntries = bdjo.getAppCacheInfo().getEntries();
        AppCache[] appCaches = new AppCache[appCacheEntries.length];
        for (int i=0;i<appCacheEntries.length;i++) {
            appCaches[i] = new AppCache(
                    appCacheEntries[i].getType(),
                    appCacheEntries[i].getName(),
                    appCacheEntries[i].getLanguage());
        }

        TableOfAccessiblePlayLists tpl = bdjo.getTableOfAccessiblePlayLists();
        PlayListTable accessiblePlaylists = new PlayListTable(
                tpl.isAccessToAllFlag(),
                tpl.isAutostartFirstPlayListFlag(),
                tpl.getPlayListFileNames());

        AppInfo[] appInfo = bdjo.getApplicationManagementTable().getApplications();
        AppEntry[] appTable = new AppEntry[appInfo.length];
        for (int i=0;i<appInfo.length;i++) {
            AppInfo entry = appInfo[i];
            ApplicationDescriptor desc = entry.getApplicationDescriptor();

            net.java.bd.tools.bdjo.AppProfile[] srcProfiles = desc.getProfiles();
            org.videolan.bdjo.AppProfile[] dstProfiles = new org.videolan.bdjo.AppProfile[srcProfiles.length];

            for (int j=0;j<srcProfiles.length;j++) {
                dstProfiles[j] = new org.videolan.bdjo.AppProfile(
                        (short)srcProfiles[j].getProfile(),
                        (byte)srcProfiles[j].getMajorVersion(),
                        (byte)srcProfiles[j].getMinorVersion(),
                        (byte)srcProfiles[j].getMicroVersion());
            }

            AppName[] srcNames = desc.getNames();
            String[][] dstNames = null;
            if (null!=srcNames) {
	            dstNames = new String[srcNames.length][2];

	            for (int j=0;j<srcNames.length;j++) {
	                dstNames[j][0] = srcNames[j].getLanguage();
	                dstNames[j][1] = srcNames[j].getName();
	            }
            }

            appTable[i] = new AppEntry(
                    entry.getControlCode(),
                    entry.getType(),
                    entry.getOrganizationId(),
                    entry.getApplicationId(),
                    dstProfiles,
                    desc.getPriority(),
                    desc.getBinding().ordinal(),
                    desc.getVisibility().ordinal(),
                    dstNames,
                    desc.getIconLocator(),
                    desc.getIconFlags(),
                    desc.getBaseDirectory(),
                    desc.getClasspathExtension(),
                    desc.getInitialClassName(),desc.getParameters());
        }

        Bdjo r = new Bdjo(
                terminalInfo,
                appCaches,
                accessiblePlaylists,
                appTable,
                bdjo.getKeyInterestTable(),
                bdjo.getFileAccessInfo());

        return r;
    }

    private static String readBdmvId() {
        byte[] idData = readBdmvFile("/CERTIFICATE/id.bdmv");
        if (null==idData) return null;

        DataInputStream din = new DataInputStream(new ByteArrayInputStream(idData));
        Id id = null;

        try {
            id = IdReader.readId(din);
            din.close();
        } catch (IOException e) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to parse id.bdmv");
            return null;
        }

        byte[] idBytes = id.getDiscId();

        StringBuilder sb = new StringBuilder(idBytes.length*2);

        for (int i=0;i<idBytes.length;i++) {
            sb.append(Integer.toHexString((idBytes[i]|0x100)&0x1ff).substring(1));
        }

        return sb.toString();
    }

    public static MovieObjects readMovieObject() {
        byte[] moData = readBdmvFile("/BDMV/MovieObject.bdmv");
        if (null==moData) return null;

        DataInputStream din = new DataInputStream(new ByteArrayInputStream(moData));
        MovieObjectFile mof = null;

        try {
            mof = MovieObjectReader.readBinary(din);
            din.close();
        } catch (IOException e) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to parse MovieObject.bdmv");
            return null;
        }

        return mof.getMovieObjects();
    }

    public static synchronized PlaylistInfo getPlaylistInfoN(long np, int playlist) {
        PlaylistInfo pi = playlists.get(new Integer(playlist));
        if (pi==null) {
            pi = readPlaylistInfo(playlist);
            if (pi!=null) {
                playlists.put(new Integer(playlist), pi);
            }
        }
        return pi;
    }

    public static PlaylistInfo readPlaylistInfo(int playlist) {
        byte[] mplsData = readBdmvFile("/BDMV/PLAYLIST/" + BDJUtil.makeFiveDigitStr(playlist)+".mpls");
        if (mplsData == null) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to read " + BDJUtil.makeFiveDigitStr(playlist) + ".mpls");
            return null;
        }

        MPLSObject mpls = new MPLSObject();

        try {
            mpls.readObject(new DataInputStream(new ByteArrayInputStream(mplsData)));
        } catch (IOException e) {
            Log.log(Log.LOG_LIBBLURAY, "Unable to parse "+BDJUtil.makeFiveDigitStr(playlist)+".mpls");
            return null;
        }

        Mark[] srcMarks = mpls.getPlayListMark().getMarks();
        TIMark[] dstMarks = new TIMark[srcMarks.length];
        long duration = 0;
        for (int i=0;i<srcMarks.length;i++) {
            if ((i+1)!=srcMarks.length) {
                duration = (srcMarks[i+1].getMarkTimeStamp() - srcMarks[i].getMarkTimeStamp())+1200;
            }
            dstMarks[i] = new TIMark(srcMarks[i].getId(),
                    srcMarks[i].getType(),
                    ((long)srcMarks[i].getMarkTimeStamp())*2,
                    ((long)duration)*2,
                    0,12345);
        }

        PlayItem[] srcClips = mpls.getPlayList().getPlayItems();
        TIClip[] dstClips = new TIClip[srcClips.length];
        for (int i=0;i<srcClips.length;i++) {

            StreamInfo[] videoStreams = convertStreamInfos(srcClips[i].getStnTable().getPrimaryVideoStreams());
            StreamInfo[] audioStreams = convertStreamInfos(srcClips[i].getStnTable().getPrimaryAudioStreams());
            StreamInfo[] pgStreams = convertStreamInfos(srcClips[i].getStnTable().getPGTextSTStreams());
            StreamInfo[] igStreams = convertStreamInfos(srcClips[i].getStnTable().getIGStreams());
            StreamInfo[] secVideoStreams = convertStreamInfos(srcClips[i].getStnTable().getSecondaryVideoStreams());
            StreamInfo[] secAudioStreams = convertStreamInfos(srcClips[i].getStnTable().getSecondaryAudioStreams());

            dstClips[i] = new TIClip(i,videoStreams,audioStreams,pgStreams,igStreams,secVideoStreams,secAudioStreams);
        }

        PlaylistInfo pi = new PlaylistInfo(playlist,56789000,1,dstMarks,dstClips);

        return pi;
    }

    private static StreamInfo[] convertStreamInfos(STNTableEntry[] src) {
        StreamInfo[] dst;
        if (src==null) {
            dst = new StreamInfo[0];
        } else {
            dst = new StreamInfo[src.length];
            for (int n = 0; n < src.length; n++) {
                dst[n] = convertStreamInfo(src[n].getStreamAttribute());
            }
        }
        return dst;
    }

    private static StreamInfo convertStreamInfo(StreamAttribute sa) {
        byte coding_type = (sa.getStreamCodingType()==null)?0:sa.getStreamCodingType().byteValue();
        byte format = (sa.getVideoFormat()==null)?0:sa.getVideoFormat().getEncoding();
        byte rate = (sa.getFrameRate()==null)?0:sa.getFrameRate().getEncoding();
        char char_code = (char) ( (sa.getCharactorCode()==null)?0:sa.getCharactorCode().byteValue());
        String lang = sa.getAudioLanguageCode();
        byte aspect = 3; // 16:9
        byte subpath_id = 0;

        return new StreamInfo(coding_type,format,rate,char_code,lang,aspect,subpath_id);
    }

    private static MovieObjects mobj = null;
    public static MovieObjects getMovieObjects() {
    	return mobj;
    }

    private static boolean stop_title_hdmv_flag = false;

    public static void stopTitleHDMV() {
        stop_title_hdmv_flag = true;
    }

    static int play_title_hdmv(int title) {
        TitleInfo ti = Libbluray.getTitleInfo(title);

        if (null == ti) {
            return -1;
        }

        if (ti.isBdj()) {
            Log.fatal("Not an HDMV title");
        }

        if ((title == 0xffff) || (title <= Libbluray.numTitles())) {
            RegisterAccessImpl.getInstance().setPSR(RegisterAccess.PSR_TITLE_NR, title);
        }

        int mobjId = ti.getHdmvOID();

        hdmv.VM vm = new hdmv.VM();
        vm.hdmv_vm_init();

        if (0 != vm.hdmv_vm_select_object(mobjId)) {
            return -1;
        }

        while (vm.hdmv_vm_running()) {
            if (stop_title_hdmv_flag) {
                return -1;
            }

            /* run VM */
            if (vm.hdmv_vm_run() < 0) {
                return -1;
            }

            HDMV_EVENT evt;
            while (null != (evt = vm.hdmv_vm_get_event())) {
                if (evt instanceof HDMV_EVENT_PLAY_STOP) {
                    Log.log(Log.LOG_LIBBLURAY, "HDMV_EVENT_PLAY_STOP");
                    // Libbluray.stopTitle(false);
                }
                if (evt instanceof HDMV_EVENT_TITLE) {
                    int newtitle = ((HDMV_EVENT_TITLE) evt).title;
                    Log.log(Log.LOG_LIBBLURAY, "HDMV_EVENT_TITLE(", Integer.toString(title), ")");
                    Libbluray.startTitle(newtitle);
                    return 0;
                }
                if (evt instanceof HDMV_EVENT_PLAY_PL) {
                    int playlist = ((HDMV_EVENT_PLAY_PL) evt).playlist;
                    int playitem = ((HDMV_EVENT_PLAY_PL) evt).playitem;
                    int playmark = ((HDMV_EVENT_PLAY_PL) evt).playmark;
                    Log.log(Log.LOG_LIBBLURAY, "HDMV_EVENT_PLAY_PL(", Integer.toString(playlist), ",",
                            Integer.toString(playitem), ",", Integer.toString(playmark), ")");
                    if (false == Libbluray.selectPlaylist(playlist, playitem, playmark, -1)) {
                        return -1;
                    }
                    vm.hdmv_vm_resume();
                }
            }
        }

        return 0;
    }

    public static long seekN(long np, int playitem, int playmark, long time) {
        Log.unimplemented();
        return 0;
    }

    public static int selectPlaylistN(final long np, final int playlist, final int playitem, final int playmark, final long time) {

        long[] pile = new long[] { np, playlist, playitem, playmark, time };
        control.Gate.callStaticBefore(pile, "blues.libbluray.LB", "selectPlaylistN", "(JIIIJ)I", 0);

        Log.unimplemented();

        int r = 1;
        r = control.Gate.callStaticAfter(r,"blues.libbluray.LB", "selectPlaylistN", "(JIIIJ)I", 0);

        Libbluray.processEventImpl(Libbluray.BDJ_EVENT_PLAYLIST,playlist);
        Libbluray.processEventImpl(Libbluray.BDJ_EVENT_PLAYITEM,playitem<0?0:playitem);
        Libbluray.processEventImpl(Libbluray.BDJ_EVENT_MARK,playmark<0?0:playmark);
        try {
            Thread.sleep(10);
        } catch (InterruptedException e) {
        }
        Libbluray.processEventImpl(Libbluray.BDJ_EVENT_END_OF_PLAYLIST,playlist);

        return r;
    }

    public static int selectTitleN(long np, int title) {

        int r;
        long[] pile = new long[] { np, title };
        control.Gate.callStaticBefore(pile, "blues.libbluray.LB", "selectTitleN", "(JI)I", 0);
        title = (int)pile[1];

        r = (play_title_hdmv(title)<0)?0:1;

        r = control.Gate.callStaticAfter(r,"blues.libbluray.LB", "selectTitleN", "(JI)I", 0);

        return r;
    }

    public static int selectAngleN(long np, int angle) {
        Log.unimplemented();
        return 1;
    }

    public static int soundEffectN(long np, int id) {
        Log.unimplemented();
        return 0;
    }

    public static long getUOMaskN(long np) {
        Log.unimplemented();
        return 0;
    }

    public static void setUOMaskN(long np, boolean menuCallMask, boolean titleSearchMask) {
        Log.unimplemented();
    }

    public static void setKeyInterestN(long np, int mask) {
        Log.unimplemented();
    }

    public static long tellTimeN(long np) {
        Log.unimplemented();
        return 0;
    }

    public static int selectRateN(long np, float rate, int reason) {
        Log.unimplemented();
        return 1;
    }

    public static void updateGraphicN(long np, int width, int height, int[] rgbArray,
                                              int x0, int y0, int x1, int y1) {
        //Log.unimplemented();
    }

    static final String[] readableProps = {
            "java.version", "java.vendor", "java.vendor.url","java.class.version",
            "os.name", "os.version", "os.arch", "file.separator", "path.separator", "line.separator",
            "java.specification.version", "java.specification.vendor", "java.specification.name",
            "java.vm.specification.version", "java.vm.specification.vendor", "java.vm.specification.name",
            "java.vm.version", "java.vm.vendor", "java.vm.name" };

    public static boolean isPropReadable(String prop) {
        for (String s : readableProps) {
            if (prop.equals(s)) return true;
        }
        return false;
    }

}
