/*
    BLUES - BD-Java emulation server

    Copyright (C) 2007-2023 GuinpinSoft inc <blues@makemkv.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/
/*
 * @(#)SecurityManager.java	1.138 06/10/10
 *
 * Copyright  1990-2008 Sun Microsystems, Inc. All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License version
 * 2 only, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License version 2 for more details (a copy is
 * included at /legal/license.txt).
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this work; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 or visit www.sun.com if you need additional
 * information or have any questions.
 *
 */

package impl.java.lang;

import impl.sun.security.util.SecurityConstants;
import java.lang.reflect.Member;
import java.security.AccessController;
import java.security.PrivilegedAction;

import blues.ClientClassLoader;

public
class SecurityManager {

    /**
     * Throws a <code>SecurityException</code> if the
     * calling thread is not allowed to access the AWT event queue, or
     * the AWT package is not available.
     * <p>
     * If you override this method, then you should make a call to
     * <code>super.checkAwtEventQueueAccess</code>
     * at the point the overridden method would normally throw an
     * exception.
     *
     * @since   JDK1.1
     * @exception  SecurityException  if the calling thread does not have
     *             permission to access the AWT event queue.
     * @see        #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkAwtEventQueueAccess(java.lang.SecurityManager athis) {
      	athis.checkPermission(SecurityConstants.CHECK_AWT_EVENTQUEUE_PERMISSION);
    }

    /**
     * Throws a <code>SecurityException</code> if the
     * calling thread is not allowed to access members.
     * <p>
     * The default policy is to allow access to PUBLIC members, as well
     * as access to classes that have the same class loader as the caller.
     * In all other cases, this method calls <code>checkPermission</code>
     * with the <code>RuntimePermission("accessDeclaredMembers")
     * </code> permission.
     * <p>
     * If this method is overridden, then a call to
     * <code>super.checkMemberAccess</code> cannot be made,
     * as the default implementation of <code>checkMemberAccess</code>
     * relies on the code being checked being at a stack depth of
     * 4.
     *
     * @param clazz the class that reflection is to be performed on.
     *
     * @param which type of access, PUBLIC or DECLARED.
     *
     * @exception  SecurityException if the caller does not have
     *             permission to access members.
     * @exception  NullPointerException if the <code>clazz</code> argument is
     *             <code>null</code>.
     * @see java.lang.reflect.Member
     * @since JDK1.1
     * @see        #checkPermission(java.security.Permission) checkPermission
     */
    public static void checkMemberAccess(java.lang.SecurityManager athis,Class clazz, int which) {
	if (clazz == null) {
	    throw new NullPointerException("class can't be null");
	}
	if (which != Member.PUBLIC) {
	    Class stack[] = ClientClassLoader.getStack();
	    /*
	     * stack depth of 4 should be the caller of one of the
	     * methods in java.lang.Class that invoke checkMember
	     * access. The stack should look like:
	     *
	     * someCaller                         [3]
	     * java.lang.Class.someReflectionAPI  [2]
	     * java.lang.Class.checkMemeberAccess [1]
	     * SecurityManager.checkMemeberAccess [0]
	     *
	     */
	    if ((stack.length<(4+ClientClassLoader.STACK_OFFSET)) ||
	    		(getClassLoader0(stack[3+ClientClassLoader.STACK_OFFSET]) != getClassLoader0(clazz))) {
	    	athis.checkPermission(SecurityConstants.CHECK_MEMBER_ACCESS_PERMISSION);
	    }
	}
    }

    private static ClassLoader getClassLoader0(final Class clazz) {
        ClassLoader r = (ClassLoader)AccessController.doPrivileged(new PrivilegedAction() {
            @Override
            public Object run() {
                return clazz.getClassLoader();
            }
        });
        return r;
    }

}
