REM
REM  This file is part of the ANSiMat project.
REM  Copyright (C) Mateusz Viste 2010
REM
REM  ANSiMat homepage: gopher://gopher.viste-family.net/1/projects/ansimat/
REM
REM  Available routines:
REM   FlcFileOpen() - Opens the FLC file and writes the header, as well as the color palette
REM   FlcFileAddBlackFrame() - Adds a plain black frame into the file
REM   FlcFileAddFrame() - adds a new frame into the file (from BigScreenBuffer())
REM   FlcFileAddEmptyFrame() - Adds an empty frame into the file (means "same frame than the previous one")
REM   FlcFileClose() - Closes the FLC file, updating fields "Number of frames" and "Filesize"
REM
REM References:
REM   http://www.compuphase.com/flic.htm
REM   http://www.fileformat.info/format/fli/egff.htm
REM

DIM SHARED AS INTEGER FlcFileHandler             ' All these variables have to be shared!
DIM SHARED AS INTEGER FlcFileFramesCounter = -1  ' Starts at -1 to not count the final ring frame


FUNCTION FlcFileOpen(BYREF FlcFile AS STRING, BYREF xsize AS INTEGER, BYREF ysize AS INTEGER) AS BYTE
  DIM AS INTEGER x
  DIM AS BYTE Result = 0
  REM Creates a file, and writes the FLC header, as well as the palette
  FlcFileHandler = FREEFILE
  OPEN FlcFile FOR BINARY AS FlcFileHandler
  PUT #FlcFileHandler,, CHR(0, 0, 0, 0)          ' filesize
  PUT #FlcFileHandler,, CHR(&h12, &hAF)          ' FLC signature
  PUT #FlcFileHandler,, CHR(0, 0)                ' Number of frames
  PUT #FlcFileHandler,, IntToDoubleByte(xsize)   ' Width
  PUT #FlcFileHandler,, IntToDoubleByte(ysize)   ' Height
  PUT #FlcFileHandler,, CHR(8,0)                 ' BPP color depth
  PUT #FlcFileHandler,, CHR(3,0)                 ' FLC update flag
  PUT #FlcFileHandler,, IntToDWORD(1000\FlcFPS)  ' Delay between frames (in milliseconds, 200 = 5 FPS)
  PUT #FlcFileHandler,, CHR(0,0)                 ' Reserved "for future use"
  PUT #FlcFileHandler,, IntToDWORD(0)            ' Creation date, in seconds since 01.01.1970
  PUT #FlcFileHandler,, "AMat"                   ' Creator Serial Number ("FLIB" is okay here)
  PUT #FlcFileHandler,, IntToDWORD(0)            ' Last updated date (in seconds since 01.01.1970)
  PUT #FlcFileHandler,, "AMat"                   ' Updater Serial Number ("FLIB" is okay here)
  PUT #FlcFileHandler,, IntToDoubleByte(1)       ' x-axis of display aspect ratio
  PUT #FlcFileHandler,, IntToDoubleByte(1)       ' y-axis of display aspect ratio
  PUT #FlcFileHandler,, STRING(38, 0)            ' Reserved "for future use" (38 bytes)
  PUT #FlcFileHandler,, IntToDWORD(128)          ' Offset of the first frame
  PUT #FlcFileHandler,, IntToDWORD(928)          ' Offset of the "second" frame (if the first is a black one + palette)
  PUT #FlcFileHandler,, STRING(40, 0)            ' Reserved "for future use" (40 bytes)
  RETURN Result
END FUNCTION


SUB FlcFileAddBlackFrame()
  DIM AS INTEGER x, y
  FlcFileFramesCounter += 1
  IF FlcFileFramesCounter = 0 THEN
      REM Frame chunk
      PUT #FlcFileHandler,, IntToDWORD(16+778+6)' Chunk's size
      PUT #FlcFileHandler,, CHR(&hFA, &hF1)     ' Chunk's type (F1FA = frame)
      PUT #FlcFileHandler,, CHR(2, 0)           ' Number of subchunks (1)
      PUT #FlcFileHandler,, STRING(8, 0)        ' Reserved
      REM Write a full FLC palette chunk...
      PUT #FlcFileHandler,, IntToDWORD(778)   ' Chunk's size
      PUT #FlcFileHandler,, CHR(4, 0)         ' Chunk type (4=256 colors palette)
      PUT #FlcFileHandler,, CHR(1, 0)         ' Number of color elements in map (1)
      PUT #FlcFileHandler,, CHR(0)            ' SkipCount byte
      PUT #FlcFileHandler,, CHR(0)            ' Number of colors in the element (0 means 256)
      FOR x = 0 TO 255
        PUT #FlcFileHandler,, CHR((AnsiPallette(x) SHR 16) AND &b11111111) ' Write Red value
        PUT #FlcFileHandler,, CHR((AnsiPallette(x) SHR 8) AND &b11111111)  ' Write Green value
        PUT #FlcFileHandler,, CHR((AnsiPallette(x)) AND &b11111111)        ' Write Blue value
      NEXT x
    ELSE
      REM Frame chunk
      PUT #FlcFileHandler,, IntToDWORD(16+6)    ' Chunk's size
      PUT #FlcFileHandler,, CHR(&hFA, &hF1)     ' Chunk's type (F1FA = frame)
      PUT #FlcFileHandler,, CHR(1, 0)           ' Number of subchunks (1)
      PUT #FlcFileHandler,, STRING(8, 0)        ' Reserved
  END IF
  REM Write frame's header...
  PUT #FlcFileHandler,, IntToDWORD(6)   ' size of the chunk (including its header)
  PUT #FlcFileHandler,, CHR(13, 0)      ' Chunk type (13=BlackFrame)
END SUB


SUB FlcFileAddFrame(BYREF xstart AS INTEGER, BYREF ystart AS INTEGER, BYREF xsize AS INTEGER, BYREF ysize AS INTEGER)
  DIM AS INTEGER x, y
  DIM AS STRING FrameDataContentRLE = ""
  DIM AS UBYTE OldPixel, RleValue
  FlcFileFramesCounter += 1
  FOR y = ystart TO ystart + ysize - 1
    FrameDataContentRLE += CHR(0) ' Line's count byte - used in oldish FLI only (set to 0 now)
    OldPixel = BigScreenBuffer(xstart, y)
    RleValue = 1
    FOR x = xstart + 1 TO xstart + xsize - 1
      IF BigScreenBuffer(x, y) = OldPixel AND RleValue < 127 THEN
          RleValue += 1
        ELSE
          FrameDataContentRLE += CHR(RleValue, OldPixel)
          RleValue = 1
          OldPixel = BigScreenBuffer(x, y)
      END IF
    NEXT x
    FrameDataContentRLE += CHR(RleValue, OldPixel) ' Write the last pixel in buffer
  NEXT y
  REM Frame chunk
  IF FlcFileFramesCounter = 0 THEN
      PUT #FlcFileHandler,, IntToDWORD(16+778+6+LEN(FrameDataContentRLE))   ' Chunk's size
      PUT #FlcFileHandler,, CHR(&hFA, &hF1)  ' Chunk's type (F1FA = frame)
      PUT #FlcFileHandler,, CHR(2, 0)        ' Number of subchunks (2)
    ELSE
      PUT #FlcFileHandler,, IntToDWORD(16+6+LEN(FrameDataContentRLE))   ' Chunk's size
      PUT #FlcFileHandler,, CHR(&hFA, &hF1)  ' Chunk's type (F1FA = frame)
      PUT #FlcFileHandler,, CHR(1, 0)        ' Number of subchunks (1)
  END IF
  PUT #FlcFileHandler,, STRING(8, 0)         ' Reserved
  REM Write a full FLC palette chunk... (only for the first frame)
  IF FlcFileFramesCounter = 0 THEN
    PUT #FlcFileHandler,, IntToDWORD(778)   ' Chunk's size
    PUT #FlcFileHandler,, CHR(4, 0)         ' Chunk type (4=256 colors palette)
    PUT #FlcFileHandler,, CHR(1, 0)         ' Number of color elements in map (1)
    PUT #FlcFileHandler,, CHR(0)            ' SkipCount byte
    PUT #FlcFileHandler,, CHR(0)            ' Number of colors in the element (0 means 256)
    FOR x = 0 TO 255
      PUT #FlcFileHandler,, CHR((AnsiPallette(x) SHR 16) AND &b11111111) ' Write Red value
      PUT #FlcFileHandler,, CHR((AnsiPallette(x) SHR 8) AND &b11111111)  ' Write Green value
      PUT #FlcFileHandler,, CHR((AnsiPallette(x)) AND &b11111111)        ' Write Blue value
    NEXT x
  END IF
  REM Write frame's header...
  PUT #FlcFileHandler,, IntToDWORD(6 + LEN(FrameDataContentRLE)) ' size of the chunk (including its header)
  PUT #FlcFileHandler,, CHR(15, 0)                      ' Chunk type (15=ByteRun RLE compressed data)
  REM Write frame's data...
  PUT #FlcFileHandler,, FrameDataContentRLE
END SUB


SUB FlcFileAddEmptyFrame()
  FlcFileFramesCounter += 1
  PUT #FlcFileHandler,, CHR(16, 0, 0, 0)   ' Chunk's size (16 bytes for an empty frame)
  PUT #FlcFileHandler,, CHR(&hFA, &hF1)    ' Chunk's type (F1FA = frame)
  PUT #FlcFileHandler,, CHR(0, 0)          ' Number of subchunks (0 = empty frame)
  PUT #FlcFileHandler,, STRING(8, 0)       ' Reserved
END SUB


SUB FlcFileClose()
  PUT #FlcFileHandler, 1, IntToDWORD(LOF(FlcFileHandler))       ' Update filesize
  PUT #FlcFileHandler, 7, IntToDoubleByte(FlcFileFramesCounter) ' Update number of frames
  CLOSE #FlcFileHandler                                         ' Close the file
END SUB
