#include <stdio.h>
#include <pc.h>
#include <sys/nearptr.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
char *error_message = "";
char *video;

// Checks the error_message string and prints it + returns 1 if the string
// isn't empty.
char check_error(void) {
  if (error_message != "") {
    printf("%s\n", error_message);
    return 1;
  }
}

// Allocates memory using malloc and returns a pointer to the memory
// or zero on failure.
char *allocate_mem(size_t amount) {
  char *pointer;
  if ((pointer = (char *)malloc(amount)) == NULL) {
    error_message = "Not enough memory.";
    return 0;
  }
  return pointer;
}

// Waits for the vertical retrace of the monitor.
void wait_vertical_retrace(void) {
  while ((inportb(0x03DA) & 0x08) != 0);
  while ((inportb(0x03DA) & 0x08) == 0);
}

// Returns a random integer between 0 and range.
long random_nr(long range) {
  return (random() / (RAND_MAX / range));
}

// Gives the time as a random seed.
void randomize(void) {
  srandom(time(0));
}

// Disables DOS memory protection for direct access to the video memory
// and sets the pointer video to the video memory.
char init_gfx(void) {
  if ((__djgpp_nearptr_enable()) == NULL) {
    error_message = "Error disabeling DOS memory protection.";
    return 1;
  }
  video = (char *)0xA0000 + __djgpp_conventional_base;
}

// Fills 64000 bytes at mem with value.
void clear_screen(char *mem, unsigned long value) {
  asm volatile ("
    cld
    rep
    stosl"
    : // no output
    : "D" (mem), "a" (value), "c" (16000)
    : "%edi", "%ecx"
  );
}

// Copies 64000 bytes from source to dest.
void copy_screen(char *dest, char *source) {
  asm volatile ("
    cld
    rep
    movsl"
    : // no output
    : "S" (source), "D" (dest), "c" (16000)
    : "%esi", "%edi", "%ecx"
  );
}

// Sets a palette.
void set_palette(char *pal) {
  int c;
  outportb(0x3C8, 0);
  for (c = 0 ; c < 256 ; c++) {
    outportb(0x3C9, pal[c * 3]);
    outportb(0x3C9, pal[c * 3 + 1]);
    outportb(0x3C9, pal[c * 3 + 2]);
  }
}

// Sets a color.
void set_color(unsigned char number, unsigned char red, unsigned char green, unsigned char blue) {
  outportb(0x3C8, number);
  outportb(0x3C9, red);
  outportb(0x3C9, green);
  outportb(0x3C9, blue);
}

// Sets the mcga video mode.
void set_video_mcga(void) {
  asm ("
    pushw %ax
    movw $0x13, %ax
    int $0x10
    popw %ax
  ");
}

// Sets 80 x 25 text video mode.
void set_video_text(void) {
  asm ("
    pushw %ax
    movw $0x03, %ax
    int $0x10
    popw %ax
  ");
  video = (char *)0xB8000 + __djgpp_conventional_base;
}

// Puts a pixel in mem.
// Clipping.
void put_pixel(short x, short y, unsigned char color, char *mem) {
  if ((x > -1) && (x < 320) && (y > -1) && (y < 200)) {
    mem[(y << 8) + (y << 6) + x] = color;
  }
}

void fade(char *pal1, char *pal2, short speed) {
  short c, c2;
  char curpal[768];
  memcpy(curpal, pal1, 768);
  for (c2 = 0; c2 < speed; c2++) {
    for (c = 0; c < 768; c++) {
      curpal[c] = pal1[c] + ((float)(pal2[c] - pal1[c]) / speed) * c2;
    }
    wait_vertical_retrace();
    set_palette(curpal);
  }
}

// Loads a 256 color palette.
char load_palette(char *filename, char *dest) {
  FILE *infile;
  if ((fopen(filename, "rb")) == NULL) {
    error_message = "Error opening file: ";
    strcat(error_message, filename);
    return 1;
  }
  fread(dest, 1, 768, infile);
  fclose(infile);
}

// Loads a pcx file of any size.
char load_pcx(char *filename, char *mem, char *pal) {
  FILE *infile;
  unsigned short	w;
  unsigned long offs = 0;
  unsigned long length;
  unsigned char x, y;
  unsigned short width, height;

  if ((infile = fopen(filename, "rb")) == NULL) {
    error_message = "Error opening file: ";
    strcat(error_message, filename);
    return 1;
  }

  fseek(infile, 4 + 4, SEEK_SET);
  fread(&width, 2, 1, infile);
  fread(&height, 2, 1, infile);
  length = (height + 1) * (width + 1);
  fseek(infile, 128, SEEK_SET);

  while (offs < length) {
    x = fgetc(infile);
    if (x > 192) {
      y = fgetc(infile);
      for (w = 0 ; w < (x - 192) ; w++) {
        mem[offs] = y;
        offs++;
      }
    }
    if (x <= 192) {
      mem[offs] = x;
      offs++;
    }
  }

  while (x != 12) x = fgetc(infile);
  for (w = 0 ; w < 768 ; w++) pal[w] = fgetc(infile) / 4;
  fclose(infile);
}

char load_font(char *filename, char nr) {
  FILE *infile;
  if ((infile = fopen(filename, "rb")) == NULL) return 1;
  font[nr].width = fgetc(infile);
  font[nr].height = fgetc(infile);
  font[nr].offset = (char *)malloc(256 * font[nr].height * font[nr].width);
  fread(font[nr].offset, 1, 256 * font[nr].height * font[nr].width, infile);
  fclose(infile);
}

void screen_dump(char *source, char *pal, char *filename) {
  FILE *infile, *outfile;
  unsigned short c;
  unsigned char b;

  outfile = fopen(filename, "wb");
  infile = fopen("data13.dat", "rb");
  for (c = 0; c < 128; c++) {
    fputc(fgetc(infile), outfile);
  }
  fclose(infile);


  for (c = 0; c < 64000; c++) {
    b = source[c];
    if (b > 192) {
      fputc(1 + 128, outfile);
    }
    fputc(b, outfile);
  }

  for (c = 0; c < 768; c++) {
    fputc(pal[c] * 4, outfile);
  }

  fclose(outfile);
}

void put_pob(short xpos, short ypos, short n, char *data, char *dest) {
  unsigned short offs, poboffs, x, y, w, c;
  unsigned char color;
  short xleft, xwidth, yupper, yheight;

  data += pobdata[n].offset;

  // Partially inside screen?
  if (((xpos - pobdata[n].xhotspot + pobdata[n].xsize) > -1) && ((xpos - pobdata[n].xhotspot) < 320) &&
     ((ypos - pobdata[n].yhotspot + pobdata[n].ysize) > -1) && ((ypos - pobdata[n].yhotspot) < 200)) {
    // Entierly inside screen?
    if (((xpos - pobdata[n].xhotspot) > -1) && ((xpos - pobdata[n].xhotspot + pobdata[n].xsize) < 320) &&
       ((ypos - pobdata[n].yhotspot) > -1) && ((ypos - pobdata[n].yhotspot + pobdata[n].ysize) < 200)) {

      poboffs = 0;
      offs = (ypos - pobdata[n].yhotspot) * 320 + xpos - pobdata[n].xhotspot;
      for (y = 0; y < pobdata[n].ysize; y++) {
        for (x = 0; x < pobdata[n].xsize; x++) {
          color = data[poboffs++];
          if (color != 0) dest[offs] = color;
          offs++;
        }
        offs += 320 - pobdata[n].xsize;
      }
    }

    // Clipping is needed
    else {

      xleft = pobdata[n].xhotspot - xpos;
      if (xleft < 0) xleft = 0;
      xwidth = 320 - xpos + pobdata[n].xhotspot;
      if (xwidth > pobdata[n].xsize) xwidth = pobdata[n].xsize;

      yupper = pobdata[n].yhotspot - ypos;
      if (yupper < 0) yupper = 0;
      yheight = 200 - ypos + pobdata[n].yhotspot;
      if (yheight > pobdata[n].ysize) yheight = pobdata[n].ysize;

      poboffs = yupper * pobdata[n].xsize + xleft;
      offs = (ypos - pobdata[n].yhotspot + yupper) * 320 + xpos - pobdata[n].xhotspot + xleft;

      for (y = yupper; y < yheight; y++) {
        for (x = xleft; x < xwidth; x++) {
          color = data[poboffs++];
          if (color != 0) dest[offs] = color;
          offs++;
        }
        poboffs += xleft + (pobdata[n].xsize - xwidth);
        offs += 320 - (xwidth - xleft);
      }
    }
  }
}

void load_pobs(char *filename, char *dest) {
	FILE *infile;
	unsigned short offs;
	short x, y, c, c2, size;

	infile = fopen(filename, "rb");
	offs = 0;

	for (c2 = 0; c2 < 38; c2++) {
		x = fgetc(infile);
		y = fgetc(infile);
		size = y * x;
		pobdata[c2].xsize = x;
		pobdata[c2].ysize = y;
		pobdata[c2].size = size;
		x = fgetc(infile);
		y = fgetc(infile);
		pobdata[c2].xhotspot = x;
		pobdata[c2].yhotspot = y;
		pobdata[c2].offset = offs;
		for (c = 0; c < size; c++) {
			dest[offs++] = fgetc(infile);
		}
	}
	fclose(infile);
}

unsigned short rnd(unsigned short max) {
  long l;

  l = ( (random() >> 16) * max) / 0x7fff;
  if (l > max - 1)
    l = max - 1;
  return l;
}