/*
 * CPP2CCMT - A C++ to C comment converter
 * Copyright (C) 2004 Thomas Weidenmueller <w3seek@reactos.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#if DEBUG
#define ASSERT(Condition)                                                      \
  do {                                                                         \
    if(!(Condition)) {                                                         \
      printf("\n\n  ASSERTION %s FAILED AT %s:%i!\n", #Condition,              \
             __FILE__, __LINE__);                                              \
      exit(1);                                                                 \
    }                                                                          \
  } while(0)
#else /* !DEBUG */
#define ASSERT(Condition)
#endif

#define READ_BUF_SIZE      0x400

#define IN_CPP_COMMENT     0x1
#define IN_C_COMMENT       0x2
#define IN_INVERTED_COMMAS 0x4
#define IN_QUOTES          0x8

#define IN_TOKEN_MASK (IN_CPP_COMMENT | IN_C_COMMENT | IN_INVERTED_COMMAS |    \
                       IN_QUOTES)

#define TI_CPP_ONLYSLASHES 0x1
#define TI_CPP_DO_SPACE    0x2

typedef struct _FILE_INFO
{
  char LastChar;
  int InToken;
  union
  {
    struct
    {
      int Flags;
      int SlashCount;
      char SpaceChar;
    } CPPComment;
  } TokenInfo;
} FILE_INFO, *PFILE_INFO;

void
ConsumeCharacter(PFILE_INFO FileInfo, char c)
{
  int ClearLastChar = 0;
  int OutputChar = 1;
  
  if ((c == '\n') || (c == '\r'))
  {
    if(FileInfo->InToken & IN_CPP_COMMENT)
    {
      FileInfo->InToken &= ~IN_CPP_COMMENT;
      ClearLastChar = 1;

      if(FileInfo->TokenInfo.CPPComment.SlashCount > 1)
      {
        if(FileInfo->TokenInfo.CPPComment.Flags & TI_CPP_DO_SPACE)
        {
          FileInfo->TokenInfo.CPPComment.SlashCount--;
        }

        /* we reached the end of the C++ comment. if the TI_CPP_ONLYSLASHES is
           set we convert all slashes inside the C++ comment to asterisks, if
           it's cleared we only convert the trailing ones (if existing) */
        while(FileInfo->TokenInfo.CPPComment.SlashCount > 2)
        {
          putchar('*');
          FileInfo->TokenInfo.CPPComment.SlashCount--;
        }
      }

      /* write one space before we close the comment */
      if(FileInfo->TokenInfo.CPPComment.Flags & TI_CPP_DO_SPACE &&
         FileInfo->LastChar != ' ')
      {
        putchar(FileInfo->TokenInfo.CPPComment.SpaceChar);
      }

      /* it's time to close the new C comment */
      putchar('*');
      putchar('/');
    }
  }
  
  if(!FileInfo->InToken)
  {

    switch(c)
    {
      case '/':
        if(FileInfo->LastChar == '/')
        {
          FileInfo->InToken |= IN_CPP_COMMENT;
          FileInfo->TokenInfo.CPPComment.Flags = TI_CPP_ONLYSLASHES;
          FileInfo->TokenInfo.CPPComment.SlashCount = 0;
          FileInfo->TokenInfo.CPPComment.SpaceChar = '\0';
          ClearLastChar = 1;

          /* simply change the second slash to an asterisk */
          c = '*';
        }
        break;

      case '*':
        if(FileInfo->LastChar == '/')
        {
          FileInfo->InToken |= IN_C_COMMENT;
          ClearLastChar = 1;
        }
        break;

      case '\"':
        FileInfo->InToken |= IN_QUOTES;
        break;

      case '\'':
        FileInfo->InToken |= IN_INVERTED_COMMAS;
        break;
    }

  }
  else /* FileInfo->InToken */
  {

    ASSERT(!(FileInfo->InToken & ~IN_TOKEN_MASK));

    if(FileInfo->InToken & IN_CPP_COMMENT)
    {
      switch(c)
      {
        case '/':
          /* if the TI_CPP_ONLYSLASHES flag is set we're counting the slashes at
             the very beginning of the C++ comment, if it's cleared we're
             counting the slashes after the last different character we found,
             so we can convert the trailing slashes to asterisks (if existing)
             at the end of the C++ comment */
          FileInfo->TokenInfo.CPPComment.SlashCount++;
          
          /* don't output this slash, it might get converted later */
          OutputChar = 0;
          break;

        default:
          /* we found a character other than a slash */
          
          switch(c)
          {
            case ' ':
            case '\t':
              if(FileInfo->LastChar == '\0')
              {
                /* the first character of the C++ comment is a space, save this
                   information so we close the comment with one space character
                   before the C comment termination token */
                FileInfo->TokenInfo.CPPComment.Flags |= TI_CPP_DO_SPACE;
                FileInfo->TokenInfo.CPPComment.SpaceChar = c;
              }
              break;
          }
          
          if(FileInfo->TokenInfo.CPPComment.Flags & TI_CPP_ONLYSLASHES)
          {
             /* let's just convert the already found slashes that are located
                directly at the beginning of the comment to asterisks. This only
                affects slashes at the very  beginning of the C++ comment since
                we're checking the TI_CPP_ONLYSLASHES flag */
            while(FileInfo->TokenInfo.CPPComment.SlashCount > 0)
            {
              putchar('*');
              FileInfo->TokenInfo.CPPComment.SlashCount--;
            }
          }
          else
          {
            /* we found several slashes inside the comment surrounded by
               different characters. we shouldn't convert them */
            while(FileInfo->TokenInfo.CPPComment.SlashCount > 0)
            {
              putchar('/');
              FileInfo->TokenInfo.CPPComment.SlashCount--;
            }
          }
          
          /* no longer count the slashes as we found other characters inside the
             comment already */
          FileInfo->TokenInfo.CPPComment.Flags &= ~TI_CPP_ONLYSLASHES;
          break;
      }
    }
    else if(FileInfo->InToken & IN_C_COMMENT)
    {
      switch(c)
      {
        case '/':
          if(FileInfo->LastChar == '*')
          {
            /* we reached the end of a C comment */
            FileInfo->InToken &= ~IN_C_COMMENT;
            ClearLastChar = 1;
          }
          break;
      }
    }
    else if(FileInfo->InToken & IN_QUOTES)
    {
      switch(c)
      {
        case '\"':
          if(FileInfo->LastChar != '\\')
          {
            /* if the quotation character we're processing wasn't escaped we
               just reached the end of the string. */
            FileInfo->InToken &= ~IN_QUOTES;
          }
          ClearLastChar = 1;
          break;

        default:
          if(FileInfo->LastChar == '\\')
          {
            /* prevent handling escaped backslashes as escape characters
               again */
            ClearLastChar = 1;
          }
          break;
      }
    }
    else if(FileInfo->InToken & IN_INVERTED_COMMAS)
    {
      switch(c)
      {
        case '\'':
          if(FileInfo->LastChar != '\\')
          {
            /* if the quotation character we're processing wasn't escaped we
               just reached the end of the string. */
            FileInfo->InToken &= ~IN_INVERTED_COMMAS;
          }
          ClearLastChar = 1;
          break;

        default:
          if(FileInfo->LastChar == '\\')
          {
            /* prevent handling escaped backslashes as escape characters
               again */
            ClearLastChar = 1;
          }
          break;
      }
    }

  }
  
  if(!ClearLastChar)
  {
    FileInfo->LastChar = c;
  }
  else
  {
    FileInfo->LastChar = '\0';
  }
  
  if(OutputChar)
  {
    putchar(c);
  }
}

void
ConsumeFinish(PFILE_INFO FileInfo)
{
  if(FileInfo->InToken)
  {
    ASSERT(!(FileInfo->InToken & ~IN_TOKEN_MASK));

    if(FileInfo->InToken & IN_CPP_COMMENT)
    {
      FileInfo->InToken &= ~IN_CPP_COMMENT;

      if(FileInfo->TokenInfo.CPPComment.SlashCount > 1)
      {
        if(FileInfo->TokenInfo.CPPComment.Flags & TI_CPP_DO_SPACE)
        {
          FileInfo->TokenInfo.CPPComment.SlashCount--;
        }
        
        /* we reached the end of the C++ comment. if the TI_CPP_ONLYSLASHES is
           set we convert all slashes inside the C++ comment to asterisks, if
           it's cleared we only convert the trailing ones (if existing) */
        while(FileInfo->TokenInfo.CPPComment.SlashCount > 2)
        {
          putchar('*');
          FileInfo->TokenInfo.CPPComment.SlashCount--;
        }
      }
      
      /* write one space before we close the comment */
      if(FileInfo->TokenInfo.CPPComment.Flags & TI_CPP_DO_SPACE &&
         FileInfo->LastChar != ' ')
      {
        putchar(FileInfo->TokenInfo.CPPComment.SpaceChar);
      }

      /* it's time to close the new C comment */
      putchar('*');
      putchar('/');
    }
  }
}

int
main (int argc, char* argv[])
{
  if(argc != 2)
  {
    printf("CPP2CCMT Version 0.5 (c) 2004 by Thomas Weidenmueller\n");
    printf("Licensed under the GNU Lesser General Public Version 2.1\n\n");
    printf("Usage: cpp2ccmt <FILENAME>\n");
    return 1;
  }
  else
  {
    int BytesRead;
    FILE* InputFile = fopen(argv[1], "rt");
    if(InputFile != NULL)
    {
      FILE_INFO FileInfo;
      char Buffer[READ_BUF_SIZE];
      
      memset(&FileInfo, 0, sizeof(FileInfo));
      
      do
      {
        BytesRead = fread(Buffer, 1, sizeof(Buffer), InputFile);
        if(BytesRead > 0)
        {
          int i;
          for(i = 0; i < BytesRead; i++)
          {
            ConsumeCharacter(&FileInfo, Buffer[i]);
          }
        }
      } while(BytesRead == READ_BUF_SIZE);
      
      if(!feof(InputFile) || ferror(InputFile))
      {
        fclose(InputFile);
        printf("\nFailed to read from input file!");
        return 1;
      }
      
      ConsumeFinish(&FileInfo);
      
      fclose(InputFile);
    }
    else
    {
      printf("\nFailed to open input file!");
      return 1;
    }
  }

  return 0;
}
