-- hip.ex - Hide In Picture 2.1 command-line interface
-- Copyright (C) 2002  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- If you wish to contact me, send an e-mail to davitf@usa.net .
--
-- You can get the latest version of this program from the HIP page:
-- http://hide-in-picture.sf.net/
--
-- This file is the main program for the command-line version of HIP.

without warning
without type_check

-- modify this to change the language
include m_en.e

include msg.e

include get.e
include wildcard.e
include graphics.e

wrap(0)

constant ESC = {27}


include formats.e


include main.e
include pictures.e
include fileutil.e

include sle.e
with trace


constant ERROR_ARG = 1,     -- Command-line error code

	-- Options sequence structure
	OPT_OPERATION = 1, OPT_FNTOWRITE = 2, OPT_ENCRYPTION = 3,
	OPT_PASSWORD = 4, OPT_WRITECRC = 5, OPT_HIDEPWD = 6,
	OPT_VIEWONLY = 7, OPT_QUIET = 8, OPT_TRANSPARENT = 9,
	OPT_FORMAT = 10, OPT_YES = 11,
	OPT_INPUTBMP = 12, OPT_FILE = 13, OPT_OUTPUTBMP = 14,

	-- Operation code
	OP_HIDE = 1, OP_RETRIEVE = 2, OP_ERASE = 3,

	ENC_DEFAULT = ALG_BLOWFISH,
	FORMAT_DEFAULT = FORMAT_BMP,
	-- Default options
	default_options = {-1,  -- no default operation
			   -1,  -- no default file name
			    0,  -- default encryption will be used later if necessary
			   -1,  -- no default pasword
			    1,  -- write CRC by default
			    1,  -- hide password by default
			    0,  -- NOT view only
			    0,  -- NOT quiet
			   -1,  -- no default transparent color
			   -1,  -- no default format yet
			    0,  -- NO auto confirmation
			   -1,  -- no default input picture
			   -1,  -- no default file to hide/retrieve
			   -1}  -- no default output picture


constant hex_values = "0123456789abcdef"

atom show_messages, auto_yes
sequence screen_properties
screen_properties = video_config()


procedure show_help ()
    -- Shows the help screen, creating a list of the available
    -- encryption algorithms
    sequence algorithms, image_formats

    algorithms = {}
    for n = 1 to length (enc_algs) do
	if ENC_DEFAULT = n then
	    algorithms &= '*'
	end if

	algorithms &= sprintf ("%s=%s", {'a' + n - 1, enc_algs [n][ALG_NAME]})
	if n < length (enc_algs) then algorithms = algorithms & "  " end if
    end for

    image_formats = {}
    for n = 1 to length (formats) do
	if FORMAT_DEFAULT = n then
	    image_formats &= '*'
	end if

	image_formats &= sprintf ("%s=%s", {'a' + n - 1, formats [n][FORMAT_NAME]})
	if n < length (image_formats) then image_formats &= "  " end if
    end for



    printf_scr ( help, {algorithms,image_formats})

end procedure


constant YES = 1, NO = 2

function yes_or_no (sequence question, atom default)
    -- Asks a question and gets an answer from the user;
    -- if the key pressed is invalid, return default value.
    atom key, answer

    -- If the 'automatically answer yes' options is enabled,
    -- do not ask the user
    if auto_yes = 1 and show_messages = 0 then
	return YES
    end if
    -- show question
    puts_scr ( question)
    if default = YES then
	printf_scr(" [%s/%s] ? ",{upper(yes_key),lower(no_key)})
    else
	printf_scr(" [%s/%s] ? ",{lower(yes_key),upper(no_key)})
    end if

    if auto_yes = 1 then
	answer = YES

    else        -- Ask the user

	key = upper (wait_key())        -- Get key
	if key = yes_key then answer = YES
	elsif key = no_key then answer = NO
	else answer = default end if
    end if

    if answer = YES then puts_scr ( yes_key & '\n')
    elsif answer = NO then puts_scr ( no_key & '\n')
    else abort_error ("hip.ex", "yes_or_no", "Not YES or NO answer")
    end if

    return answer
end function


function get_color (sequence string)
    -- Converts a command-line color index into an atom or sequence
    atom val1, val2, index
    sequence rgb

    string = lower (string)
    if length(string) = 2 then
	val1 = find (string[1], hex_values) - 1
	val2 = find (string[2], hex_values) - 1
	if val1 = -1 or val2 = -1 then return -1 end if
	index = val1 * #10 + val2
	return index

    elsif length(string) = 6 then

	rgb = {-1, -1, -1}

	-- Red
	val1 = find (string[1], hex_values) - 1
	val2 = find (string[2], hex_values) - 1
	if val1 = -1 or val2 = -1 then return -1 end if
	rgb [PAL_RED] = val1 * #10 + val2

	-- Green
	val1 = find (string[3], hex_values) - 1
	val2 = find (string[4], hex_values) - 1
	if val1 = -1 or val2 = -1 then return -1 end if
	rgb [PAL_GREEN] = val1 * #10 + val2

	-- Blue
	val1 = find (string[5], hex_values) - 1
	val2 = find (string[6], hex_values) - 1
	if val1 = -1 or val2 = -1 then return -1 end if
	rgb [PAL_BLUE] = val1 * #10 + val2

	return rgb

    else
	return -1
    end if


end function




function get_command ()
    -- Get command-line arguments

    atom command, pos
    sequence comm_line, file_names, options, arg
    object transparent

    options = default_options

    comm_line = command_line()


    comm_line = comm_line [3..length(comm_line)]
    if length(comm_line) < 2 then return ERROR_ARG end if

    command = comm_line[1][1]

    -- Get operation
    if command = 'h' then
	options [OPT_OPERATION] = OP_HIDE
    elsif command = 'r' then
	options [OPT_OPERATION] = OP_RETRIEVE
    elsif command = 'e' then
	options [OPT_OPERATION] = OP_ERASE
    else
	return ERROR_ARG
    end if


    if length(comm_line[1]) >= 2 then
	if comm_line[1][2] = '-' then   -- Options
	    comm_line [1] = comm_line [1][2..length(comm_line[1])]
	else return ERROR_ARG end if    -- Error

    else
	comm_line = comm_line [2..length(comm_line)]
    end if

    file_names = {}

    while length(comm_line) do
	-- get one argument at a time

	arg = comm_line [1]
	comm_line = comm_line [2..length(comm_line)]


	if arg[1] != '-' then       -- does not start with '-'
	    -- it is not an option, so it is a file name
	    file_names = append (file_names, arg)

	else    -- get option

	    arg = arg [2..length(arg)]  -- rest of argument

	    -- first the options that depend on a variable number
	    -- of characters following it
	    if compare (arg[1], 'f') = 0 then
		-- File name to write
		options [OPT_FNTOWRITE] = arg [2..length(arg)]

	    elsif compare (arg[1], 'p') = 0 then
		-- password to get
		options [OPT_PASSWORD] = arg [2..length(arg)]

	    elsif compare (arg[1], 't') = 0 then
		-- transparent color index/hex
		transparent = get_color (arg [2..length(arg)])
		if compare (transparent, -1) = 0 then
		    -- Error
		    return ERROR_ARG
		end if

		options [OPT_TRANSPARENT] = transparent

	    else

		pos = find ('-', arg)   -- split argument?
		if pos then
		    comm_line = prepend (comm_line, arg [pos..length(arg)])
		    arg = arg [1..pos-1]
		end if

		if length (arg) = 2 and arg[1] = 'e' and    -- encryption algorithm
		    arg[2] >= 'a' and arg[2] <= 'a' + length (enc_algs)-1 then
			-- if valid algorithm, set it
		       options [OPT_ENCRYPTION] = arg[2] - 'a' + 1


		elsif length (arg) = 2 and arg[1] = 'i' and    -- image format
		    arg[2] >= 'a' and arg[2] <= 'a' + length (formats)-1 then
			-- if valid algorithm, set it
		       options [OPT_FORMAT] = arg[2] - 'a' + 1


		elsif compare (arg, "c") = 0 then       -- write CRC
		    options [OPT_WRITECRC] = 1
		elsif compare (arg, "C") = 0 then       -- do not write CRC
		    options [OPT_WRITECRC] = 0

		elsif compare (arg, "h") = 0 then       -- mask password prompt
		    options [OPT_HIDEPWD] = 1
		elsif compare (arg, "H") = 0 then       -- do not mask prompt
		    options [OPT_HIDEPWD] = 0

--                 elsif compare (arg, "b") = 0 then       -- save as BMP picture
--                     options [OPT_FORMAT] = FORMAT_BMP
--                 elsif compare (arg, "g") = 0 then       -- save as GIF picture
--                     options [OPT_FORMAT] = FORMAT_GIF

		elsif compare (arg, "v") = 0 then   -- just testing
		    options [OPT_VIEWONLY] = 1

		elsif compare (arg, "q") = 0 then   -- quiet
		    options [OPT_QUIET] = 1
		elsif compare(arg, "y") = 0 then    -- auto confirmation
		    options [OPT_YES] = 1
		else                                -- invalid argument
		    return ERROR_ARG
		end if

	    end if

	end if

    end while


    -- all arguments parsed, sort them out

    if options [OPT_OPERATION] = OP_HIDE then
	if length(file_names) = 2 then
	    -- overwrite image
	    options [OPT_INPUTBMP] = file_names [1]
	    options [OPT_FILE] = file_names [2]
	    options [OPT_OUTPUTBMP] = file_names [1]

	elsif length(file_names) = 3 then
	    -- write to new image
	    options [OPT_INPUTBMP] = file_names [1]
	    options [OPT_FILE] = file_names [2]
	    options [OPT_OUTPUTBMP] = file_names [3]
	else
	    -- invalid number of file names
	    return ERROR_ARG
	end if

	if atom (options [OPT_FNTOWRITE]) then
	    options [OPT_FNTOWRITE] = get_name (options [OPT_FILE])
	end if

    elsif options [OPT_OPERATION] = OP_RETRIEVE then

	if length(file_names) = 1 then
	    -- output file name not specified
	    options [OPT_INPUTBMP] = file_names [1]
	elsif length(file_names) = 2 then
	    -- output file name specified
	    options [OPT_INPUTBMP] = file_names [1]
	    options [OPT_FILE] = file_names [2]
	else
	    -- invalid number of file names
	    return ERROR_ARG
	end if

    elsif options [OPT_OPERATION] = OP_ERASE then
	if length(file_names) = 1 then
	    -- overwrite image
	    options [OPT_INPUTBMP] = file_names [1]
	    options [OPT_OUTPUTBMP] = file_names [1]

	elsif length(file_names) = 2 then
	    -- write to new image
	    options [OPT_INPUTBMP] = file_names [1]
	    options [OPT_OUTPUTBMP] = file_names [2]
	else
	    -- invalid number of file names
	    return ERROR_ARG
	end if


    end if


    if (options [OPT_OPERATION] = OP_HIDE or options [OPT_OPERATION] = OP_ERASE)
      and options [OPT_FORMAT] = -1 then

	-- set default format
	options [OPT_FORMAT] = identify_format_from_filename (options[OPT_OUTPUTBMP])

	-- if unable to identify format, use default
	if options [OPT_FORMAT] = 0 then options[OPT_FORMAT]=FORMAT_DEFAULT end if

    end if

    return options
end function

procedure show_information (sequence info, atom always)
    if always or show_messages then
	puts_scr ( info)
    end if
end procedure

function ask_password (atom confirmation, atom hide)
    -- get password from user

    atom char
    sequence cursor_pos
    object password, confirm


    if hide then char = '*' else char = -1 end if

    while 1 do

	-- show prompt and get password
	puts_scr ( prompt_enterpassword)
	cursor_pos = get_position()
	password = ask (cursor_pos[1], cursor_pos[2], screen_properties [VC_COLUMNS],
		    "", char)

	puts_scr ( "\n")
	if atom (password) then return -1 end if    -- aborted

	if confirmation then    -- confirm password?
	    -- if so, prompt for confirmation
	    puts_scr ( prompt_confirmpassword)
	    cursor_pos = get_position()
	    confirm = ask (cursor_pos[1], cursor_pos[2],
		      screen_properties [VC_COLUMNS], "", char)
	    puts_scr ( "\n")

	    if atom (confirm) then return -1    -- aborted
	    elsif compare (password, confirm) = 0 then  -- compare passwords
		return password end if  -- OK

	else return password    -- No confirmation needed
	end if

	puts_scr ( error_differentpasswords & '\n')    -- different pwds, ask again

    end while


end function

constant highlighted_color = BRIGHT_WHITE, normal_color = BLUE,
	 highlighted_char = '*', normal_char = '*',
	 txt_color = WHITE, back_color = BLACK

procedure show_status (atom line, atom col, atom width, atom status)
    atom highlighted

    if show_messages = 0 then return end if     -- Quiet mode

    width = width - 5

    highlighted = floor (status * width)

    bk_color (back_color)

    position (line, col)
    text_color (highlighted_color)
    puts_scr ( repeat (highlighted_char, highlighted))

    text_color (normal_color)
    puts_scr ( repeat (normal_char, width - highlighted))

    text_color (txt_color)
    printf_scr( " %3d%%", status*100)

end procedure

constant status_interval = .05

procedure hide (sequence options)
    -- hide a file in a picture
    atom input_bmp, output_bmp, input_fn,
	 error, timer, bar_width
    object img_info
    sequence pos, hip_info


    -- Confirm overwrite

    if options [OPT_VIEWONLY] = 0 then

	if compare (options [OPT_INPUTBMP], options [OPT_OUTPUTBMP]) = 0 then
	    if yes_or_no (sprintf (prompt_overwritepicture, {get_name(options [OPT_INPUTBMP]) }),
	      YES) = NO then puts_scr (error_abortedbyuser) abort(1) end if

	elsif file_exists (options [OPT_OUTPUTBMP]) then
	    if yes_or_no (sprintf(prompt_overwrite, { get_name (options [OPT_OUTPUTBMP] )}),
	      NO) = NO then puts_scr (error_abortedbyuser) abort(1) end if

	end if

    end if


    -- Open input file
    input_fn = open (options [OPT_FILE], "rb")
    if input_fn = -1 then  -- Could not open file
	printf_scr (error_unabletoopen, {options [OPT_FILE]})  abort(1)
    end if


    show_information (sprintf (status_opening_bmp & '\n', {get_name(options [OPT_INPUTBMP])}), 0)
    -- Open input bitmap
    input_bmp = open (options [OPT_INPUTBMP], "rb")
    if input_bmp = -1 then  -- Could not open file
	printf_scr (error_unabletoopen, {options [OPT_INPUTBMP]})  abort(1)
    end if



    -- Read bitmap information
    img_info = read_img (input_bmp)

    if atom (img_info) then     -- Error in bitmap
	close (input_bmp)
	if img_info = ERROR_FILEERROR then
	printf_scr (error_filesystemerror, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_CORRUPTED then
	printf_scr (error_corrupted, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_UNSUPPORTED then
	printf_scr (error_unsupportedbmp, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_INVALIDBMP then
	printf_scr (error_invalidbmp, {options [OPT_INPUTBMP]})

	else
--        printf_scr (error_unknown, {options [OPT_INPUTBMP], img_info}))
	abort_error ("hip.ex", "hide", sprintf("Unknown error code from read_img: %d", img_info))
	end if

	abort(1)
    end if

    -- No errors, bitmap loaded successfully
    close (input_bmp)


    -- compute hip_info from img_info
    hip_info = compute_hip_info (img_info)      -- in pal.e

    trace(1)
    -- See whether output format supports BPP

    if find(img_info[IMG_BPP], formats[options[OPT_FORMAT]][FORMAT_BPPS])=0 then
	-- error
	printf_scr (error_unsupportedbpp,
	 {img_info[IMG_BPP], -- image bpp
	 formats[options[OPT_FORMAT]][FORMAT_NAME]}) -- file format name
	abort(1)
    end if



    -- Set transparent color
    if compare(options [OPT_TRANSPARENT] , -1) then
	if img_info[IMG_BPP] = 8 and sequence (options [OPT_TRANSPARENT]) then
		options [OPT_TRANSPARENT] =
		find (options [OPT_TRANSPARENT], img_info[IMG_PAL]) - 1
		if options [OPT_TRANSPARENT] = -1 then
		    puts_scr (error_colornotfound)      abort(1)
		end if
	elsif img_info[IMG_BPP] = 24 and atom (options [OPT_TRANSPARENT]) then
	    puts_scr (error_mustbergb)      abort(1)
	end if

	img_info[IMG_TRANSPARENT] = options [OPT_TRANSPARENT]
	hip_info = compute_hip_info (img_info)

    end if

    -- Is picture big enough to store the whole file?
    if total_bits (file_size (input_fn), 0,
       options [OPT_WRITECRC], length (options [OPT_FNTOWRITE]), 0) >
       hip_info[HIP_MAXBITSPERBYTE] * hip_info[HIP_AVAILABLEBYTES]

       then close (input_fn) puts_scr(error_filetoobig) abort(1)

    end if

    if options [OPT_VIEWONLY] then      -- Do not really write, just check space
	close (input_fn)
	puts_scr ( info_filecanbehidden & '\n')
	return
    end if

    -- No problems
    -- Is password already set?
    if atom (options [OPT_PASSWORD]) then
	options [OPT_PASSWORD] = ask_password (1, options [OPT_HIDEPWD])
	if atom(options [OPT_PASSWORD]) then    -- User exited password prompt
	    puts_scr (error_abortedbyuser) abort(1)  end if

    end if


--     if atom (options [OPT_FNTOWRITE]) then  -- no file name chosen, write real one
--         options [OPT_FNTOWRITE] = options [OPT_FILE] end if

    if options [OPT_ENCRYPTION] = 0 then
	options [OPT_ENCRYPTION] = ENC_DEFAULT
    end if

    show_information (status_initializing & '\n', 0)

    -- write header
    error = init_writing (hip_info[HIP_AVAILABLEBYTES],
	    input_fn,
	    options [OPT_WRITECRC],
	    options [OPT_FNTOWRITE],
	    {},
	    options [OPT_ENCRYPTION],
	    options [OPT_PASSWORD],
	    hip_info[HIP_MAXBITSPERBYTE] * hip_info[HIP_AVAILABLEBYTES])

    if error = ERROR_FILETOOBIG then
	-- This test has already been done, this should not happen
	abort_error ("hip.ex", "hide", "init_writing reports file_too_big")

    elsif error = ERROR_FILEERROR then
	printf_scr (error_filesystemerror, {options [OPT_FILE]})   abort(1)
    end if


    -- Now it's the fun part - let's hide the file!

    show_information (status_hiding, 0)

    pos = get_position()
    timer = time() + status_interval
    bar_width = screen_properties [VC_COLUMNS] - pos [2] - 1

    error = 0
    while error = 0 do
	if time() >= timer then
	    show_status (pos[1], pos[2], bar_width, completion_status () )
	    timer = time() + status_interval
	    if find (get_key() , ESC) then
		show_information ("\n", 0)
		puts_scr (error_abortedbyuser)  abort(1)
	    end if
	end if
	error = go_write()
    end while

    close (input_fn)

    if error = ERROR_FILEERROR then
	printf_scr (error_filesystemerror, {options [OPT_FILE]})   abort(1)
    elsif error != INFO_LASTBLOCK then
	abort_error ("hip.ex", "hide", sprintf("Unknown error code from go_write: %d", error))
    end if

    -- Done, now write the hidden file in the picture

    show_status (pos[1], pos[2], bar_width, 1)

    show_information ("\n", 0)
    show_information (status_writing_in_picture, 0)

    pos = get_position()
    timer = time() + status_interval
    bar_width = screen_properties [VC_COLUMNS] - pos [2] - 1

    init_write_data_in_img ( get_scattered_data (), get_current_bit(),
		img_info, hip_info)


    end_scatter ()

    error = 0

    while error = 0 do
	if time() >= timer then
	    show_status (pos[1], pos[2], bar_width, pic_completion_status () )
	    timer = time() + status_interval
	    if find (get_key() , ESC) then
		show_information ("\n", 0)
		puts_scr (error_abortedbyuser)  abort(1)
	    end if
	end if
	error = go_write_in_img()
    end while

    if error != INFO_LASTBLOCK then
	abort_error ("hip.ex", "hide", sprintf("Unknown error code from go_write_in_bmp: %d", error))
    end if

    show_status (pos[1], pos[2], bar_width, 1)

    -- Done writing file in picture, now save picture

    show_information ("\n", 0)
    show_information (sprintf(status_saving_bmp & '\n', {get_name(options [OPT_OUTPUTBMP])} ), 0)

    output_bmp = open (options [OPT_OUTPUTBMP], "wb")

    if output_bmp = -1 then
	printf_scr(error_unabletoopen, {options [OPT_OUTPUTBMP]})   abort(1)
    end if


    if save_img (options [OPT_FORMAT], output_bmp, img_info) then
	abort_error ("hip.ex", "hide", "Error in save picture routine")
    end if

    close (output_bmp)

    show_information ("\n", 0)
    show_information (status_done & '\n', 0)
    -- Done!
end procedure


procedure retrieve (sequence options)
    -- retrieve a file from a picture
    atom input_bmp, output_fn,
	 error, timer, bar_width
    object img_info, hidden_info
    sequence pos, data, hip_info


    show_information (sprintf (status_opening_bmp & '\n', {get_name(options [OPT_INPUTBMP])}),0)
    -- Open input bitmap
    input_bmp = open (options [OPT_INPUTBMP], "rb")
    if input_bmp = -1 then  -- Could not open file
	printf_scr (error_unabletoopen, {options [OPT_INPUTBMP]})  abort(1)
    end if


    -- Read bitmap information
    img_info = read_img (input_bmp)

    if atom (img_info) then     -- Error in bitmap
	close (input_bmp)
	if img_info = ERROR_FILEERROR then
	printf_scr (error_filesystemerror, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_CORRUPTED then
	printf_scr (error_corrupted, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_UNSUPPORTED then
	printf_scr (error_unsupportedbmp, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_INVALIDBMP then
	printf_scr (error_invalidbmp, {options [OPT_INPUTBMP]})

	else
--        printf_scr (error_unknown, {options [OPT_INPUTBMP], img_info}))
	abort_error ("hip.ex", "retrieve", sprintf("Unknown error code from read_img: %d", img_info))
	end if

	abort(1)
    end if


    -- No errors, bitmap loaded successfully
    close (input_bmp)

    -- compute hip_info from img_info
    hip_info = compute_hip_info (img_info)      -- in pal.e


    -- Set transparent color
    if compare(options [OPT_TRANSPARENT] , -1) then

	if img_info[IMG_BPP] = 8 and sequence (options [OPT_TRANSPARENT]) then
		options [OPT_TRANSPARENT] =
		find (options [OPT_TRANSPARENT], img_info[IMG_PAL]) - 1
		if options [OPT_TRANSPARENT] = -1 then
		    puts_scr(error_colornotfound)   abort(1)
		end if
	elsif img_info[IMG_BPP] = 24 and atom (options [OPT_TRANSPARENT]) then
	    puts_scr (error_mustbergb)      abort(1)
	end if

	img_info[IMG_TRANSPARENT] = options [OPT_TRANSPARENT]
	hip_info = compute_hip_info (img_info)

    end if



    -- Is password already set?
    if atom (options [OPT_PASSWORD]) then
	options [OPT_PASSWORD] = ask_password (0, options [OPT_HIDEPWD])
	if atom(options [OPT_PASSWORD]) then    -- User exited password prompt
	    puts_scr (error_abortedbyuser) abort(1)     end if

    end if



    show_information (status_reading_from_picture, 0)

    pos = get_position()
    timer = time() + status_interval
    bar_width = screen_properties [VC_COLUMNS] - pos [2] - 1

    init_read_data_from_img (img_info, hip_info)



    error = 0

    while error = 0 do
	if time() >= timer then
	    show_status (pos[1], pos[2], bar_width, pic_completion_status () )
	    timer = time() + status_interval
	    if find (get_key() , ESC) then
		show_information ("\n", 0)
		puts_scr (error_abortedbyuser) abort(1)
	    end if
	end if
	error = go_read_from_img()
    end while

    if error != INFO_LASTBLOCK then
	abort_error ("hip.ex", "retrieve", sprintf("Unknown error code from go_read_in_bmp: %d", error))
    end if

    show_status (pos[1], pos[2], bar_width, 1)

    show_information ("\n", 0)

    show_information( status_initializing & '\n', 0)

    init_scatter_read (get_pic_data () )
    end_pic ()

    hidden_info = get_hidden_info (options [OPT_ENCRYPTION],
		  options [OPT_PASSWORD],
	    hip_info[HIP_MAXBITSPERBYTE] * hip_info[HIP_AVAILABLEBYTES] )

    if atom (hidden_info) then
	if hidden_info = ERROR_NOHIDDENFILE then
	    puts_scr (error_nohiddenfile)
	elsif hidden_info = ERROR_UNSUPPORTED then
	    puts_scr (error_unsupported)
	elsif hidden_info = ERROR_CORRUPTED then
	    puts_scr (error_hiddencorrupted)
	else
	    abort_error ("hip.ex", "retrieve", sprintf("Unknown error code from get_hidden_info: %d", hidden_info))
	end if
	abort(1)
    end if

    show_information (sprintf(info_filesize & '\n', hidden_info [2]), 0)


    if length( hidden_info [1]) = 0 then
	show_information (info_nofilename & '\n', 0)
	if atom (options [OPT_FILE]) and options [OPT_VIEWONLY] = 0 then
	    puts_scr (error_filenameneeded & '\n')  abort(1)
	end if
    else
	show_information (sprintf(info_filename & '\n',  {hidden_info [1]}),0)
	if atom (options [OPT_FILE]) then
	    options [OPT_FILE] = hidden_info [1]
	end if
    end if

    if options [OPT_VIEWONLY] then
	show_information (sprintf(info_algorithmused & '\n', {enc_algs [hidden_info[3]][ALG_NAME]}),0)

	if hidden_info [4] then
	    show_information (info_crcpresent & '\n', 0)
	else
	    show_information (info_crcnotpresent & '\n', 0)
	end if

	return
    end if


    -- Confirm overwrite
    if file_exists (options [OPT_FILE]) then
	if yes_or_no (sprintf(prompt_overwrite, { options [OPT_FILE] }),
	NO) = NO then puts_scr (error_abortedbyuser) abort(1) end if
    end if

    -- Open file
    output_fn = open (options [OPT_FILE], "wb")

    if output_fn = -1 then  -- Could not open file
	printf_scr (error_unabletoopen, {options [OPT_FILE]})   abort(1)
    end if

    init_reading (output_fn)

    show_information (status_retrieving, 0)


    -- Now it's the fun part - let's retrieve the file!

    pos = get_position()
    timer = time() + status_interval
    bar_width = screen_properties [VC_COLUMNS] - pos [2] - 1

    error = 0
    while error = 0 do
	if time() >= timer then
	    show_status (pos[1], pos[2], bar_width, completion_status () )
	    timer = time() + status_interval
	    if find (get_key() , ESC) then
		show_information ("\n", 0)
		puts_scr (error_abortedbyuser)  abort(1)
	    end if
	end if
	error = go_read()
    end while

    close (output_fn)

    show_status (pos[1], pos[2], bar_width, 1)

    show_information ("\n", 0)

    if error = ERROR_CORRUPTED then
	puts_scr (error_hiddencorrupted)    abort(1)
    elsif error != INFO_LASTBLOCK then
	abort_error ("hip.ex", "retrieve", sprintf("Unknown error code from go_read: %d", error))
    end if

    show_information (status_done & '\n', 0)
    -- Done!
end procedure

procedure erase (sequence options)
    -- erase hidden data from picture
    atom input_bmp, output_bmp, input_fn,
	 error, timer, bar_width
    object img_info
    sequence pos, erase_data, hip_info

    -- Confirm overwrite
    if options [OPT_VIEWONLY] then
	show_information (info_nothingtodo, 1)
	return
    end if

    if compare (options [OPT_INPUTBMP], options [OPT_OUTPUTBMP]) = 0 then
	if yes_or_no (sprintf (prompt_overwritepicture, {get_name(options [OPT_INPUTBMP]) }),
	  YES) = NO then puts_scr (error_abortedbyuser)  abort(1) end if

    elsif file_exists (options [OPT_OUTPUTBMP]) then
	if yes_or_no (sprintf(prompt_overwrite, { get_name (options [OPT_OUTPUTBMP] )}),
	  NO) = NO then puts_scr (error_abortedbyuser)  abort(1) end if

    end if

    show_information (sprintf (status_opening_bmp & '\n', {get_name(options [OPT_INPUTBMP])}), 0)
    -- Open input bitmap
    input_bmp = open (options [OPT_INPUTBMP], "rb")
    if input_bmp = -1 then  -- Could not open file
	printf_scr (error_unabletoopen, {options [OPT_INPUTBMP]})  abort(1)
    end if



    -- Read bitmap information
    img_info = read_img (input_bmp)

    if atom (img_info) then     -- Error in bitmap
	close (input_bmp)
	if img_info = ERROR_FILEERROR then
	printf_scr (error_filesystemerror, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_CORRUPTED then
	printf_scr (error_corrupted, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_UNSUPPORTED then
	printf_scr (error_unsupportedbmp, {options [OPT_INPUTBMP]})

	elsif img_info = ERROR_INVALIDBMP then
	printf_scr (error_invalidbmp, {options [OPT_INPUTBMP]})

	else
	printf_scr (error_unknown, {options [OPT_INPUTBMP], img_info}) abort(1)
	end if
	abort(1)
    end if

    -- No errors, bitmap loaded successfully
    close (input_bmp)

    -- compute hip_info from img_info
    hip_info = compute_hip_info (img_info)      -- in pal.e


    -- Set transparent color
    if compare(options [OPT_TRANSPARENT] , -1) then
	if img_info[IMG_BPP] = 8 and sequence (options [OPT_TRANSPARENT]) then
		options [OPT_TRANSPARENT] =
		find (options [OPT_TRANSPARENT], img_info[IMG_PAL]) - 1
		if options [OPT_TRANSPARENT] = -1 then
		    puts_scr(error_colornotfound)   abort(1)
		end if
	elsif img_info[IMG_BPP] = 24 and atom (options [OPT_TRANSPARENT]) then
	    puts_scr (error_mustbergb)  abort(1)
	end if

	img_info[IMG_TRANSPARENT] = options [OPT_TRANSPARENT]
	hip_info = compute_hip_info (img_info)

    end if

    -- No problems

    show_information (status_initializing & '\n', 0)

    -- Get erase data
    erase_data = get_erase_data (hip_info[HIP_AVAILABLEBYTES],
	    options [OPT_PASSWORD])

    show_information ("\n", 0)
    show_information (status_writing_in_picture, 0)

    -- Init writing in picture
    pos = get_position()

    timer = time() + status_interval
    bar_width = screen_properties [VC_COLUMNS] - pos [2] - 1

    init_write_data_in_img ( erase_data, 1, img_info, hip_info)

    erase_data = {}
    error = 0

    while error = 0 do
	if time() >= timer then
	    show_status (pos[1], pos[2], bar_width, pic_completion_status () )
	    timer = time() + status_interval
	    if find (get_key() , ESC) then
		show_information ("\n", 0)
		puts_scr (error_abortedbyuser)  abort(1)
	   end if
	end if
	error = go_write_in_img()
    end while

    if error != INFO_LASTBLOCK then
	abort_error ("hip.ex", "erase", sprintf("Unknown error code from go_write_in_bmp: %d", error))
    end if

    show_status (pos[1], pos[2], bar_width, 1)

    -- Done writing data in picture, now save picture

    show_information ("\n", 0)
    show_information (sprintf(status_saving_bmp & '\n', {get_name(options [OPT_OUTPUTBMP])} ), 0)

    output_bmp = open (options [OPT_OUTPUTBMP], "wb")

    if output_bmp = -1 then
	 printf_scr(error_unabletoopen, {options [OPT_OUTPUTBMP]}) abort(1)
    end if

    if save_img (options [OPT_FORMAT], output_bmp, img_info) then
	abort_error ("hip.ex", "hide", "Error in save picture routine")
    end if

    close (output_bmp)

    show_information ("\n", 0)
    show_information (status_done & '\n', 0)
    -- Done!

end procedure


object x

x = get_command ()

if atom(x) then show_help() abort(0) end if -- Error in parsing command line

auto_yes = x [OPT_YES]      -- Automatically answer yes?
show_messages = not (x [OPT_QUIET])     -- Show messages?

show_information(program_id & "\n\n", 0)

if x [OPT_OPERATION] = OP_HIDE then
    hide (x)
elsif x [OPT_OPERATION] = OP_RETRIEVE then
    retrieve (x)
elsif x [OPT_OPERATION] = OP_ERASE then
    erase (x)
else    -- Bug, this should not happen
    abort_error ("hip.ex", "none", sprintf("Unknown operation code: %d", x[OPT_OPERATION]))
end if

abort(0)

