-- pal.e - Palette matching/sorting - HIP 2.0
-- Copyright (C) 2001  Davi Tassinari de Figueiredo
--
-- This program is distributed under the terms of the GNU General
-- Public License. Please read the documentation for more information.
--
-- This file contains routines related to palette handling/matching.
-- They are used when working with bitmap images that use a palette.


constant include_name = "pal.e"


include errors.e
include constant.e


global function color_distance (sequence rgb1, sequence rgb2)
    -- Compute the distance between two RGB values.
    -- The closer they are, the smaller the result will be.
    -- Algorithm by Thiadmer Riemersma (http://www.compuphase.com)

    atom rmean, r, g, b

    rmean = (rgb1[PAL_RED] + rgb2[PAL_RED]) / 2
    r = rgb1[PAL_RED] - rgb2[PAL_RED]
    g = rgb1[PAL_GREEN] - rgb2[PAL_GREEN]
    b = rgb1[PAL_BLUE] - rgb2[PAL_BLUE]

    return  floor ( (512+rmean)*r*r / 256 + 4*g*g + (767-rmean)*b*b/256 )

end function

global constant max_distance = color_distance ({0,0,0}, {255,255,255}) + 1


global function sort_pal(sequence pal, atom transparent)
    -- Sort the palette entries according to their blue, green and red
    -- components, in this order; higher (lighter) values come first

    atom color_val
    sequence sorted,current, last_color

    sorted={}

    if transparent>=0 then
	pal[transparent+1]=-1
    end if



    last_color = {256,256,256}

    for pos_in_sorted=1 to length(pal) do

	-- find first color

	color_val=-1
	current = {-1, -1, -1} --256,256,256}

	for pal_entry=1 to length(pal) do
	    if sequence(pal[pal_entry]) then

		if compare (pal[pal_entry],current) = 1 then
		    current = pal[pal_entry]
		    color_val = pal_entry
		end if

	    end if

	end for

	if color_val = -1 then
	    -- There is no color left in the palette
	    exit
	end if

	-- Prevent identical colors in palette from counting as two entries
	if compare(current, last_color) = 0 then
	    -- Color is the same as the previous palette entry
	    sorted[length(sorted)]=sorted[length(sorted)]&color_val-1
	else
	    sorted=append(sorted,{color_val-1})
	end if
	current=pal[color_val]
	pal[color_val]=-1


	last_color = current
    end for


    return sorted
end function


global function inverse_transform (sequence pal_order)
    -- Create a sequence containing the position of each entry
    -- in the sorted palette

    sequence inverse
    inverse = repeat(-1, 256)

    for x = 1 to length(pal_order) do
	for k = 1 to length (pal_order[x]) do
	    inverse [pal_order[x][k]+1] = x-1
	end for
    end for

    return inverse
end function

function closest_color (sequence rgb, sequence list)
    -- Find the closest color in the color list
    atom min_distance, min_index, distance

    min_distance = color_distance (rgb, list[1])
    min_index = 1
    for index = 2 to length(list) do
	distance = color_distance (rgb, list[index])

	if distance < min_distance then
	    min_distance = distance
	    min_index = index
	end if

    end for

    return min_index
end function

global function color_lists (sequence pal, sequence sorted, atom bits)
    -- Generate lists of palette entries which end with
    -- each combination of bits

    -- lists contains the lists, original contains the original indexes
    -- for each of the entries in lists

    -- lists [n] contains the palette entries with the least-significant
    -- bits being n-1

    atom number_of_lists, original_index
    sequence lists, cur_list, original, cur_original

    number_of_lists = power (2,bits)
    lists = repeat ({}, number_of_lists)
    original = repeat ({}, number_of_lists)

    for n = 1 to number_of_lists do
	cur_list = {}
	cur_original = {}

	-- Create one list
	for x = n to length(sorted) by number_of_lists do
	    -- Create a list of colors ending with the bits
	    original_index = sorted [x][1]
	    cur_list = append (cur_list, pal [original_index+1])
	    cur_original = cur_original & original_index
	end for

	lists [n] = cur_list
	original [n] = cur_original

    end for

    return {lists, original}

end function


global function create_substitution_table (sequence pal, sequence lists, sequence original)
    -- Find the closest match in pal for each of the substitution
    -- possibilities, and create a table with the results

    -- This table will be used when dithering is not enabled,
    -- to speed up the process (currently dithering is not used in HIP)

    atom closest_match
    sequence replace_tables, cur_table

    replace_tables = repeat ({}, length(lists))

    for n = 1 to length(lists) do
	cur_table = repeat (-1, length(pal))
	for x = 1 to length(pal) do

	    closest_match = closest_color (pal[x], lists [n])
	    cur_table [x] = original [n][closest_match]

	end for

	replace_tables [n] = cur_table
    end for


    return replace_tables
end function

-- A simple routine which returs the log (base 2) of a number.
constant log2divisor = log (2)
function log2 (atom n)    return log (n) / log2divisor    end function


global function max_bits_per_byte (atom bpp, atom transparent, atom sorted_pal_length)
    -- Determines the maximum number of bits a picture can hide per byte.

    -- log_info ("max_bits_per_byte: start", LOG_IMPORTANT)

    if bpp <= 8 then
	-- log_info ("max_bits_per_byte: 8-bit calculation", LOG_DETAIL)
	-- Palette image
	return floor ( log2 ( sorted_pal_length ) )

    else
	-- log_info ("max_bits_per_byte: 24-bit calculation", LOG_DETAIL)
	-- If there is a transparent color, subtract one (not all bits
	-- can be used, or there is a risk of converting a color to the
	-- transparent color)
	return bpp/3 - transparent

    end if

end function


global function compute_hip_info (sequence img_info)
    -- Sets the transparent color of a bitmap, counts the number of
    -- transparent pixels, and returns a hip_info structure.

    atom pos_in_bmp, counter, line_size
    sequence img_line, hip_info
    object transparent

    -- log_info ("set_transparent_color: start", LOG_IMPORTANT)
    -- log_info ("set_transparent_color: transparent = "&sprint(transparent), LOG_DETAIL)

    hip_info = repeat (-1,HIP_INFO_SIZE)

    transparent = img_info [IMG_TRANSPARENT]

    if compare(transparent, -1) = 0 then
	-- No transparency

	counter = 0

    else
	-- Count transparent pixels

	line_size = img_info[IMG_WIDTH] * (img_info[IMG_BPP] / 8)

	counter = 0

	pos_in_bmp = img_info [IMG_ADDRESS]

	if img_info [IMG_BPP] = 8 then
	    -- log_info ("set_transparent_color: start 8-bit counter", LOG_DETAIL)

	    for y = 1 to img_info [IMG_HEIGHT] do
		-- Read one line
		img_line = peek ({pos_in_bmp, line_size})

		for x = 1 to length(img_line) do
		    if img_line [x] = transparent then counter = counter + 1 end if
		end for

		pos_in_bmp = pos_in_bmp + line_size
	    end for

	    -- log_info (sprintf("set_transparent_color: counter = %d",counter), LOG_DETAIL)

	elsif img_info [IMG_BPP] = 24 then
	    -- log_info ("set_transparent_color: start 24-bit counter", LOG_DETAIL)

	    for y = 1 to img_info [IMG_HEIGHT] do
		-- Read one line
		img_line = peek ({pos_in_bmp, line_size})

		for x = 1 to length(img_line) by 3 do
		    if compare(img_line [x..x+2], transparent) = 0 then
			counter = counter + 1 end if
		end for

		pos_in_bmp = pos_in_bmp + line_size
	    end for

	    -- log_info (sprintf("set_transparent_color: counter = %d",counter), LOG_DETAIL)

	else
	    abort_error ( include_name, "set_transparent_color", "Unsupported BPP")
	end if

    end if

    -- Update hip_info fields
--    hip_info [HIP_TRANSPARENT] = transparent        -- transparent color
    hip_info [HIP_TRANSPARENT_PIXELS] = counter     -- transparent pixels

    hip_info [HIP_AVAILABLEBYTES] =                 -- bytes available for hiding
	(img_info [IMG_WIDTH] * img_info [IMG_HEIGHT] - counter) * img_info [IMG_BPP]/8

    -- log_info (sprintf("set_transparent_color: available_bytes = %d",img_info [IMG_AVAILABLEBYTES]), LOG_DETAIL)


    if img_info [IMG_BPP]  <= 8 then
	-- Sort the palette
	-- log_info ("set_transparent_color: calling sort_pal", LOG_DETAIL)
	hip_info [HIP_SORTEDPAL] = sort_pal (img_info [IMG_PAL], img_info [IMG_TRANSPARENT])

    else

	-- 24-bit picture, no palette
	hip_info [HIP_SORTEDPAL] = {}
    end if

    -- Maximum number of bits per picture byte
    -- log_info ("set_transparent_color: calling max_bits_per_byte", LOG_DETAIL)
    hip_info [HIP_MAXBITSPERBYTE] = max_bits_per_byte (img_info [IMG_BPP] ,
	compare( img_info [IMG_TRANSPARENT], -1)!=0, length(hip_info [HIP_SORTEDPAL]))

    -- log_info (sprintf("set_transparent_color: max_bits_per_byte = %d",img_info [IMG_MAXBITSPERBYTE]), LOG_DETAIL)

    -- log_info ("set_transparent_color: exit", LOG_DETAIL)

    return hip_info
end function

