/*
 * Functions to get/set date and time in DOS.
 *
 * This file is part of the picoSNTP project.
 * Copyright (C) 2015-2016 Mateusz Viste
 *
 * http://picosntp.sourceforge.net
 */

/* * * * * * * * * * Extracts from the RBIL documentation * * * * * * * * * *
 * INT 21 / AH=2Ah - DOS 1+ - GET SYSTEM DATE
 *
 * AH = 2Ah
 *
 * Return:  CX = year (1980-2099)
 *          DH = month
 *          DL = day
 *          AL = day of week (00h=Sunday) - DOS 1.10+
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * INT 21 / AH=2Ch - DOS 1+ - GET SYSTEM TIME
 *
 * AH = 2Ch
 *
 * Return:  CH = hour
 *          CL = minute
 *          DH = second
 *          DL = 1/100 seconds
 *
 * Note: On most systems, the resolution of the system clock is about
 *       5/100sec, so returned times generally do not increment by 1. On some
 *       systems, DL may always return 00h.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * INT 21 / AH=2Bh - DOS 1+ - SET SYSTEM DATE
 *
 * AH = 2Bh
 * CX = year (1980-2099)
 * DH = month (1-12)
 * DL = day (1-31)
 *
 * Return:  AL = status
 *          00h successful
 *          FFh invalid date, system date unchanged
 *
 * Note: DOS 3.3+ also sets CMOS clock; due to the limitations of the CLOCK$
 *       driver interface, the CMOS time is also updated to the current DOS
 *       time (which is the BIOS time-of-day clock with the default CLOCK$
 *       driver).
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * INT 21 / AH=2Dh - DOS 1+ - SET SYSTEM TIME
 *
 * AH = 2Dh
 * CH = hour
 * CL = minute
 * DH = second
 * DL = 1/100 seconds
 *
 * Return:  AL = result
 *          00h successful
 *          FFh invalid time, system time unchanged
 *
 * Note: DOS 3.3+ also sets CMOS clock; due to the limitations of the CLOCK$
 *       driver interface, the CMOS date is also updated to the current DOS
 *       date.
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <dos.h>

#include "dostime.h" /* include self for control */


void dostime_get(int *year, int *mon, int *day, int *h, int *m, int *s) {
  union REGS r;
  /* fetch date */
  r.h.ah = 0x2A; /* DOS 1+ ; GET SYSTEM DATE */
  intdos(&r, &r);
  *year = r.x.cx;
  *mon = r.h.dh;
  *day = r.h.dl;
  /* fetch time */
  r.h.ah = 0x2C; /* DOS 1+ ; GET SYSTEM TIME */
  intdos(&r, &r);
  *h = r.h.ch;
  *m = r.h.cl;
  *s = r.h.dh;
}

void dostime_set(int year, int mon, int day, int h, int m, int s) {
  union REGS r;
  /* set date */
  r.h.ah = 0x2B; /* DOS 1+ ; SET SYSTEM DATE */
  r.x.cx = year;
  r.h.dh = mon;
  r.h.dl = day;
  intdos(&r, &r);
  /* set time */
  r.h.ah = 0x2D; /* DOS 1+ ; SET SYSTEM TIME */
  r.h.ch = h;
  r.h.cl = m;
  r.h.dh = s;
  r.h.dl = 0;
  intdos(&r, &r);
}
