/*
 * uHex is a small and fast multiplatform hex editor.
 * Copyright (C) 2013, 2014, 2015 Mateusz Viste
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h> /* malloc() */
#include <string.h> /* strdup() */

#include "io.h"
#include "file.h"

#define pVER "1.0.4"
#define pDATE "2013-2015"


/* color scheme        FG |  BG      */
#define COL_BG          0 | (1 << 4)
#define COL_OFFSET      6 | (1 << 4)
#define COL_DATA        7 | (1 << 4)
#define COL_DATACUR    15 | (6 << 4)
#define COL_DATAEMPTY   8 | (1 << 4)
#define COL_STATUSBAR   0 | (7 << 4)
#define COL_USERMSG    14 | (4 << 4)
#define COL_COLUMNS     7 | (1 << 4)
#define COL_CHANGES    14 | (1 << 4)
#define COL_CHANGESCUR 14 | (3 << 4)

/* mono scheme */
#define MONO_BG          0
#define MONO_OFFSET      7
#define MONO_DATA        7
#define MONO_DATACUR     0 | (7 << 4)
#define MONO_DATAEMPTY   8
#define MONO_STATUSBAR   0 | (7 << 4)
#define MONO_USERMSG     0 | (7 << 4)
#define MONO_COLUMNS     7
#define MONO_CHANGES    15
#define MONO_CHANGESCUR 15 | (7 << 4)


/* The frame '|' character to use */
#ifndef FRAMCHAR
#define FRAMCHAR 0xB3
#endif

/* The switch character is '/' by default, unless already forced */
#ifndef SWITCHCHAR
#define SWITCHCHAR '/'
#define SWITCHSTR "/"
#endif


struct changeitem {
  long offset;             /* the offset of the changed byte */
  unsigned char byte;      /* the new content at offset's position */
  unsigned char orig;      /* the original content at this position */
  struct changeitem *next; /* the next item of the list */
};


struct programState {
  char *filename;   /* the full path/filename of the file */
  char *filename_base; /* the base filename (without path) */
  char *usermsg;    /* a user message to display (NULL if none) */
  struct changeitem *changelist; /* a linked list with changes */
  long filepos;     /* position of the cursor in the file */
  long screenpos;   /* position of the screen's start in the file */
  long filesize;    /* the total file size */
  int termheight;   /* number of rows */
  int termwidth;    /* number of columns */
  int col_bg;       /* background color */
  int col_offset;   /* color of the offset table */
  int col_data[2];  /* color of data (normal / changed) */
  int col_datacur[2]; /* color of data when cursor on it (normal / changed) */
  int col_dataempty; /* color of data when empty fields */
  int col_statusbar; /* status bar */
  int col_usermsg;  /* color for messages */
  int col_columns;  /* pseudo-graphic delimiters */
  int ro;           /* readonly mode (0=rw / 1=ro) */
  int termcolor;    /* is it a color terminal? 0=no ; 1=yes */
  int mode;         /* 0 = cursor on hex side  ;  1 = cursor on ascii side */
};


enum inputType {
  INPUT_NONE,
  INPUT_LEFT,
  INPUT_RIGHT,
  INPUT_UP,
  INPUT_DOWN,
  INPUT_PAGEUP,
  INPUT_PAGEDOWN,
  INPUT_TAB,
  INPUT_HOME,
  INPUT_END,
  INPUT_QUIT,
  INPUT_HELP,
  INPUT_JUMP,
  INPUT_FIND,
  INPUT_SAVE,
  INPUT_UNDO,
  INPUT_UNKNOWN
};


/* converts a single hex char (0..F) and returns it's integer value.
 * Returns -1 if the hexchar wasn't a valid hex digit. */
static int hexchar2int(char hexchar) {
  if (hexchar >= 'a') hexchar -= 32; /* normalize lower case to upper case */
  if ((hexchar >= '0') && (hexchar <= '9')) return(hexchar - '0');
  if ((hexchar >= 'A') && (hexchar <= 'F')) return(hexchar - ('A' - 10));
  return(-1);
}


/* parses the cmdline and fills pState. returns 0 on success, non-zero on failure. */
static int parsecmdline(int argc, char **argv, struct programState *pState) {
  char *option;
  while (--argc > 0) {
    argv++;
    option = *argv;
    if (*option != SWITCHCHAR) { /* anything that doesn't start by a '/' is supposed to be a filename */
      if (pState->filename != NULL) return(-1); /* a filename is already set */
      pState->filename = option;
      continue;
    }
    /* othwerwise we are dealing with an option */
    option++; /* skip the '/' char */
    if (strcmp(option, "mono") == 0) { /* force bw mode */
        pState->termcolor = 0;
      } else if (strcmp(option, "color") == 0) { /* force color mode */
        pState->termcolor = 1;
      } else if (strcmp(option, "ro") == 0) { /* read-only mode */
        pState->ro = 1;
      } else {
        return(-1); /* unknown parameter */
    }
  }
  if (pState->filename == NULL) return(-2); /* no filename has been provided */
  return(0);
}


static void setmsg(char *msg, struct programState *pState) {
  if (pState->usermsg != NULL) return; /* there is already a message set, abort */
  if (msg == NULL) return;  /* no message provided */
  pState->usermsg = strdup(msg);
  if (pState->usermsg == NULL) return; /* strdup() failed, probably not enough memory */
}


static void about(void) {
  printf("uHex v" pVER " Copyright (C) Mateusz Viste " pDATE "\n"
         "\n"
         "uHex is a small and fast, multiplatform hex editor. It supports large files,\n"
         "and runs on any 8086 compatible CPU.\n"
         "\n");
  printf("Redistribution and use in source and binary forms, with or without\n"
         "modification, are permitted provided that the following conditions are met:\n"
         " 1. Redistributions of source code must retain the above copyright notice,\n"
         "    this list of conditions and the following disclaimer.\n"
         " 2. Redistributions in binary form must reproduce the above copyright notice,\n"
         "    this list of conditions and the following disclaimer in the documentation\n"
         "    and/or other materials provided with the distribution.\n"
         "\n");
  printf("Usage: uhex file [" SWITCHSTR "mono | " SWITCHSTR "color] [" SWITCHSTR "ro]\n"
         "\n"
         " where:\n"
         "  " SWITCHSTR "mono   forces monochrome display mode\n"
         "  " SWITCHSTR "color  forces color display mode\n"
         "  " SWITCHSTR "ro     opens the file in read-only mode\n"
         "\n");
}


/* checks the cursor's position, and scroll the screen if needed */
static void adjustscreenposition(struct programState *pState) {
  if ((pState->filepos >> 4) - (pState->screenpos >> 4) >= (pState->termheight - 3)) { /* cursor is going to get out of the screen (under lower edge) */
    pState->screenpos = 3 + (pState->filepos >> 4) - (pState->termheight);
    pState->screenpos <<= 4;
  } else if ((pState->filepos >> 4) < (pState->screenpos >> 4)) { /* cursor is going to get out of the screen (above higher edge) */
    pState->screenpos = (pState->filepos >> 4);
    pState->screenpos <<= 4;
  }
}


/* Get the next offset with modified byte in the file, since pos 'startpos'.
 * Returns a pointer to the changeitem with the next modification, or NULL if no change found. */
static struct changeitem *getnextmodificationoffset(struct programState *pState, long startpos) {
  struct changeitem *curitem;
  for (curitem = pState->changelist; curitem != NULL; curitem = curitem->next) {
    if (curitem->offset >= startpos) return(curitem);
  }
  return(NULL);
}


static void drawcontent(struct programState *pState) {
  int x, y, z, gotbytes, colattr;
  long file_offset;
  unsigned char linebuff[48];
  struct changeitem *nextmodification;
  static char symbuff[81];
  static int attrbuff[81];
  char *hexlist = "0123456789ABCDEF";

  cursor_hide(); /* hide the cursor */
  /* Initialize arrays for status bar */
  memset(symbuff, ' ', 80);
  for (x = 0; x < 80; x++) {
    attrbuff[x] = pState->col_statusbar;
  }
  /* Display status bar */
  if (pState->usermsg != NULL) { /* Display the user message, if any */
    attrbuff[1] = pState->col_usermsg;
    for (x = 0; pState->usermsg[x] != 0; x++) {
      attrbuff[x + 2] = pState->col_usermsg;
      symbuff[x + 2] = pState->usermsg[x];
    }
    attrbuff[x + 2] = pState->col_usermsg;
    free(pState->usermsg);
    pState->usermsg = NULL;
  } else { /* pState->usermsg == NULL */
    /* print the filename (without path) in the status bar */
    for (x = 0; (pState->filename_base[x] != 0) && (x < 28); x++) {
      symbuff[x] = pState->filename_base[x];
    }
    /* print file's size */
    x += sprintf(symbuff + x, " (%ld bytes)", pState->filesize);
    /* remove the NULL character added by sprintf */
    symbuff[x] = ' ';
    /* Print the help key binding and current offset */
    x = sprintf((char *)linebuff, "ALT+H: Help  offset: 0x%08lX", pState->filepos);
    memcpy(symbuff + 48, linebuff, x);
  }
  /* Print status bar */
  printattrstringyx(pState->termheight - 1, 0, symbuff, attrbuff, 80);

  /* Initialize arrays for the hex dump string */
  memset(symbuff, ' ', 80);
  symbuff[60] = FRAMCHAR;
  symbuff[79] = FRAMCHAR;
  attrbuff[0] = pState->col_data[0];
  for (x = 1; x < 9; x++) {
    attrbuff[x] = pState->col_offset;
  }
  for (; x < 80; x++) {
    attrbuff[x] = pState->col_data[0];
  }
  /* Display the actual file */
  nextmodification = getnextmodificationoffset(pState, pState->screenpos);
  file_offset = pState->screenpos;
  for (y = 0; y < pState->termheight - 1; y++) {
    int hex_x = 11, sym_x = 62; /* current screen coordinates in hex and sym fields */
    /* display the offsets column */
    sprintf(symbuff, "%c%08lX%c ", FRAMCHAR, file_offset, FRAMCHAR);
    /* proceed to hex and ascii columns (and read a chunk of the file) */
    gotbytes = file_read(linebuff, 16, pState->screenpos + (y << 4));
    for (x = 0; x < 16; x++) {
      int digit1, digit2, sym;
      if (x == 8) hex_x++;
      if (x < gotbytes) { /* real content */
        int ischanged = 0;
        if (nextmodification != NULL) {
          if (file_offset == nextmodification->offset) { /* this byte was modified */
            ischanged = 1;
            linebuff[x] = nextmodification->byte;
            nextmodification = getnextmodificationoffset(pState, nextmodification->offset + 1);
          }
        }
        if (file_offset == pState->filepos) {  /* this is the currently selected byte */
          colattr = pState->col_datacur[ischanged];
        } else {
          colattr = pState->col_data[ischanged];
        }
        file_offset++;
        sym = linebuff[x];
        digit1 = hexlist[sym >> 4];
        digit2 = hexlist[sym & 0x0F];
      } else { /* got eof -> fill the rest of the line */
        sym = '.';
        digit1 = '.';
        digit2 = '.';
        colattr = pState->col_dataempty;
      }
      /* prepare the ascii content */
      symbuff[sym_x] = sym;
      attrbuff[sym_x++] = colattr;
      /* prepare the hex content */
      symbuff[hex_x] = digit1;
      attrbuff[hex_x++] = colattr;
      symbuff[hex_x] = digit2;
      attrbuff[hex_x++] = colattr;
      hex_x++;
    }
    /* Print the entire string */
    printattrstringyx(y, 0, symbuff, attrbuff, 80);
  }
  /* replace the cursor at its previous place */
  if (pState->mode == 0) { /* we are in the hex space */
      if ((pState->filepos & 8) != 0) { /* if I am on the right part of the hex screen... */
        z = 13;
      } else {
        z = 12;
      }
      locate((pState->filepos - pState->screenpos) >> 4, z + ((pState->filepos & 15) * 3));
    } else { /* we are in the ascii space */
      locate((pState->filepos - pState->screenpos) >> 4, 62 + (pState->filepos & 15));
  }
  cursor_show(); /* unhide the cursor */
}


static void freemodifications(struct programState *pState) {
  /* if no modifications were done, quit immediately */
  if (pState->changelist == NULL) {
    setmsg("No modifications were done.", pState);
    return;
  }
  /* otherwise save the modifications */
  while (pState->changelist != NULL) {
    struct changeitem *victim;
    victim = pState->changelist;
    pState->changelist = pState->changelist->next;
    free(victim);
  }
  setmsg("All modifications have been undone.", pState);
}


static void findstr(struct programState *pState) {
  char *label;
  #define findstr_size 32
  unsigned char findstr[findstr_size];
  unsigned char readbuff[findstr_size];
  unsigned char readbuff2[findstr_size];
  char *hexlist = "0123456789ABCDEF";
  int x, y, gotbytes, inkey, findstr_len = 0;
  long searchpos;
  struct changeitem *nextmodification;
  if (pState->mode == 0) { /* hex mode */
      label = "Find [HEX]:";
    } else { /* ascii mode */
      label = "Find [ASCII]:";
  }
  locate(pState->termheight - 1, 0);
  printchar(' ', pState->col_usermsg);
  for (x = 1; label[x - 1] != 0; x++) {
    locate(pState->termheight - 1, x);
    printchar(label[x - 1], pState->col_usermsg);
  }
  for (inkey = x; inkey < 60; inkey ++) { /* finish drawing the usermsg color on the status bar */
    locate(pState->termheight - 1, inkey);
    printchar(' ', pState->col_usermsg);
  }
  x++; /* skip a single space after the label */
  for (;;) {
    if (pState->mode == 0) { /* display routine for hex mode */
        for (inkey = 0; inkey < findstr_len; inkey++) {
          locate(pState->termheight - 1, x + inkey + (inkey >> 1));
          printchar(findstr[inkey], pState->col_usermsg);
        }
        locate(pState->termheight - 1, x + findstr_len + (findstr_len >> 1)); /* locate the cursor at the end of the string */
        printchar(' ', pState->col_usermsg);  /* clear out the last char (in case there was something there before) */
        locate(pState->termheight - 1, x + findstr_len + (findstr_len >> 1)); /* locate the cursor at the end of the string */
      } else { /* display routine for ascii mode */
        for (inkey = 0; inkey < findstr_len; inkey++) {
          locate(pState->termheight - 1, x + inkey);
          printchar(findstr[inkey], pState->col_usermsg);
        }
        locate(pState->termheight - 1, x + findstr_len); /* locate the cursor at the end of the string */
        printchar(' ', pState->col_usermsg);  /* clear out the last char (in case there was something there before) */
        locate(pState->termheight - 1, x + findstr_len); /* locate the cursor at the end of the string */
    }
    inkey = getkey();
    if (inkey == 0) { /* getkey returns 0 when an extended key has been pressed */
      inkey = 0x100 | getkey();
    }
    /* a few rules common to hex and ascii mode */
    if ((inkey < 0) || (inkey > 0xff) || (inkey == 0x1B)) return;
    if (inkey == 8) { /* backspace */
      if (findstr_len > 0) findstr_len--;
      continue;
    }
    if (inkey == 0x0D) {
      if (findstr_len > 0) break; /* user pressed ENTER with non-empty search string -> let's go find some stuff */
      /* if no search term entered, abort without searching anything */
      setmsg("No search string provided!", pState);
      return;
    }
    /* check that the input is within expected range */
    if (pState->mode == 0) { /* hex mode */
        inkey = hexchar2int(inkey); /* this to normalize the input */
        if (inkey < 0) return;
        inkey = hexlist[inkey];
        if (findstr_len >= (findstr_size >> 1)) continue; /* ignore if the buffer is full already */
      } else { /* ascii mode */
        if (inkey < 0x20) return; /* must be actual ascii, otherwise abort search */
        if (findstr_len >= findstr_size) continue; /* ignore if the buffer is full already */
    }
    /* add the new input to findstr, if not too long yet */
    findstr[findstr_len] = inkey;
    findstr_len++;
  }
  /* If we got here, then we have something to search for */
  if (pState->mode == 0) { /* if in hex mode, change the search string to its real (binary) form */
    for (x = 0; x < findstr_len; x += 2) {
      if (x == findstr_len - 1) { /* we are at the last digit, and the digit is 4 bits only */
        findstr[x >> 1] = hexchar2int(findstr[x]);
        findstr_len++; /* increase the length of the string to compute correct binary length later */
      } else { /* normal 2x4bits hex byte */
        findstr[x >> 1] = hexchar2int(findstr[x]);
        findstr[x >> 1] <<= 4;
        findstr[x >> 1] |= hexchar2int(findstr[x+1]);
      }
    }
    findstr_len >>= 1; /* the findstr string is cut in half after packing bytes */
  }
  /* print out 'searching' message */
  label = "Searching...";
  for (x = 0; label[x] != 0; x++) {
    locate(pState->termheight - 1, x + 1);
    printchar(label[x], pState->col_usermsg);
  }
  for (inkey = x; inkey < 60; inkey++) { /* finish drawing the usermsg color on the status bar */
    locate(pState->termheight - 1, inkey);
    printchar(' ', pState->col_usermsg);
  }
  locate(pState->termheight - 1, x); /* put the blinking cursor at the end of the label */
  /* Do the actual research */
  for (searchpos = pState->filepos; searchpos < (pState->filesize - findstr_len); ) {
    nextmodification = getnextmodificationoffset(pState, searchpos);
    gotbytes = file_read(readbuff, findstr_size, searchpos);
    for (x = 0; x < gotbytes; x++) {
      if (nextmodification != NULL) { /* apply any possible changes to the read chunk of bytes */
        if ((searchpos + x) == nextmodification->offset) { /* this byte was modified */
          readbuff[x] = nextmodification->byte;
          nextmodification = getnextmodificationoffset(pState, nextmodification->offset + 1);
        }
      }
      if (readbuff[x] == findstr[0]) { /* if the byte is the same than the first byte we are looking for.. */
        gotbytes = file_read(readbuff2, findstr_len - 1, searchpos + x + 1);
        if (gotbytes != findstr_len - 1) break;
        for (y = 1; y < findstr_len; y++) {
          if ((nextmodification != NULL) && ((searchpos + x + y) == nextmodification->offset)) { /* this byte was modified */
            readbuff2[y - 1] = nextmodification->byte;
            nextmodification = getnextmodificationoffset(pState, nextmodification->offset + 1);
          }
          if (readbuff2[y - 1] != findstr[y]) break;
        }
        if (y == findstr_len) { /* found! */
          pState->filepos = searchpos + x;
          return;
        }
      }
    }
    if (gotbytes < 1) break;
    searchpos += gotbytes;
  }
  setmsg("No match found", pState);
}


static int savefile(struct programState *pState) {
  struct changeitem *victim;
  if (pState->changelist == NULL) {
    setmsg("No modifications were done. File not saved.", pState);
    return(-1);
  }
  while (pState->changelist != NULL) {
    /* apply the modification */
    if (file_writebyte(pState->changelist->offset, pState->changelist->byte) != 1) {
      setmsg("Write error!", pState);
      return(-1);
    }
    /* free the modification */
    victim = pState->changelist;
    pState->changelist = pState->changelist->next;
    free(victim);
  }
  setmsg("File saved.", pState);
  return(0);
}


static void jumpto (struct programState *pState) {
  char *label = " Jump to offset: 0x";
  char offsetstr[16];
  long newoffset = 0;
  int x, inkey;
  for (x = 0; label[x] != 0; x++) {
    locate(pState->termheight - 1, x);
    printchar(label[x], pState->col_usermsg);
  }
  for (inkey = x; inkey < 60; inkey ++) { /* finish drawing the usermsg color on the status bar */
    locate(pState->termheight - 1, inkey);
    printchar(' ', pState->col_usermsg);
  }
  for (;;) {
    sprintf(offsetstr, "%lX ", newoffset);
    for (inkey = 0; offsetstr[inkey] != 0; inkey++) {
      locate(pState->termheight - 1, x + inkey);
      printchar(offsetstr[inkey], pState->col_usermsg);
    }
    inkey = getkey();
    if (hexchar2int(inkey) >= 0) { /* validate hex input */
      if (newoffset < 0x8000000l) {
        newoffset <<= 4;
        newoffset |= hexchar2int(inkey);
      }
    } else if (inkey == 8) { /* Back space */
      if (newoffset > 0) newoffset >>= 4;
    } else if (inkey == 13) { /* Return */
      pState->filepos = newoffset;
      if (pState->filepos >= pState->filesize) pState->filepos = pState->filesize - 1;
      return;
    } else { /* Invalid entry */
      return;
    }
  }
}


static void processInput(enum inputType inputRequest, struct programState *pState) {
  switch (inputRequest) {
    case INPUT_UP:
      if (pState->filepos > 15) pState->filepos -= 16;
      break;
    case INPUT_DOWN:
      if (pState->filepos + 16 < pState->filesize) pState->filepos += 16;
      break;
    case INPUT_LEFT:
      if (pState->filepos > 0) pState->filepos -= 1;
      break;
    case INPUT_RIGHT:
      if (pState->filepos < pState->filesize - 1) pState->filepos += 1;
      break;
    case INPUT_PAGEUP:
      if (pState->filepos - ((pState->termheight - 2) << 4) >= 0) {
        pState->filepos -= ((pState->termheight - 2) << 4);
        if (pState->screenpos - ((pState->termheight - 2) << 4) >= 0) {
          pState->screenpos -= ((pState->termheight - 2) << 4);
        } else {
          pState->screenpos = 0;
        }
      } else {
        pState->filepos = 0;
      }
      break;
    case INPUT_PAGEDOWN:
      if (pState->filepos + ((pState->termheight - 2) << 4) < pState->filesize) {
        pState->filepos += ((pState->termheight - 2) << 4);
        pState->screenpos += ((pState->termheight - 2) << 4);
      } else {
        pState->filepos = pState->filesize - 1;
      }
      break;
    case INPUT_HOME:
      pState->filepos = 0;
      break;
    case INPUT_END:
      pState->filepos = pState->filesize - 1;
      break;
    case INPUT_TAB:
      pState->mode = 1 - pState->mode;
      break;
    case INPUT_JUMP: /* jump to offset... */
      jumpto(pState);
      break;
    case INPUT_HELP:
      setmsg("ALT+H Help   ALT+J Jump   ALT+F Find   ALT+S Save   ALT+U Undo   ESC Quit   ", pState);
      break;
    case INPUT_SAVE:
      savefile(pState);
      break;
    case INPUT_UNDO:
      freemodifications(pState);
      break;
    case INPUT_FIND:
      findstr(pState);
      break;
    default:
      /* do nothing */
      break;
  }
  /* check if the screen position needs to be adjusted */
  adjustscreenposition(pState);
}


static int getcurbyte(struct programState *pState) {
  struct changeitem *curchange;
  unsigned char bytebuff;
  for (curchange = pState->changelist; curchange != NULL; curchange = curchange->next) {
    if (curchange->offset > pState->filepos) break; /* we stop here, changelist entries are sorted */
    if (curchange->offset == pState->filepos) return(curchange->byte);
  }
  /* the cur byte wasn't changed. let's read it from disk */
  file_read(&bytebuff, 1, pState->filepos);
  return(bytebuff);
}


/* adds an entry to the changelist. returns 0 on success, nonzero on out-of-memory error. */
static int addnewchangeitem(struct programState *pState, struct changeitem *newchange) {
  struct changeitem *parentchange, *newentry, *curpos;
  curpos = pState->changelist;
  parentchange = NULL;
  for (;;) {
    if ((curpos == NULL) || (curpos->offset > newchange->offset)) { /* leaf position */
      /* insert it before this entry */
      newentry = (struct changeitem *) malloc(sizeof(struct changeitem));
      if (newentry == NULL) return(-1); /* out of memory */
      newentry->offset = newchange->offset;
      newentry->byte = newchange->byte;
      newentry->orig = newchange->orig;
      if (parentchange == NULL) { /* no parent */
        newentry->next = pState->changelist;
        pState->changelist = newentry;
      } else { /* a parent exists */
        newentry->next = parentchange->next;
        parentchange->next = newentry;
      }
      return(0);
    }
    if (curpos->offset == newchange->offset) { /* update this entry */
      curpos->byte = newchange->byte;
      if (curpos->byte == curpos->orig) { /* we got back to the original state - remove this change, it's not a 'change' anymore! */
        if (parentchange == NULL) {
          pState->changelist = curpos->next;
        } else {
          parentchange->next = curpos->next;
        }
        free(curpos);
      }
      return(0);
    }
    parentchange = curpos;
    curpos = curpos->next;
  }
}


static enum inputType getInput(struct programState *pState) {
  int x;
  int keypress = getkey();
  struct changeitem bytechange;
  char errmsg[24];
  if (keypress == 0) {  /* extended keystroke require a second call */
    keypress = 0x100 | getkey();
  }
  switch (keypress) {
    case 0x009: /* TAB */
      return(INPUT_TAB);
    case 0x01B: /* ESC */
      return(INPUT_QUIT);
    case 0x13B: /* F1 */
    case 0x123: /* ALT+h */
      return(INPUT_HELP);
    case 0x147: /* HOME */
      return(INPUT_HOME);
    case 0x148: /* UP */
      return(INPUT_UP);
    case 0x149: /* PAGEUP */
      return(INPUT_PAGEUP);
    case 0x14B: /* LEFT */
      return(INPUT_LEFT);
    case 0x14D: /* RIGHT */
      return(INPUT_RIGHT);
    case 0x14F: /* END */
      return(INPUT_END);
    case 0x150: /* DOWN */
      return(INPUT_DOWN);
    case 0x151: /* PAGEDOWN */
      return(INPUT_PAGEDOWN);
    case 0x124: /* JUMP (ALT+j) */
      return(INPUT_JUMP);
    case 0x11F: /* SAVE (ALT+s) */
      return(INPUT_SAVE);
    case 0x116: /* UNDO (ALT+u) */
      return(INPUT_UNDO);
    case 0x121: /* FIND (ALT+f) */
      return(INPUT_FIND);
    default:
      if (pState->mode == 0) { /* uHex is in hex mode */
        x = hexchar2int(keypress);
        if (x >= 0) { /* valid hex char */
          if (pState->ro != 0) { /* are we in read-only mode? */
            setmsg("This file is opened in read-only mode.", pState);
            return(INPUT_NONE);
          }
          bytechange.offset = pState->filepos;
          bytechange.orig = getcurbyte(pState);
          bytechange.byte = bytechange.orig & 0x0F;
          bytechange.byte <<= 4;
          bytechange.byte |= x;
          if (addnewchangeitem(pState, &bytechange) != 0) setmsg("Out of memory! Change aborted.", pState);
          return(INPUT_NONE);
        }
      } else { /* uHex is in ASCII mode */
        if ((keypress >= 0) && (keypress <= 0xFF)) {
          if (pState->ro != 0) { /* are we in read-only mode? */
            setmsg("This file is opened in read-only mode.", pState);
            return(INPUT_NONE);
          }
          bytechange.orig = getcurbyte(pState);
          bytechange.offset = pState->filepos;
          bytechange.byte = keypress;
          if (addnewchangeitem(pState, &bytechange) != 0) {
            setmsg("Out of memory! Change aborted.", pState);
          }
          return(INPUT_NONE);
        }
      }
      sprintf(errmsg, "invalid key 0x%04X", keypress);
      setmsg(errmsg, pState);
      return(INPUT_UNKNOWN);
  }
}


static char *getbasefilename(char *fullfilename) {
  char *lastseparator = fullfilename;
  while (*fullfilename != 0) {
    if ((*fullfilename == '\\') || (*fullfilename == '/')) lastseparator = fullfilename + 1;
    fullfilename++;
  }
  return(lastseparator);
}


int main(int argc, char **argv) {
  struct programState pState;
  enum inputType inputRequest;
  int errvar = 0;

  /* init all internal variables */
  memset(&pState, 0, sizeof(pState));

  io_init();

  getcurvideomode(&pState.termwidth, &pState.termheight, &pState.termcolor);

  if (parsecmdline(argc, argv, &pState) != 0) {
    io_close();
    about();
    return(1);
  }

  if (pState.termwidth < 80) {
    io_close();
    printf("Terminal's size detected as %dx%d. This program requires a 80 columns width.\n", pState.termwidth, pState.termheight);
    return(6);
  }

  pState.filename_base = getbasefilename(pState.filename);

  if (pState.termcolor == 0) { /* load the mono scheme */
    pState.col_bg = computecolor(MONO_BG);
    pState.col_offset = computecolor(MONO_OFFSET);
    pState.col_data[0] = computecolor(MONO_DATA);
    pState.col_data[1] = computecolor(MONO_CHANGES);
    pState.col_datacur[0] = computecolor(MONO_DATACUR);
    pState.col_datacur[1] = computecolor(MONO_CHANGESCUR);
    pState.col_dataempty = computecolor(MONO_DATAEMPTY);
    pState.col_statusbar = computecolor(MONO_STATUSBAR);
    pState.col_usermsg = computecolor(MONO_USERMSG);
    pState.col_columns = computecolor(MONO_COLUMNS);
  } else { /* load the color scheme */
    pState.col_bg = computecolor(COL_BG);
    pState.col_offset = computecolor(COL_OFFSET);
    pState.col_data[0] = computecolor(COL_DATA);
    pState.col_data[1] = computecolor(COL_CHANGES);
    pState.col_datacur[0] = computecolor(COL_DATACUR);
    pState.col_datacur[1] = computecolor(COL_CHANGESCUR);
    pState.col_dataempty = computecolor(COL_DATAEMPTY);
    pState.col_statusbar = computecolor(COL_STATUSBAR);
    pState.col_usermsg = computecolor(COL_USERMSG);
    pState.col_columns = computecolor(COL_COLUMNS);
  }

  if (pState.ro == 0) { /* open the file in read/write mode */
    pState.filesize = file_open(pState.filename, "rb+", &errvar);
  } else { /* open the file in strict readonly mode */
    pState.filesize = file_open(pState.filename, "rb", &errvar);
  }
  if (pState.filesize < 0) {
    if (pState.ro == 0) { /* try to open it again in readonly mode */
      pState.ro = 1;
      pState.filesize = file_open(pState.filename, "rb", &errvar);
    }
    if (pState.filesize < 0) {
      io_close();
      printf("Error: Failed to open '%s' (code %d) -> %s\n", pState.filename, errvar, strerror(errvar));
      return(2);
    }
  }
  cls(pState.col_bg);        /* clear screen and fill with BG color */

  for (;;) {
    drawcontent(&pState);
    inputRequest = getInput(&pState);
    if (inputRequest == INPUT_QUIT) {
      if (pState.changelist == NULL) break; /* if no modifications, quit */
      /* otherwise ask for a confirmation */
      setmsg("There are unsaved modifications. Press Esc again if you really want to quit.", &pState);
      drawcontent(&pState);
      inputRequest = getInput(&pState);
      if (inputRequest == INPUT_QUIT) break;
      inputRequest = INPUT_NONE;
    }
    processInput(inputRequest, &pState);
  }

  file_close();
  io_close();

  freemodifications(&pState);
  if (pState.usermsg != NULL) free(pState.usermsg);  /* free the user message string, if any (you must call this AFTER freemodifications!) */

  return(0);
}
